#' uniRel
#'
#' @description
#' A function for unifying the reliabilities of multiple tests
#'
#' @param rel Vector of reliabilities of the tests
#' @param b Vector of weightings for the reliabilities; must match order of `rel`. Default: 1
#' @param corm Correlation matrix of the tests; must be square with 1s on the diagonal
#'
#' @return
#' A single value of the class `numeric` representing the unified reliability estimate across multiple tests
#'
#' @examples
#' rel = c(0.70, 0.90, 0.80)
#' b = c(1, 1, 1)
#' corm = matrix(c(1, 0.50, 0.35, 0.50, 1, 0.75, 0.35, 0.70, 1), nrow = 3, ncol = 3)
#' uniRel(rel, b, corm)
#'
#' @importFrom utils combn
#'
#' @export
uniRel <- function(rel, b = NULL, corm) {
  num_rel = length(rel)
  if (is.null(b)) b = rep(1, num_rel)
  if (length(b) != num_rel) {
    stop("rel and b must have the same length.")
  }
  if (!is.matrix(corm) || nrow(corm) != num_rel || ncol(corm) != num_rel) {
    stop("corm must be a square matrix with dimensions matching the length of reliabilities")
  }
  weighted_rel_sum = sum(b^2 * rel)
  combs = combn(c(1:length(rel)), 2, simplify = FALSE)
  combs = do.call(rbind, lapply(combs, function(x) data.frame(v1 = x[1], v2 = x[2])))
  weighted_r_sum = 0
  for(i in seq_along(combs$v1)) {
    weighted_r_sum = weighted_r_sum + 2 * b[combs[i, 1]] * b[combs[i, 2]] * corm[combs[i, 1], combs[i, 2]]
  }
  numerator = weighted_rel_sum + weighted_r_sum
  denominator = sum(b^2) + weighted_r_sum
  reliability = numerator / denominator
  return(reliability)
}
