% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/run_twig.R
\name{run_twig}
\alias{run_twig}
\title{Run a twig model}
\usage{
run_twig(
  twig_obj,
  params,
  n_cycles = NULL,
  verbose = FALSE,
  parallel = FALSE,
  offset_trace_cycle = 1,
  ncore = NULL,
  progress_bar = TRUE
)
}
\arguments{
\item{twig_obj}{A twig object created by the \code{twig} function.}

\item{params}{A data frame or list of parameters to be used in the model.}

\item{n_cycles}{An integer specifying the number of cycles for a Markov model. Default is NULL.}

\item{verbose}{A logical value indicating whether to print detailed output. Default is FALSE.}

\item{parallel}{A logical value indicating whether to run the model in parallel. Default is FALSE.}

\item{offset_trace_cycle}{An integer specifying the offset trace cycle. Default is 1.
This is used to adjust the cycle number in the trace output. If set to 0, the initial state distribution will be used
as the first cycle. If set to 1, the initial state distribution will be ignored in the Markov trace.
In both situations, the total number of cycles will be the same as the input n_cycles.}

\item{ncore}{An integer specifying the number of cores to use for parallel processing. Default is total number of cores - 1.}

\item{progress_bar}{A logical value indicating whether to display a progress bar. Default is TRUE.}
}
\value{
A list containing the results of the model run. The list includes the following elements:
\itemize{
\item mean_ev A matrix of size decision x payoff containing the mean expected values (EV)s across simulations if params is a data.frame with more than 1 row.
\item sim_ev An array of size decision x payoff x simulation containing the simulated expected values (EV) by simulation.
}
The following will also be returned if verbose is TRUE for Markov models:
\itemize{
\item sim: The simulation ID.  If params is a dataset, only the first simulation will be used (sim = 1).
\item evaluated_funs: A list of dataframes of evaluated functions. Each function returns a data frame enumerating the dependencies of the function along with the value returned by that function for each combination of values.
\item evaluated_prob_funs_combined: A data frame containing the evaluated probability function values for each state, cycle, decision, and event that are merged into a single dataframe.
\item path_event_options: A data frame of the event options along each path. Rows = paths, columns = event_options.
\item path_probs: A data frame containing the path probabilities for each state, cycle, and decision.
\item event_probs: A data frame containing the event options along each path and the destination.
\item markov_trans_probs: An array containing the transition probabilities for each origin state, destination state, cycle, and decision.
\item markov_trace: An array containing the Markov trace for each cycle, state, and decision.
\item cycle_payoffs: An array containing the payoffs for each cycle, state, decision, and payoff.
\item cycle_ev: An array containing the Expected Value (EV) for each cycle, state, decision and payoff. cycle_ev = markov_trace * cycle_payoffs.
\item sim_ev: A matrix of total expected values (EV) of size decision x payoffs.
\item mean_ev: A matrix of mean expected values (EV) across simulations. Since this is for a single simulation, mean_ev = sim_ev.
}

The following will also be returned if verbose is TRUE for decision trees:
\itemize{
\item sim: The simulation ID. If params is a dataset, only the first simulation will be used (sim = 1).
\item evaluated_funs: A list of dataframes of evaluated functions. Each function returns a data frame enumerating the dependencies of the function along with the value returned by that function for each combination of values.
\item evaluated_prob_funs_combined: A data frame of the probability function values evaluated by decision and events harmonized to the same combinations of decisions and events across all probability functions.
\item event_probs: A data frame containing the probability of event options by decision.
\item outcome_probs: A matrix of outcome probabilities. Outcomes are the terminal event transitions of size decision x outcomes.
\item path_event_options: A data frame of the event options along each path. Rows = paths, columns = events.
\item path_probs: A matrix containing the path probabilities of size decision x paths.
\item path_payoffs: An array containing the path payoffs of size decision x paths x payoffs. Paths are indexed by their final outcomes in the twig and a key to event options is provided in path_event_options.
\item path_ev: An array containing the path expected values (EV) = path_probs x path_payoffs. This is also of size decision x paths x payoffs.
\item sim_ev: A matrix of total expected values (EV) of size decision x payoffs.
\item mean_ev: A matrix of mean expected values (EV) across simulations. Since this is for a single simulation, mean_ev = sim_ev.
}
}
\description{
This function runs a twig model, which currently can be either a decision tree or a Markov model.
}
\examples{
library(twig)

# define a Markov model twig
mytwig <- twig() +
  decisions(names = c(A,B)) +
  states(names = c(H,D),
         init_probs = c(1,0)) +
  event(name = death_event,
        options = c(yes, none),
        probs = c(pDie, leftover),
        transitions = c(D, stay)) +
  payoffs(names = c(utility))

# define the parameters
params <- list(prob_die = 0.1, rrA = 0.9)

# define vectorized functions
pDie <- function(decision, state, prob_die, rrA){
  # prob death is 0.1 if healthy and 0 otherwise
  prob_die * (state=="H") *
    # multiplied by a relative risk of 0.9 if the decision is A, and 1 otherwise
    rrA ^ (decision=="A")
}

utility <- function(state){
  1 * (state=="H") # utility is 1 if healthy and 0 otherwise
}

# run the model for 10 cycles
run_twig(mytwig, params = params, n_cycles = 10)

# see the vignettes for more examples
}
\seealso{
\href{https://hjalal.github.io/twig/index.html}{Getting started guide and vignettes}
}
