% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simulateTvcureData.R
\name{simulateTVcureData}
\alias{simulateTVcureData}
\title{Simulation of survival data with a cure fraction and time-varying covariates.}
\usage{
simulateTVcureData(n, seed, Tmax=300,
       f0F0 = list(f0=function(x) dweibull(x, 2.65, 100),
                   F0=function(x) pweibull(x, 2.65, 100)),
       beta, gam, shared.cov=TRUE,
       RC.dist=c("uniform","exponential","Tmax"),
       tRC.min = 120, mu.cens=40, get.details=TRUE)
}
\arguments{
\item{n}{Number of units.}

\item{seed}{Seed (integer) for the random TVcure data generator.}

\item{Tmax}{Maximum follow-up time after which a unit is considered cured in the absence of a previous event. (Default: 300).}

\item{f0F0}{List of length 2 providing the density \eqn{f_0(t)} and associated cdf \eqn{F_0(t)} governing the bounded hazard dynamic on (0,Tmax), with \eqn{F_0}(Tmax)=1.0. (Default: f0F0 = list(f0=function(x) dweibull(x, 2.65, 100), F0=function(x) pweibull(x, 2.65, 100))).}

\item{beta}{3-vector with the regression coefficients <beta> in the long-term (cure) survival (or quantum) submodel.}

\item{gam}{2-vector with the regression coefficients <gamma> in the short-term (cure) survival (or timing) submodel.}

\item{shared.cov}{Logical indicating whether shared covariates for both submodels are assumed, with then \eqn{x_1(t)=x_3(t)}. (Default: TRUE).}

\item{RC.dist}{Right-censoring distribution: "uniform" (Uniform on (\code{tRC.min},\code{Tmax})),"exponential" (with mean \code{mu.cens}) or "Tmax" (when right-censoring occurs at Tmax)}

\item{tRC.min}{Minimum right-censoring time value if the right-censoring time distribution is Uniform. (Default: 120).}

\item{mu.cens}{Mean of the right-censoring time distribution if it is Exponential. (Default: 40).}

\item{get.details}{Logical indicating if a detailed data frame \code{rawdata} including the sequence of time-varying covariate values should also be reported. (Default: TRUE).}
}
\value{
A list with following elements:
\itemize{
\item \code{seeds} : Seeds used to generate the data for each of the n units.
\item \code{tRC.min} : Minimum right-censoring time value if the right-censoring time distribution is Uniform.
\item \code{RC.dist} : Right-censoring distribution ("Uniform", "Exponential" or "Tmax").
\item \code{cure.rate} : Underlying proportion of cured units (i.e. without an observed event by \code{Tmax} if the follow-up is not interrupted by that time due to right-censoring).
\item \code{RC.rate} : Observed right-censoring rate.
\item \code{rawdata} : Data frame containing the generated data in a counting process format with the detailed follow-up for each unit until the event or right-censoring occurs:
  \itemize{
  \item \code{id} : Unit identificator for each row.
  \item \code{time} : Discrete observation times, starting at 1 for a given unit, until the end of its follow-up. The number of rows associated to a given unit corresponds to the follow-up duration.
  \item \code{event} : Event indicator (1 if it occured, 0 otherwise) for given unit at a given time.
  \item \code{z1, z2, z3, z4, x1, x2, x3, x4} : Covariate values for a given unit at a given time.
  }
\item \code{data.summary} : Data frame with n rows containing summarized information on the generated data for each unit:
  \itemize{
  \item \code{id} : Unit identificator (the ith row corresponding to the ith unit).
  \item \code{t.obs} : Observed event or right-censoring time.
  \item \code{delta} : Event indicator (1 if it occured, 0 otherwise).
  \item \code{t.true} : True (possibly unobserved) event time (Inf for a cured unit).
  \item \code{t.cens} : True (possibly unobserved) right-censoring time.
  \item \code{cured} : True (possibly unobserved) cure status.
  }
\item \code{parameters} : List containing the defining elements of the tvcure model:
  \itemize{
  \item \code{beta} : The regression parameters in the long-term survival (or quantum) submodel.
  \item \code{gam} : The regression parameters in the short-term survival (or timing) submodel.
  \item \code{f.theta} : A list of length 2 containing the functions defining the additive terms in the long-term survival (or quantum) submodel.
  \item \code{f.gam} : A list of length 2 containing the functions defining the additive terms in the short-term survival (or timing) submodel.
  \item \code{f0} : Density function governing the dynamic of the reference cumulative hazard on (0,Tmax).
  \item \code{F0} : CDF governing the dynamic of the reference cumulative hazard on (0,Tmax).
  }
\item \code{call} : Function call.
}
}
\description{
Simulation of cure survival data in a counting process format with time-varying covariates.
The population hazard at time t  underlying the tvcure model is, for given covariate values,
\deqn{h_p(t|{\bf v}(t),\tilde{\bf v}(t)) = \mathrm{e}^{\eta_\vartheta({\bf v}(t))+\eta_F(\tilde{\bf v}(t))}
f_0(t)S_0(t)^{\exp(\eta_F(\tilde{\bf v}(t)))-1}}
with linear predictors
\deqn{\eta_\vartheta({\bf v}(t)) = \beta_0 + \beta_1 z_1(t) + \beta_2 z_2 + f_1(x_1(t)) + f_2(x_2(t))}
\deqn{\eta_F(\tilde{{\bf v}}(t)) = \gamma_1 z_3(t) +  \gamma_2 z_4 +  \tilde{f}_1(x_3(t)) + \tilde{f}_2(x_4(t))}
where \eqn{{\bf v}(t)=(z_1(t),z_2,x_1(t),x_2(t))}, \eqn{\tilde{\bf v}(t)=(z_3(t),z_4,x_3(t),x_4(t))},
with time-varying covariates \eqn{x_1(t)}, \eqn{x_3(t)} assumed identical and shared by the 2 submodels when
\code{shared.cov} is TRUE.

The density \eqn{f_0(t)} governing the reference cumulative hazard dynamic is,
by default, a Weibull with shape parameter 2.65 and scale parameter 100,
ensuring that all susceptible units will experience the monitored event by time Tmax=300.

The functions defining the additive terms are
 \deqn{f_1(x_1)= -.63 + .57\arctan(4x_1) ~;~f_2(x_2)= -.5 \cos(2\pi x_2)}
 \deqn{\tilde{f}_1(\tilde{x}_3) = .15 - .5 \cos\big(\pi(\tilde{x}_3-.75)\big)~;~
 \tilde{f}_2(\tilde{x}_4) = .8\big(\tilde{x}_4-.5\big)}

Covariates are generated as follows:
\itemize{
 \item{ } \eqn{z_1(t), z_3(t)} are recentered time-varying Bernoulli(0.5) on \eqn{(0,T_{max})} ;
 \item{ } \eqn{z_2, z_4 \sim N(0,1)} ;
 \item{ } \eqn{x_1(t), x_2(t), x_3(t), x_4(t)} follow random cubic polynomial trajectories on \eqn{(0,T_{max})}.
}
More details can be found in Lambert and Kreyenfeld (2024).
}
\examples{
\donttest{
require(tvcure)
## Regression parameters
beta = c(beta0=.4, beta1=-.2, beta2=.15) ##  beta0 tunes the cure rate
gam = c(gam1=.2, gam2=.2)
## Data simulation
temp = simulateTVcureData(n=500, seed=123, beta=beta, gam=gam,
                          RC.dist="exponential",mu.cens=550)
head(temp$rawdata) ## Overview of the simulated raw data
head(temp$data.summary) ## Overview of the summarized data
with(temp, c(cure.rate=cure.rate,RC.rate=RC.rate)) ## Cure and RC rates
}

}
\references{
Lambert, P. and Kreyenfeld, M. (2025).
Time-varying exogenous covariates with frequently changing values in double additive cure survival model: an application to fertility.
\emph{Journal of the Royal Statistical Society, Series A}. <doi:10.1093/jrsssa/qnaf035>
}
\author{
Philippe Lambert \email{p.lambert@uliege.be}
}
