.character_entity_reference_modes <- c("named", "hex", "dec")

.fix_character_entity_references <- function(x, mode = "hex")
{
  mode <- match.arg(mode, .character_entity_reference_modes)
  for(i in .character_entity_reference_modes[.character_entity_reference_modes != mode]) {
    byc <- bycodes[, c(i, mode), drop = FALSE]
    byc <- byc[!duplicated(byc[, 1L]), , drop = FALSE]
    byc <- byc[byc[, 1L] %in% .get_character_entity_references(x, mode = i), , drop = FALSE]
    if(nrow(byc) > 0L) {
      for(j in 1:nrow(byc)) {
        x <- gsub(byc[j, 1L], byc[j, 2L], x, fixed = TRUE)
      }
    }
  }
  return(x)
}

.get_character_entity_references <- function(x, mode = "hex")
{
  mode <- match.arg(mode, .character_entity_reference_modes)
  gr <- switch(mode,
    "named" = gregexpr("\\&[^#&]+;", x),
    "hex" = gregexpr("\\&\\#x[[:alnum:]]+;", x),
    "dec" = gregexpr("\\&\\#[[:digit:]]+;", x)
  )
  unique(unlist(regmatches(x, gr)))
}
        

## ## List of entity names (title, unicode, named, hex, and dec)
##
## ## constructed from w3.org resource in the following way:
## bycodes <- readLines("http://www.w3.org/TR/xml-entity-names/bycodes.html")
##
## ## a nicely formatted page is: http://dev.w3.org/html5/html-author/charref
## 
## ## strip <a> tags
## bycodes <- bycodes[substr(bycodes, 1, 8) == "<a name="]
## bycodes <- sapply(strsplit(bycodes, ".png\">", fixed = TRUE), "[", 2L)
## bycodes <- strsplit(bycodes, "</a>, ", fixed = TRUE)
## 
## ## extract unicode symbols
## bycodes_u <- sapply(bycodes, "[", 1L)
## bycodes <- sapply(bycodes, "[", 2L)
## 
## ## extract title and names
## bycodes <- strsplit(bycodes, ", ", fixed = TRUE)
## bycodes_t <- gsub(" +$", "", sapply(bycodes, "[", 1L))
## bycodes_n <- sapply(bycodes, function(y) paste(y[-1L], collapse = " "))
## 
## ## split multiple unicodes and names
## bycodes_u <- strsplit(bycodes_u, " ", fixed = TRUE)
## bycodes_n <- strsplit(bycodes_n, " ", fixed = TRUE)
## n_u <- sapply(bycodes_u, length)
## n_n <- sapply(bycodes_n, length)
## 
## ## replicate multiple versions of same symbol
## bycodes <- do.call("rbind", lapply(seq_along(bycodes_t), function(i)
##   cbind(bycodes_t[i], rep(bycodes_u[[i]], each = n_n[i]), rep(bycodes_n[[i]], n_u[i]))))
## 
## ## hex codes
## bycodes_h <- gsub("U+", "", bycodes[, 2], fixed = TRUE)
## 
## ## collect in data.frame
## bycodes <- data.frame(title = bycodes[, 1L], unicode = bycodes[, 2L],
##   named = paste("&", bycodes[, 3L], ";", sep = ""),
##   hex = paste("&#x", bycodes_h, ";", sep = ""),
##   dec = paste("&#", as.numeric(as.hexmode(bycodes_h)), ";", sep = ""),
##   stringsAsFactors = FALSE)

bycodes <- structure(list(title = c("CHARACTER TABULATION", "LINE FEED (LF)", 
"EXCLAMATION MARK", "QUOTATION MARK", "QUOTATION MARK", "NUMBER SIGN", 
"DOLLAR SIGN", "PERCENT SIGN", "AMPERSAND", "AMPERSAND", "APOSTROPHE", 
"LEFT PARENTHESIS", "RIGHT PARENTHESIS", "ASTERISK", "ASTERISK", 
"PLUS SIGN", "COMMA", "FULL STOP", "SOLIDUS", "COLON", "SEMICOLON", 
"LESS-THAN SIGN", "LESS-THAN SIGN", "LESS-THAN SIGN with vertical line", 
"LESS-THAN SIGN with vertical line", "EQUALS SIGN", "EQUALS SIGN with reverse slash", 
"EQUALS SIGN with reverse slash", "GREATER-THAN SIGN", "GREATER-THAN SIGN", 
"GREATER-THAN SIGN with vertical line", "GREATER-THAN SIGN with vertical line", 
"QUESTION MARK", "COMMERCIAL AT", "LEFT SQUARE BRACKET", "LEFT SQUARE BRACKET", 
"REVERSE SOLIDUS", "RIGHT SQUARE BRACKET", "RIGHT SQUARE BRACKET", 
"CIRCUMFLEX ACCENT", "LOW LINE", "LOW LINE", "GRAVE ACCENT", 
"GRAVE ACCENT", "fj ligature", "fj ligature", "LEFT CURLY BRACKET", 
"LEFT CURLY BRACKET", "VERTICAL LINE", "VERTICAL LINE", "VERTICAL LINE", 
"RIGHT CURLY BRACKET", "RIGHT CURLY BRACKET", "NO-BREAK SPACE", 
"NO-BREAK SPACE", "INVERTED EXCLAMATION MARK", "CENT SIGN", "POUND SIGN", 
"CURRENCY SIGN", "YEN SIGN", "BROKEN BAR", "SECTION SIGN", "DIAERESIS", 
"DIAERESIS", "DIAERESIS", "DIAERESIS", "COPYRIGHT SIGN", "COPYRIGHT SIGN", 
"FEMININE ORDINAL INDICATOR", "LEFT-POINTING DOUBLE ANGLE QUOTATION MARK", 
"NOT SIGN", "SOFT HYPHEN", "REGISTERED SIGN", "REGISTERED SIGN", 
"REGISTERED SIGN", "MACRON", "MACRON", "DEGREE SIGN", "PLUS-MINUS SIGN", 
"PLUS-MINUS SIGN", "PLUS-MINUS SIGN", "SUPERSCRIPT TWO", "SUPERSCRIPT THREE", 
"ACUTE ACCENT", "ACUTE ACCENT", "MICRO SIGN", "PILCROW SIGN", 
"MIDDLE DOT", "MIDDLE DOT", "MIDDLE DOT", "CEDILLA", "CEDILLA", 
"SUPERSCRIPT ONE", "MASCULINE ORDINAL INDICATOR", "RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK", 
"VULGAR FRACTION ONE QUARTER", "VULGAR FRACTION ONE HALF", "VULGAR FRACTION ONE HALF", 
"VULGAR FRACTION THREE QUARTERS", "INVERTED QUESTION MARK", "LATIN CAPITAL LETTER A WITH GRAVE", 
"LATIN CAPITAL LETTER A WITH ACUTE", "LATIN CAPITAL LETTER A WITH CIRCUMFLEX", 
"LATIN CAPITAL LETTER A WITH TILDE", "LATIN CAPITAL LETTER A WITH DIAERESIS", 
"LATIN CAPITAL LETTER A WITH RING ABOVE", "LATIN CAPITAL LETTER A WITH RING ABOVE", 
"LATIN CAPITAL LETTER AE", "LATIN CAPITAL LETTER C WITH CEDILLA", 
"LATIN CAPITAL LETTER E WITH GRAVE", "LATIN CAPITAL LETTER E WITH ACUTE", 
"LATIN CAPITAL LETTER E WITH CIRCUMFLEX", "LATIN CAPITAL LETTER E WITH DIAERESIS", 
"LATIN CAPITAL LETTER I WITH GRAVE", "LATIN CAPITAL LETTER I WITH ACUTE", 
"LATIN CAPITAL LETTER I WITH CIRCUMFLEX", "LATIN CAPITAL LETTER I WITH DIAERESIS", 
"LATIN CAPITAL LETTER ETH", "LATIN CAPITAL LETTER N WITH TILDE", 
"LATIN CAPITAL LETTER O WITH GRAVE", "LATIN CAPITAL LETTER O WITH ACUTE", 
"LATIN CAPITAL LETTER O WITH CIRCUMFLEX", "LATIN CAPITAL LETTER O WITH TILDE", 
"LATIN CAPITAL LETTER O WITH DIAERESIS", "MULTIPLICATION SIGN", 
"LATIN CAPITAL LETTER O WITH STROKE", "LATIN CAPITAL LETTER U WITH GRAVE", 
"LATIN CAPITAL LETTER U WITH ACUTE", "LATIN CAPITAL LETTER U WITH CIRCUMFLEX", 
"LATIN CAPITAL LETTER U WITH DIAERESIS", "LATIN CAPITAL LETTER Y WITH ACUTE", 
"LATIN CAPITAL LETTER THORN", "LATIN SMALL LETTER SHARP S", "LATIN SMALL LETTER A WITH GRAVE", 
"LATIN SMALL LETTER A WITH ACUTE", "LATIN SMALL LETTER A WITH CIRCUMFLEX", 
"LATIN SMALL LETTER A WITH TILDE", "LATIN SMALL LETTER A WITH DIAERESIS", 
"LATIN SMALL LETTER A WITH RING ABOVE", "LATIN SMALL LETTER AE", 
"LATIN SMALL LETTER C WITH CEDILLA", "LATIN SMALL LETTER E WITH GRAVE", 
"LATIN SMALL LETTER E WITH ACUTE", "LATIN SMALL LETTER E WITH CIRCUMFLEX", 
"LATIN SMALL LETTER E WITH DIAERESIS", "LATIN SMALL LETTER I WITH GRAVE", 
"LATIN SMALL LETTER I WITH ACUTE", "LATIN SMALL LETTER I WITH CIRCUMFLEX", 
"LATIN SMALL LETTER I WITH DIAERESIS", "LATIN SMALL LETTER ETH", 
"LATIN SMALL LETTER N WITH TILDE", "LATIN SMALL LETTER O WITH GRAVE", 
"LATIN SMALL LETTER O WITH ACUTE", "LATIN SMALL LETTER O WITH CIRCUMFLEX", 
"LATIN SMALL LETTER O WITH TILDE", "LATIN SMALL LETTER O WITH DIAERESIS", 
"DIVISION SIGN", "DIVISION SIGN", "LATIN SMALL LETTER O WITH STROKE", 
"LATIN SMALL LETTER U WITH GRAVE", "LATIN SMALL LETTER U WITH ACUTE", 
"LATIN SMALL LETTER U WITH CIRCUMFLEX", "LATIN SMALL LETTER U WITH DIAERESIS", 
"LATIN SMALL LETTER Y WITH ACUTE", "LATIN SMALL LETTER THORN", 
"LATIN SMALL LETTER Y WITH DIAERESIS", "LATIN CAPITAL LETTER A WITH MACRON", 
"LATIN SMALL LETTER A WITH MACRON", "LATIN CAPITAL LETTER A WITH BREVE", 
"LATIN SMALL LETTER A WITH BREVE", "LATIN CAPITAL LETTER A WITH OGONEK", 
"LATIN SMALL LETTER A WITH OGONEK", "LATIN CAPITAL LETTER C WITH ACUTE", 
"LATIN SMALL LETTER C WITH ACUTE", "LATIN CAPITAL LETTER C WITH CIRCUMFLEX", 
"LATIN SMALL LETTER C WITH CIRCUMFLEX", "LATIN CAPITAL LETTER C WITH DOT ABOVE", 
"LATIN SMALL LETTER C WITH DOT ABOVE", "LATIN CAPITAL LETTER C WITH CARON", 
"LATIN SMALL LETTER C WITH CARON", "LATIN CAPITAL LETTER D WITH CARON", 
"LATIN SMALL LETTER D WITH CARON", "LATIN CAPITAL LETTER D WITH STROKE", 
"LATIN SMALL LETTER D WITH STROKE", "LATIN CAPITAL LETTER E WITH MACRON", 
"LATIN SMALL LETTER E WITH MACRON", "LATIN CAPITAL LETTER E WITH DOT ABOVE", 
"LATIN SMALL LETTER E WITH DOT ABOVE", "LATIN CAPITAL LETTER E WITH OGONEK", 
"LATIN SMALL LETTER E WITH OGONEK", "LATIN CAPITAL LETTER E WITH CARON", 
"LATIN SMALL LETTER E WITH CARON", "LATIN CAPITAL LETTER G WITH CIRCUMFLEX", 
"LATIN SMALL LETTER G WITH CIRCUMFLEX", "LATIN CAPITAL LETTER G WITH BREVE", 
"LATIN SMALL LETTER G WITH BREVE", "LATIN CAPITAL LETTER G WITH DOT ABOVE", 
"LATIN SMALL LETTER G WITH DOT ABOVE", "LATIN CAPITAL LETTER G WITH CEDILLA", 
"LATIN CAPITAL LETTER H WITH CIRCUMFLEX", "LATIN SMALL LETTER H WITH CIRCUMFLEX", 
"LATIN CAPITAL LETTER H WITH STROKE", "LATIN SMALL LETTER H WITH STROKE", 
"LATIN CAPITAL LETTER I WITH TILDE", "LATIN SMALL LETTER I WITH TILDE", 
"LATIN CAPITAL LETTER I WITH MACRON", "LATIN SMALL LETTER I WITH MACRON", 
"LATIN CAPITAL LETTER I WITH OGONEK", "LATIN SMALL LETTER I WITH OGONEK", 
"LATIN CAPITAL LETTER I WITH DOT ABOVE", "LATIN SMALL LETTER DOTLESS I", 
"LATIN SMALL LETTER DOTLESS I", "LATIN CAPITAL LIGATURE IJ", 
"LATIN SMALL LIGATURE IJ", "LATIN CAPITAL LETTER J WITH CIRCUMFLEX", 
"LATIN SMALL LETTER J WITH CIRCUMFLEX", "LATIN CAPITAL LETTER K WITH CEDILLA", 
"LATIN SMALL LETTER K WITH CEDILLA", "LATIN SMALL LETTER KRA", 
"LATIN CAPITAL LETTER L WITH ACUTE", "LATIN SMALL LETTER L WITH ACUTE", 
"LATIN CAPITAL LETTER L WITH CEDILLA", "LATIN SMALL LETTER L WITH CEDILLA", 
"LATIN CAPITAL LETTER L WITH CARON", "LATIN SMALL LETTER L WITH CARON", 
"LATIN CAPITAL LETTER L WITH MIDDLE DOT", "LATIN SMALL LETTER L WITH MIDDLE DOT", 
"LATIN CAPITAL LETTER L WITH STROKE", "LATIN SMALL LETTER L WITH STROKE", 
"LATIN CAPITAL LETTER N WITH ACUTE", "LATIN SMALL LETTER N WITH ACUTE", 
"LATIN CAPITAL LETTER N WITH CEDILLA", "LATIN SMALL LETTER N WITH CEDILLA", 
"LATIN CAPITAL LETTER N WITH CARON", "LATIN SMALL LETTER N WITH CARON", 
"LATIN SMALL LETTER N PRECEDED BY APOSTROPHE", "LATIN CAPITAL LETTER ENG", 
"LATIN SMALL LETTER ENG", "LATIN CAPITAL LETTER O WITH MACRON", 
"LATIN SMALL LETTER O WITH MACRON", "LATIN CAPITAL LETTER O WITH DOUBLE ACUTE", 
"LATIN SMALL LETTER O WITH DOUBLE ACUTE", "LATIN CAPITAL LIGATURE OE", 
"LATIN SMALL LIGATURE OE", "LATIN CAPITAL LETTER R WITH ACUTE", 
"LATIN SMALL LETTER R WITH ACUTE", "LATIN CAPITAL LETTER R WITH CEDILLA", 
"LATIN SMALL LETTER R WITH CEDILLA", "LATIN CAPITAL LETTER R WITH CARON", 
"LATIN SMALL LETTER R WITH CARON", "LATIN CAPITAL LETTER S WITH ACUTE", 
"LATIN SMALL LETTER S WITH ACUTE", "LATIN CAPITAL LETTER S WITH CIRCUMFLEX", 
"LATIN SMALL LETTER S WITH CIRCUMFLEX", "LATIN CAPITAL LETTER S WITH CEDILLA", 
"LATIN SMALL LETTER S WITH CEDILLA", "LATIN CAPITAL LETTER S WITH CARON", 
"LATIN SMALL LETTER S WITH CARON", "LATIN CAPITAL LETTER T WITH CEDILLA", 
"LATIN SMALL LETTER T WITH CEDILLA", "LATIN CAPITAL LETTER T WITH CARON", 
"LATIN SMALL LETTER T WITH CARON", "LATIN CAPITAL LETTER T WITH STROKE", 
"LATIN SMALL LETTER T WITH STROKE", "LATIN CAPITAL LETTER U WITH TILDE", 
"LATIN SMALL LETTER U WITH TILDE", "LATIN CAPITAL LETTER U WITH MACRON", 
"LATIN SMALL LETTER U WITH MACRON", "LATIN CAPITAL LETTER U WITH BREVE", 
"LATIN SMALL LETTER U WITH BREVE", "LATIN CAPITAL LETTER U WITH RING ABOVE", 
"LATIN SMALL LETTER U WITH RING ABOVE", "LATIN CAPITAL LETTER U WITH DOUBLE ACUTE", 
"LATIN SMALL LETTER U WITH DOUBLE ACUTE", "LATIN CAPITAL LETTER U WITH OGONEK", 
"LATIN SMALL LETTER U WITH OGONEK", "LATIN CAPITAL LETTER W WITH CIRCUMFLEX", 
"LATIN SMALL LETTER W WITH CIRCUMFLEX", "LATIN CAPITAL LETTER Y WITH CIRCUMFLEX", 
"LATIN SMALL LETTER Y WITH CIRCUMFLEX", "LATIN CAPITAL LETTER Y WITH DIAERESIS", 
"LATIN CAPITAL LETTER Z WITH ACUTE", "LATIN SMALL LETTER Z WITH ACUTE", 
"LATIN CAPITAL LETTER Z WITH DOT ABOVE", "LATIN SMALL LETTER Z WITH DOT ABOVE", 
"LATIN CAPITAL LETTER Z WITH CARON", "LATIN SMALL LETTER Z WITH CARON", 
"LATIN SMALL LETTER F WITH HOOK", "LATIN CAPITAL LETTER Z WITH STROKE", 
"LATIN SMALL LETTER G WITH ACUTE", "LATIN SMALL LETTER DOTLESS J", 
"MODIFIER LETTER CIRCUMFLEX ACCENT", "CARON", "CARON", "BREVE", 
"BREVE", "DOT ABOVE", "DOT ABOVE", "RING ABOVE", "OGONEK", "SMALL TILDE", 
"SMALL TILDE", "DOUBLE ACUTE ACCENT", "DOUBLE ACUTE ACCENT", 
"COMBINING INVERTED BREVE", "GREEK CAPITAL LETTER ALPHA WITH TONOS", 
"GREEK CAPITAL LETTER EPSILON WITH TONOS", "GREEK CAPITAL LETTER ETA WITH TONOS", 
"GREEK CAPITAL LETTER IOTA WITH TONOS", "GREEK CAPITAL LETTER OMICRON WITH TONOS", 
"GREEK CAPITAL LETTER UPSILON WITH TONOS", "GREEK CAPITAL LETTER OMEGA WITH TONOS", 
"GREEK SMALL LETTER IOTA WITH DIALYTIKA AND TONOS", "GREEK CAPITAL LETTER ALPHA", 
"GREEK CAPITAL LETTER ALPHA", "GREEK CAPITAL LETTER BETA", "GREEK CAPITAL LETTER BETA", 
"GREEK CAPITAL LETTER GAMMA", "GREEK CAPITAL LETTER GAMMA", "GREEK CAPITAL LETTER DELTA", 
"GREEK CAPITAL LETTER DELTA", "GREEK CAPITAL LETTER EPSILON", 
"GREEK CAPITAL LETTER EPSILON", "GREEK CAPITAL LETTER ZETA", 
"GREEK CAPITAL LETTER ZETA", "GREEK CAPITAL LETTER ETA", "GREEK CAPITAL LETTER ETA", 
"GREEK CAPITAL LETTER THETA", "GREEK CAPITAL LETTER THETA", "GREEK CAPITAL LETTER IOTA", 
"GREEK CAPITAL LETTER IOTA", "GREEK CAPITAL LETTER KAPPA", "GREEK CAPITAL LETTER KAPPA", 
"GREEK CAPITAL LETTER LAMDA", "GREEK CAPITAL LETTER LAMDA", "GREEK CAPITAL LETTER MU", 
"GREEK CAPITAL LETTER MU", "GREEK CAPITAL LETTER NU", "GREEK CAPITAL LETTER NU", 
"GREEK CAPITAL LETTER XI", "GREEK CAPITAL LETTER XI", "GREEK CAPITAL LETTER OMICRON", 
"GREEK CAPITAL LETTER OMICRON", "GREEK CAPITAL LETTER PI", "GREEK CAPITAL LETTER PI", 
"GREEK CAPITAL LETTER RHO", "GREEK CAPITAL LETTER RHO", "GREEK CAPITAL LETTER SIGMA", 
"GREEK CAPITAL LETTER SIGMA", "GREEK CAPITAL LETTER TAU", "GREEK CAPITAL LETTER TAU", 
"GREEK CAPITAL LETTER UPSILON", "GREEK CAPITAL LETTER UPSILON", 
"GREEK CAPITAL LETTER PHI", "GREEK CAPITAL LETTER PHI", "GREEK CAPITAL LETTER CHI", 
"GREEK CAPITAL LETTER CHI", "GREEK CAPITAL LETTER PSI", "GREEK CAPITAL LETTER PSI", 
"GREEK CAPITAL LETTER OMEGA", "GREEK CAPITAL LETTER OMEGA", "GREEK CAPITAL LETTER OMEGA", 
"GREEK CAPITAL LETTER IOTA WITH DIALYTIKA", "GREEK CAPITAL LETTER UPSILON WITH DIALYTIKA", 
"GREEK SMALL LETTER ALPHA WITH TONOS", "GREEK SMALL LETTER EPSILON WITH TONOS", 
"GREEK SMALL LETTER ETA WITH TONOS", "GREEK SMALL LETTER IOTA WITH TONOS", 
"GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND TONOS", "GREEK SMALL LETTER ALPHA", 
"GREEK SMALL LETTER ALPHA", "GREEK SMALL LETTER BETA", "GREEK SMALL LETTER BETA", 
"GREEK SMALL LETTER GAMMA", "GREEK SMALL LETTER GAMMA", "GREEK SMALL LETTER DELTA", 
"GREEK SMALL LETTER DELTA", "GREEK SMALL LETTER EPSILON", "GREEK SMALL LETTER EPSILON", 
"GREEK SMALL LETTER EPSILON", "GREEK SMALL LETTER ZETA", "GREEK SMALL LETTER ZETA", 
"GREEK SMALL LETTER ETA", "GREEK SMALL LETTER ETA", "GREEK SMALL LETTER THETA", 
"GREEK SMALL LETTER THETA", "GREEK SMALL LETTER IOTA", "GREEK SMALL LETTER IOTA", 
"GREEK SMALL LETTER KAPPA", "GREEK SMALL LETTER KAPPA", "GREEK SMALL LETTER LAMDA", 
"GREEK SMALL LETTER LAMDA", "GREEK SMALL LETTER MU", "GREEK SMALL LETTER MU", 
"GREEK SMALL LETTER NU", "GREEK SMALL LETTER NU", "GREEK SMALL LETTER XI", 
"GREEK SMALL LETTER XI", "GREEK SMALL LETTER OMICRON", "GREEK SMALL LETTER OMICRON", 
"GREEK SMALL LETTER PI", "GREEK SMALL LETTER PI", "GREEK SMALL LETTER RHO", 
"GREEK SMALL LETTER RHO", "GREEK SMALL LETTER FINAL SIGMA", "GREEK SMALL LETTER FINAL SIGMA", 
"GREEK SMALL LETTER FINAL SIGMA", "GREEK SMALL LETTER FINAL SIGMA", 
"GREEK SMALL LETTER SIGMA", "GREEK SMALL LETTER SIGMA", "GREEK SMALL LETTER TAU", 
"GREEK SMALL LETTER TAU", "GREEK SMALL LETTER UPSILON", "GREEK SMALL LETTER UPSILON", 
"GREEK SMALL LETTER UPSILON", "GREEK SMALL LETTER PHI", "GREEK SMALL LETTER PHI", 
"GREEK SMALL LETTER CHI", "GREEK SMALL LETTER CHI", "GREEK SMALL LETTER PSI", 
"GREEK SMALL LETTER PSI", "GREEK SMALL LETTER OMEGA", "GREEK SMALL LETTER OMEGA", 
"GREEK SMALL LETTER IOTA WITH DIALYTIKA", "GREEK SMALL LETTER UPSILON WITH DIALYTIKA", 
"GREEK SMALL LETTER OMICRON WITH TONOS", "GREEK SMALL LETTER UPSILON WITH TONOS", 
"GREEK SMALL LETTER OMEGA WITH TONOS", "GREEK THETA SYMBOL", 
"GREEK THETA SYMBOL", "GREEK THETA SYMBOL", "GREEK UPSILON WITH HOOK SYMBOL", 
"GREEK UPSILON WITH HOOK SYMBOL", "GREEK PHI SYMBOL", "GREEK PHI SYMBOL", 
"GREEK PHI SYMBOL", "GREEK PI SYMBOL", "GREEK PI SYMBOL", "GREEK LETTER DIGAMMA", 
"GREEK SMALL LETTER DIGAMMA", "GREEK SMALL LETTER DIGAMMA", "GREEK KAPPA SYMBOL", 
"GREEK KAPPA SYMBOL", "GREEK RHO SYMBOL", "GREEK RHO SYMBOL", 
"GREEK LUNATE EPSILON SYMBOL", "GREEK LUNATE EPSILON SYMBOL", 
"GREEK LUNATE EPSILON SYMBOL", "GREEK REVERSED LUNATE EPSILON SYMBOL", 
"GREEK REVERSED LUNATE EPSILON SYMBOL", "CYRILLIC CAPITAL LETTER IO", 
"CYRILLIC CAPITAL LETTER DJE", "CYRILLIC CAPITAL LETTER GJE", 
"CYRILLIC CAPITAL LETTER UKRAINIAN IE", "CYRILLIC CAPITAL LETTER DZE", 
"CYRILLIC CAPITAL LETTER BYELORUSSIAN-UKRAINIAN I", "CYRILLIC CAPITAL LETTER YI", 
"CYRILLIC CAPITAL LETTER JE", "CYRILLIC CAPITAL LETTER LJE", 
"CYRILLIC CAPITAL LETTER NJE", "CYRILLIC CAPITAL LETTER TSHE", 
"CYRILLIC CAPITAL LETTER KJE", "CYRILLIC CAPITAL LETTER SHORT U", 
"CYRILLIC CAPITAL LETTER DZHE", "CYRILLIC CAPITAL LETTER A", 
"CYRILLIC CAPITAL LETTER BE", "CYRILLIC CAPITAL LETTER VE", "CYRILLIC CAPITAL LETTER GHE", 
"CYRILLIC CAPITAL LETTER DE", "CYRILLIC CAPITAL LETTER IE", "CYRILLIC CAPITAL LETTER ZHE", 
"CYRILLIC CAPITAL LETTER ZE", "CYRILLIC CAPITAL LETTER I", "CYRILLIC CAPITAL LETTER SHORT I", 
"CYRILLIC CAPITAL LETTER KA", "CYRILLIC CAPITAL LETTER EL", "CYRILLIC CAPITAL LETTER EM", 
"CYRILLIC CAPITAL LETTER EN", "CYRILLIC CAPITAL LETTER O", "CYRILLIC CAPITAL LETTER PE", 
"CYRILLIC CAPITAL LETTER ER", "CYRILLIC CAPITAL LETTER ES", "CYRILLIC CAPITAL LETTER TE", 
"CYRILLIC CAPITAL LETTER U", "CYRILLIC CAPITAL LETTER EF", "CYRILLIC CAPITAL LETTER HA", 
"CYRILLIC CAPITAL LETTER TSE", "CYRILLIC CAPITAL LETTER CHE", 
"CYRILLIC CAPITAL LETTER SHA", "CYRILLIC CAPITAL LETTER SHCHA", 
"CYRILLIC CAPITAL LETTER HARD SIGN", "CYRILLIC CAPITAL LETTER YERU", 
"CYRILLIC CAPITAL LETTER SOFT SIGN", "CYRILLIC CAPITAL LETTER E", 
"CYRILLIC CAPITAL LETTER YU", "CYRILLIC CAPITAL LETTER YA", "CYRILLIC SMALL LETTER A", 
"CYRILLIC SMALL LETTER BE", "CYRILLIC SMALL LETTER VE", "CYRILLIC SMALL LETTER GHE", 
"CYRILLIC SMALL LETTER DE", "CYRILLIC SMALL LETTER IE", "CYRILLIC SMALL LETTER ZHE", 
"CYRILLIC SMALL LETTER ZE", "CYRILLIC SMALL LETTER I", "CYRILLIC SMALL LETTER SHORT I", 
"CYRILLIC SMALL LETTER KA", "CYRILLIC SMALL LETTER EL", "CYRILLIC SMALL LETTER EM", 
"CYRILLIC SMALL LETTER EN", "CYRILLIC SMALL LETTER O", "CYRILLIC SMALL LETTER PE", 
"CYRILLIC SMALL LETTER ER", "CYRILLIC SMALL LETTER ES", "CYRILLIC SMALL LETTER TE", 
"CYRILLIC SMALL LETTER U", "CYRILLIC SMALL LETTER EF", "CYRILLIC SMALL LETTER HA", 
"CYRILLIC SMALL LETTER TSE", "CYRILLIC SMALL LETTER CHE", "CYRILLIC SMALL LETTER SHA", 
"CYRILLIC SMALL LETTER SHCHA", "CYRILLIC SMALL LETTER HARD SIGN", 
"CYRILLIC SMALL LETTER YERU", "CYRILLIC SMALL LETTER SOFT SIGN", 
"CYRILLIC SMALL LETTER E", "CYRILLIC SMALL LETTER YU", "CYRILLIC SMALL LETTER YA", 
"CYRILLIC SMALL LETTER IO", "CYRILLIC SMALL LETTER DJE", "CYRILLIC SMALL LETTER GJE", 
"CYRILLIC SMALL LETTER UKRAINIAN IE", "CYRILLIC SMALL LETTER DZE", 
"CYRILLIC SMALL LETTER BYELORUSSIAN-UKRAINIAN I", "CYRILLIC SMALL LETTER YI", 
"CYRILLIC SMALL LETTER JE", "CYRILLIC SMALL LETTER LJE", "CYRILLIC SMALL LETTER NJE", 
"CYRILLIC SMALL LETTER TSHE", "CYRILLIC SMALL LETTER KJE", "CYRILLIC SMALL LETTER SHORT U", 
"CYRILLIC SMALL LETTER DZHE", "EN SPACE", "EM SPACE", "THREE-PER-EM SPACE", 
"FOUR-PER-EM SPACE", "FIGURE SPACE", "PUNCTUATION SPACE", "THIN SPACE", 
"THIN SPACE", "HAIR SPACE", "HAIR SPACE", "ZERO WIDTH SPACE", 
"ZERO WIDTH SPACE", "ZERO WIDTH SPACE", "ZERO WIDTH SPACE", "ZERO WIDTH SPACE", 
"ZERO WIDTH NON-JOINER", "ZERO WIDTH JOINER", "LEFT-TO-RIGHT MARK", 
"RIGHT-TO-LEFT MARK", "HYPHEN", "HYPHEN", "EN DASH", "EM DASH", 
"HORIZONTAL BAR", "DOUBLE VERTICAL LINE", "DOUBLE VERTICAL LINE", 
"LEFT SINGLE QUOTATION MARK", "LEFT SINGLE QUOTATION MARK", "RIGHT SINGLE QUOTATION MARK", 
"RIGHT SINGLE QUOTATION MARK", "RIGHT SINGLE QUOTATION MARK", 
"SINGLE LOW-9 QUOTATION MARK", "SINGLE LOW-9 QUOTATION MARK", 
"LEFT DOUBLE QUOTATION MARK", "LEFT DOUBLE QUOTATION MARK", "RIGHT DOUBLE QUOTATION MARK", 
"RIGHT DOUBLE QUOTATION MARK", "RIGHT DOUBLE QUOTATION MARK", 
"DOUBLE LOW-9 QUOTATION MARK", "DOUBLE LOW-9 QUOTATION MARK", 
"DAGGER", "DOUBLE DAGGER", "DOUBLE DAGGER", "BULLET", "BULLET", 
"TWO DOT LEADER", "HORIZONTAL ELLIPSIS", "HORIZONTAL ELLIPSIS", 
"PER MILLE SIGN", "PER TEN THOUSAND SIGN", "PRIME", "DOUBLE PRIME", 
"TRIPLE PRIME", "REVERSED PRIME", "REVERSED PRIME", "SINGLE LEFT-POINTING ANGLE QUOTATION MARK", 
"SINGLE RIGHT-POINTING ANGLE QUOTATION MARK", "OVERLINE", "OVERLINE", 
"CARET INSERTION POINT", "HYPHEN BULLET", "FRACTION SLASH", "REVERSED SEMICOLON", 
"QUADRUPLE PRIME", "MEDIUM MATHEMATICAL SPACE", "space of width 5/18 em", 
"space of width 5/18 em", "WORD JOINER", "FUNCTION APPLICATION", 
"FUNCTION APPLICATION", "INVISIBLE TIMES", "INVISIBLE TIMES", 
"INVISIBLE SEPARATOR", "INVISIBLE SEPARATOR", "EURO SIGN", "COMBINING THREE DOTS ABOVE", 
"COMBINING THREE DOTS ABOVE", "COMBINING FOUR DOTS ABOVE", "DOUBLE-STRUCK CAPITAL C", 
"DOUBLE-STRUCK CAPITAL C", "CARE OF", "SCRIPT SMALL G", "SCRIPT CAPITAL H", 
"SCRIPT CAPITAL H", "SCRIPT CAPITAL H", "BLACK-LETTER CAPITAL H", 
"BLACK-LETTER CAPITAL H", "DOUBLE-STRUCK CAPITAL H", "DOUBLE-STRUCK CAPITAL H", 
"PLANCK CONSTANT", "PLANCK CONSTANT OVER TWO PI", "PLANCK CONSTANT OVER TWO PI", 
"PLANCK CONSTANT OVER TWO PI", "PLANCK CONSTANT OVER TWO PI", 
"SCRIPT CAPITAL I", "SCRIPT CAPITAL I", "BLACK-LETTER CAPITAL I", 
"BLACK-LETTER CAPITAL I", "BLACK-LETTER CAPITAL I", "BLACK-LETTER CAPITAL I", 
"SCRIPT CAPITAL L", "SCRIPT CAPITAL L", "SCRIPT CAPITAL L", "SCRIPT SMALL L", 
"DOUBLE-STRUCK CAPITAL N", "DOUBLE-STRUCK CAPITAL N", "NUMERO SIGN", 
"SOUND RECORDING COPYRIGHT", "SCRIPT CAPITAL P", "SCRIPT CAPITAL P", 
"DOUBLE-STRUCK CAPITAL P", "DOUBLE-STRUCK CAPITAL P", "DOUBLE-STRUCK CAPITAL Q", 
"DOUBLE-STRUCK CAPITAL Q", "SCRIPT CAPITAL R", "SCRIPT CAPITAL R", 
"BLACK-LETTER CAPITAL R", "BLACK-LETTER CAPITAL R", "BLACK-LETTER CAPITAL R", 
"BLACK-LETTER CAPITAL R", "DOUBLE-STRUCK CAPITAL R", "DOUBLE-STRUCK CAPITAL R", 
"PRESCRIPTION TAKE", "TRADE MARK SIGN", "TRADE MARK SIGN", "DOUBLE-STRUCK CAPITAL Z", 
"DOUBLE-STRUCK CAPITAL Z", "INVERTED OHM SIGN", "BLACK-LETTER CAPITAL Z", 
"BLACK-LETTER CAPITAL Z", "TURNED GREEK SMALL LETTER IOTA", "SCRIPT CAPITAL B", 
"SCRIPT CAPITAL B", "SCRIPT CAPITAL B", "BLACK-LETTER CAPITAL C", 
"BLACK-LETTER CAPITAL C", "SCRIPT SMALL E", "SCRIPT CAPITAL E", 
"SCRIPT CAPITAL E", "SCRIPT CAPITAL F", "SCRIPT CAPITAL F", "SCRIPT CAPITAL M", 
"SCRIPT CAPITAL M", "SCRIPT CAPITAL M", "SCRIPT SMALL O", "SCRIPT SMALL O", 
"SCRIPT SMALL O", "ALEF SYMBOL", "ALEF SYMBOL", "BET SYMBOL", 
"GIMEL SYMBOL", "DALET SYMBOL", "DOUBLE-STRUCK ITALIC CAPITAL D", 
"DOUBLE-STRUCK ITALIC CAPITAL D", "DOUBLE-STRUCK ITALIC SMALL D", 
"DOUBLE-STRUCK ITALIC SMALL D", "DOUBLE-STRUCK ITALIC SMALL E", 
"DOUBLE-STRUCK ITALIC SMALL E", "DOUBLE-STRUCK ITALIC SMALL E", 
"DOUBLE-STRUCK ITALIC SMALL I", "DOUBLE-STRUCK ITALIC SMALL I", 
"VULGAR FRACTION ONE THIRD", "VULGAR FRACTION TWO THIRDS", "VULGAR FRACTION ONE FIFTH", 
"VULGAR FRACTION TWO FIFTHS", "VULGAR FRACTION THREE FIFTHS", 
"VULGAR FRACTION FOUR FIFTHS", "VULGAR FRACTION ONE SIXTH", "VULGAR FRACTION FIVE SIXTHS", 
"VULGAR FRACTION ONE EIGHTH", "VULGAR FRACTION THREE EIGHTHS", 
"VULGAR FRACTION FIVE EIGHTHS", "VULGAR FRACTION SEVEN EIGHTHS", 
"LEFTWARDS ARROW", "LEFTWARDS ARROW", "LEFTWARDS ARROW", "LEFTWARDS ARROW", 
"LEFTWARDS ARROW", "UPWARDS ARROW", "UPWARDS ARROW", "UPWARDS ARROW", 
"UPWARDS ARROW", "RIGHTWARDS ARROW", "RIGHTWARDS ARROW", "RIGHTWARDS ARROW", 
"RIGHTWARDS ARROW", "RIGHTWARDS ARROW", "DOWNWARDS ARROW", "DOWNWARDS ARROW", 
"DOWNWARDS ARROW", "DOWNWARDS ARROW", "LEFT RIGHT ARROW", "LEFT RIGHT ARROW", 
"LEFT RIGHT ARROW", "UP DOWN ARROW", "UP DOWN ARROW", "UP DOWN ARROW", 
"NORTH WEST ARROW", "NORTH WEST ARROW", "NORTH WEST ARROW", "NORTH EAST ARROW", 
"NORTH EAST ARROW", "NORTH EAST ARROW", "SOUTH EAST ARROW", "SOUTH EAST ARROW", 
"SOUTH EAST ARROW", "SOUTH WEST ARROW", "SOUTH WEST ARROW", "SOUTH WEST ARROW", 
"LEFTWARDS ARROW WITH STROKE", "LEFTWARDS ARROW WITH STROKE", 
"RIGHTWARDS ARROW WITH STROKE", "RIGHTWARDS ARROW WITH STROKE", 
"RIGHTWARDS WAVE ARROW", "RIGHTWARDS WAVE ARROW", "RIGHTWARDS WAVE ARROW with slash", 
"RIGHTWARDS WAVE ARROW with slash", "LEFTWARDS TWO HEADED ARROW", 
"LEFTWARDS TWO HEADED ARROW", "UPWARDS TWO HEADED ARROW", "RIGHTWARDS TWO HEADED ARROW", 
"RIGHTWARDS TWO HEADED ARROW", "DOWNWARDS TWO HEADED ARROW", 
"LEFTWARDS ARROW WITH TAIL", "LEFTWARDS ARROW WITH TAIL", "RIGHTWARDS ARROW WITH TAIL", 
"RIGHTWARDS ARROW WITH TAIL", "LEFTWARDS ARROW FROM BAR", "LEFTWARDS ARROW FROM BAR", 
"UPWARDS ARROW FROM BAR", "UPWARDS ARROW FROM BAR", "RIGHTWARDS ARROW FROM BAR", 
"RIGHTWARDS ARROW FROM BAR", "RIGHTWARDS ARROW FROM BAR", "DOWNWARDS ARROW FROM BAR", 
"DOWNWARDS ARROW FROM BAR", "LEFTWARDS ARROW WITH HOOK", "LEFTWARDS ARROW WITH HOOK", 
"RIGHTWARDS ARROW WITH HOOK", "RIGHTWARDS ARROW WITH HOOK", "LEFTWARDS ARROW WITH LOOP", 
"LEFTWARDS ARROW WITH LOOP", "RIGHTWARDS ARROW WITH LOOP", "RIGHTWARDS ARROW WITH LOOP", 
"LEFT RIGHT WAVE ARROW", "LEFT RIGHT WAVE ARROW", "LEFT RIGHT ARROW WITH STROKE", 
"LEFT RIGHT ARROW WITH STROKE", "UPWARDS ARROW WITH TIP LEFTWARDS", 
"UPWARDS ARROW WITH TIP LEFTWARDS", "UPWARDS ARROW WITH TIP RIGHTWARDS", 
"UPWARDS ARROW WITH TIP RIGHTWARDS", "DOWNWARDS ARROW WITH TIP LEFTWARDS", 
"DOWNWARDS ARROW WITH TIP RIGHTWARDS", "DOWNWARDS ARROW WITH CORNER LEFTWARDS", 
"ANTICLOCKWISE TOP SEMICIRCLE ARROW", "ANTICLOCKWISE TOP SEMICIRCLE ARROW", 
"CLOCKWISE TOP SEMICIRCLE ARROW", "CLOCKWISE TOP SEMICIRCLE ARROW", 
"ANTICLOCKWISE OPEN CIRCLE ARROW", "ANTICLOCKWISE OPEN CIRCLE ARROW", 
"CLOCKWISE OPEN CIRCLE ARROW", "CLOCKWISE OPEN CIRCLE ARROW", 
"LEFTWARDS HARPOON WITH BARB UPWARDS", "LEFTWARDS HARPOON WITH BARB UPWARDS", 
"LEFTWARDS HARPOON WITH BARB UPWARDS", "LEFTWARDS HARPOON WITH BARB DOWNWARDS", 
"LEFTWARDS HARPOON WITH BARB DOWNWARDS", "LEFTWARDS HARPOON WITH BARB DOWNWARDS", 
"UPWARDS HARPOON WITH BARB RIGHTWARDS", "UPWARDS HARPOON WITH BARB RIGHTWARDS", 
"UPWARDS HARPOON WITH BARB RIGHTWARDS", "UPWARDS HARPOON WITH BARB LEFTWARDS", 
"UPWARDS HARPOON WITH BARB LEFTWARDS", "UPWARDS HARPOON WITH BARB LEFTWARDS", 
"RIGHTWARDS HARPOON WITH BARB UPWARDS", "RIGHTWARDS HARPOON WITH BARB UPWARDS", 
"RIGHTWARDS HARPOON WITH BARB UPWARDS", "RIGHTWARDS HARPOON WITH BARB DOWNWARDS", 
"RIGHTWARDS HARPOON WITH BARB DOWNWARDS", "RIGHTWARDS HARPOON WITH BARB DOWNWARDS", 
"DOWNWARDS HARPOON WITH BARB RIGHTWARDS", "DOWNWARDS HARPOON WITH BARB RIGHTWARDS", 
"DOWNWARDS HARPOON WITH BARB RIGHTWARDS", "DOWNWARDS HARPOON WITH BARB LEFTWARDS", 
"DOWNWARDS HARPOON WITH BARB LEFTWARDS", "DOWNWARDS HARPOON WITH BARB LEFTWARDS", 
"RIGHTWARDS ARROW OVER LEFTWARDS ARROW", "RIGHTWARDS ARROW OVER LEFTWARDS ARROW", 
"RIGHTWARDS ARROW OVER LEFTWARDS ARROW", "UPWARDS ARROW LEFTWARDS OF DOWNWARDS ARROW", 
"UPWARDS ARROW LEFTWARDS OF DOWNWARDS ARROW", "LEFTWARDS ARROW OVER RIGHTWARDS ARROW", 
"LEFTWARDS ARROW OVER RIGHTWARDS ARROW", "LEFTWARDS ARROW OVER RIGHTWARDS ARROW", 
"LEFTWARDS PAIRED ARROWS", "LEFTWARDS PAIRED ARROWS", "UPWARDS PAIRED ARROWS", 
"UPWARDS PAIRED ARROWS", "RIGHTWARDS PAIRED ARROWS", "RIGHTWARDS PAIRED ARROWS", 
"DOWNWARDS PAIRED ARROWS", "DOWNWARDS PAIRED ARROWS", "LEFTWARDS HARPOON OVER RIGHTWARDS HARPOON", 
"LEFTWARDS HARPOON OVER RIGHTWARDS HARPOON", "LEFTWARDS HARPOON OVER RIGHTWARDS HARPOON", 
"RIGHTWARDS HARPOON OVER LEFTWARDS HARPOON", "RIGHTWARDS HARPOON OVER LEFTWARDS HARPOON", 
"RIGHTWARDS HARPOON OVER LEFTWARDS HARPOON", "LEFTWARDS DOUBLE ARROW WITH STROKE", 
"LEFTWARDS DOUBLE ARROW WITH STROKE", "LEFT RIGHT DOUBLE ARROW WITH STROKE", 
"LEFT RIGHT DOUBLE ARROW WITH STROKE", "RIGHTWARDS DOUBLE ARROW WITH STROKE", 
"RIGHTWARDS DOUBLE ARROW WITH STROKE", "LEFTWARDS DOUBLE ARROW", 
"LEFTWARDS DOUBLE ARROW", "LEFTWARDS DOUBLE ARROW", "UPWARDS DOUBLE ARROW", 
"UPWARDS DOUBLE ARROW", "UPWARDS DOUBLE ARROW", "RIGHTWARDS DOUBLE ARROW", 
"RIGHTWARDS DOUBLE ARROW", "RIGHTWARDS DOUBLE ARROW", "RIGHTWARDS DOUBLE ARROW", 
"DOWNWARDS DOUBLE ARROW", "DOWNWARDS DOUBLE ARROW", "DOWNWARDS DOUBLE ARROW", 
"LEFT RIGHT DOUBLE ARROW", "LEFT RIGHT DOUBLE ARROW", "LEFT RIGHT DOUBLE ARROW", 
"LEFT RIGHT DOUBLE ARROW", "UP DOWN DOUBLE ARROW", "UP DOWN DOUBLE ARROW", 
"UP DOWN DOUBLE ARROW", "NORTH WEST DOUBLE ARROW", "NORTH EAST DOUBLE ARROW", 
"SOUTH EAST DOUBLE ARROW", "SOUTH WEST DOUBLE ARROW", "LEFTWARDS TRIPLE ARROW", 
"LEFTWARDS TRIPLE ARROW", "RIGHTWARDS TRIPLE ARROW", "RIGHTWARDS TRIPLE ARROW", 
"RIGHTWARDS SQUIGGLE ARROW", "LEFTWARDS ARROW TO BAR", "LEFTWARDS ARROW TO BAR", 
"RIGHTWARDS ARROW TO BAR", "RIGHTWARDS ARROW TO BAR", "DOWNWARDS ARROW LEFTWARDS OF UPWARDS ARROW", 
"DOWNWARDS ARROW LEFTWARDS OF UPWARDS ARROW", "LEFTWARDS OPEN-HEADED ARROW", 
"RIGHTWARDS OPEN-HEADED ARROW", "LEFT RIGHT OPEN-HEADED ARROW", 
"FOR ALL", "FOR ALL", "COMPLEMENT", "COMPLEMENT", "PARTIAL DIFFERENTIAL", 
"PARTIAL DIFFERENTIAL", "PARTIAL DIFFERENTIAL with slash", "PARTIAL DIFFERENTIAL with slash", 
"THERE EXISTS", "THERE EXISTS", "THERE DOES NOT EXIST", "THERE DOES NOT EXIST", 
"THERE DOES NOT EXIST", "EMPTY SET", "EMPTY SET", "EMPTY SET", 
"EMPTY SET", "NABLA", "NABLA", "ELEMENT OF", "ELEMENT OF", "ELEMENT OF", 
"ELEMENT OF", "NOT AN ELEMENT OF", "NOT AN ELEMENT OF", "NOT AN ELEMENT OF", 
"CONTAINS AS MEMBER", "CONTAINS AS MEMBER", "CONTAINS AS MEMBER", 
"CONTAINS AS MEMBER", "DOES NOT CONTAIN AS MEMBER", "DOES NOT CONTAIN AS MEMBER", 
"DOES NOT CONTAIN AS MEMBER", "N-ARY PRODUCT", "N-ARY PRODUCT", 
"N-ARY COPRODUCT", "N-ARY COPRODUCT", "N-ARY SUMMATION", "N-ARY SUMMATION", 
"MINUS SIGN", "MINUS-OR-PLUS SIGN", "MINUS-OR-PLUS SIGN", "MINUS-OR-PLUS SIGN", 
"DOT PLUS", "DOT PLUS", "SET MINUS", "SET MINUS", "SET MINUS", 
"SET MINUS", "SET MINUS", "ASTERISK OPERATOR", "RING OPERATOR", 
"RING OPERATOR", "SQUARE ROOT", "SQUARE ROOT", "PROPORTIONAL TO", 
"PROPORTIONAL TO", "PROPORTIONAL TO", "PROPORTIONAL TO", "PROPORTIONAL TO", 
"INFINITY", "RIGHT ANGLE", "ANGLE", "ANGLE", "ANGLE with vertical line", 
"ANGLE with vertical line", "MEASURED ANGLE", "MEASURED ANGLE", 
"SPHERICAL ANGLE", "DIVIDES", "DIVIDES", "DIVIDES", "DIVIDES", 
"DOES NOT DIVIDE", "DOES NOT DIVIDE", "DOES NOT DIVIDE", "DOES NOT DIVIDE", 
"PARALLEL TO", "PARALLEL TO", "PARALLEL TO", "PARALLEL TO", "PARALLEL TO", 
"NOT PARALLEL TO", "NOT PARALLEL TO", "NOT PARALLEL TO", "NOT PARALLEL TO", 
"NOT PARALLEL TO", "LOGICAL AND", "LOGICAL AND", "LOGICAL OR", 
"LOGICAL OR", "INTERSECTION", "INTERSECTION with serifs", "INTERSECTION with serifs", 
"UNION", "UNION with serifs", "UNION with serifs", "INTEGRAL", 
"INTEGRAL", "DOUBLE INTEGRAL", "TRIPLE INTEGRAL", "TRIPLE INTEGRAL", 
"CONTOUR INTEGRAL", "CONTOUR INTEGRAL", "CONTOUR INTEGRAL", "SURFACE INTEGRAL", 
"SURFACE INTEGRAL", "VOLUME INTEGRAL", "CLOCKWISE INTEGRAL", 
"CLOCKWISE CONTOUR INTEGRAL", "CLOCKWISE CONTOUR INTEGRAL", "ANTICLOCKWISE CONTOUR INTEGRAL", 
"ANTICLOCKWISE CONTOUR INTEGRAL", "THEREFORE", "THEREFORE", "THEREFORE", 
"BECAUSE", "BECAUSE", "BECAUSE", "RATIO", "PROPORTION", "PROPORTION", 
"DOT MINUS", "DOT MINUS", "GEOMETRIC PROPORTION", "HOMOTHETIC", 
"TILDE OPERATOR", "TILDE OPERATOR", "TILDE OPERATOR", "TILDE OPERATOR", 
"TILDE OPERATOR with vertical line", "TILDE OPERATOR with vertical line", 
"REVERSED TILDE", "REVERSED TILDE", "REVERSED TILDE with underline", 
"REVERSED TILDE with underline", "INVERTED LAZY S", "INVERTED LAZY S", 
"INVERTED LAZY S with double underline", "INVERTED LAZY S with double underline", 
"SINE WAVE", "WREATH PRODUCT", "WREATH PRODUCT", "WREATH PRODUCT", 
"NOT TILDE", "NOT TILDE", "MINUS TILDE", "MINUS TILDE", "MINUS TILDE", 
"MINUS TILDE with slash", "MINUS TILDE with slash", "MINUS TILDE with slash", 
"MINUS TILDE with slash", "ASYMPTOTICALLY EQUAL TO", "ASYMPTOTICALLY EQUAL TO", 
"ASYMPTOTICALLY EQUAL TO", "NOT ASYMPTOTICALLY EQUAL TO", "NOT ASYMPTOTICALLY EQUAL TO", 
"NOT ASYMPTOTICALLY EQUAL TO", "APPROXIMATELY EQUAL TO", "APPROXIMATELY EQUAL TO", 
"APPROXIMATELY BUT NOT ACTUALLY EQUAL TO", "NEITHER APPROXIMATELY NOR ACTUALLY EQUAL TO", 
"NEITHER APPROXIMATELY NOR ACTUALLY EQUAL TO", "ALMOST EQUAL TO", 
"ALMOST EQUAL TO", "ALMOST EQUAL TO", "ALMOST EQUAL TO", "ALMOST EQUAL TO", 
"ALMOST EQUAL TO", "NOT ALMOST EQUAL TO", "NOT ALMOST EQUAL TO", 
"NOT ALMOST EQUAL TO", "ALMOST EQUAL OR EQUAL TO", "ALMOST EQUAL OR EQUAL TO", 
"TRIPLE TILDE", "TRIPLE TILDE with slash", "TRIPLE TILDE with slash", 
"ALL EQUAL TO", "ALL EQUAL TO", "EQUIVALENT TO", "EQUIVALENT TO", 
"EQUIVALENT TO with vertical line", "EQUIVALENT TO with vertical line", 
"GEOMETRICALLY EQUIVALENT TO", "GEOMETRICALLY EQUIVALENT TO", 
"GEOMETRICALLY EQUIVALENT TO", "GEOMETRICALLY EQUIVALENT TO with slash", 
"GEOMETRICALLY EQUIVALENT TO with slash", "GEOMETRICALLY EQUIVALENT TO with slash", 
"GEOMETRICALLY EQUIVALENT TO with slash", "DIFFERENCE BETWEEN", 
"DIFFERENCE BETWEEN", "DIFFERENCE BETWEEN", "DIFFERENCE BETWEEN with slash", 
"DIFFERENCE BETWEEN with slash", "DIFFERENCE BETWEEN with slash", 
"DIFFERENCE BETWEEN with slash", "APPROACHES THE LIMIT", "APPROACHES THE LIMIT", 
"APPROACHES THE LIMIT", "APPROACHES THE LIMIT with slash", "APPROACHES THE LIMIT with slash", 
"GEOMETRICALLY EQUAL TO", "GEOMETRICALLY EQUAL TO", "APPROXIMATELY EQUAL TO OR THE IMAGE OF", 
"APPROXIMATELY EQUAL TO OR THE IMAGE OF", "IMAGE OF OR APPROXIMATELY EQUAL TO", 
"IMAGE OF OR APPROXIMATELY EQUAL TO", "COLON EQUALS", "COLON EQUALS", 
"COLON EQUALS", "EQUALS COLON", "EQUALS COLON", "RING IN EQUAL TO", 
"RING IN EQUAL TO", "RING EQUAL TO", "RING EQUAL TO", "ESTIMATES", 
"EQUIANGULAR TO", "DELTA EQUAL TO", "DELTA EQUAL TO", "QUESTIONED EQUAL TO", 
"QUESTIONED EQUAL TO", "NOT EQUAL TO", "NOT EQUAL TO", "IDENTICAL TO", 
"IDENTICAL TO", "IDENTICAL TO with reverse slash", "IDENTICAL TO with reverse slash", 
"NOT IDENTICAL TO", "NOT IDENTICAL TO", "LESS-THAN OR EQUAL TO", 
"LESS-THAN OR EQUAL TO", "LESS-THAN OR EQUAL TO with vertical line", 
"LESS-THAN OR EQUAL TO with vertical line", "GREATER-THAN OR EQUAL TO", 
"GREATER-THAN OR EQUAL TO", "GREATER-THAN OR EQUAL TO", "GREATER-THAN OR EQUAL TO with vertical line", 
"GREATER-THAN OR EQUAL TO with vertical line", "LESS-THAN OVER EQUAL TO", 
"LESS-THAN OVER EQUAL TO", "LESS-THAN OVER EQUAL TO", "LESS-THAN OVER EQUAL TO with slash", 
"LESS-THAN OVER EQUAL TO with slash", "LESS-THAN OVER EQUAL TO with slash", 
"LESS-THAN OVER EQUAL TO with slash", "GREATER-THAN OVER EQUAL TO", 
"GREATER-THAN OVER EQUAL TO", "GREATER-THAN OVER EQUAL TO", "GREATER-THAN OVER EQUAL TO with slash", 
"GREATER-THAN OVER EQUAL TO with slash", "GREATER-THAN OVER EQUAL TO with slash", 
"GREATER-THAN OVER EQUAL TO with slash", "GREATER-THAN OVER EQUAL TO with slash", 
"GREATER-THAN OVER EQUAL TO with slash", "LESS-THAN BUT NOT EQUAL TO", 
"LESS-THAN BUT NOT EQUAL TO", "LESS-THAN BUT NOT EQUAL TO - with vertical stroke", 
"LESS-THAN BUT NOT EQUAL TO - with vertical stroke", "LESS-THAN BUT NOT EQUAL TO - with vertical stroke", 
"LESS-THAN BUT NOT EQUAL TO - with vertical stroke", "GREATER-THAN BUT NOT EQUAL TO", 
"GREATER-THAN BUT NOT EQUAL TO", "GREATER-THAN BUT NOT EQUAL TO - with vertical stroke", 
"GREATER-THAN BUT NOT EQUAL TO - with vertical stroke", "GREATER-THAN BUT NOT EQUAL TO - with vertical stroke", 
"GREATER-THAN BUT NOT EQUAL TO - with vertical stroke", "MUCH LESS-THAN", 
"MUCH LESS-THAN", "MUCH LESS-THAN", "MUCH LESS THAN with slash", 
"MUCH LESS THAN with slash", "MUCH LESS THAN with slash", "MUCH LESS THAN with slash", 
"MUCH LESS THAN with vertical line", "MUCH LESS THAN with vertical line", 
"MUCH GREATER-THAN", "MUCH GREATER-THAN", "MUCH GREATER-THAN", 
"MUCH GREATER THAN with slash", "MUCH GREATER THAN with slash", 
"MUCH GREATER THAN with slash", "MUCH GREATER THAN with slash", 
"MUCH GREATER THAN with vertical line", "MUCH GREATER THAN with vertical line", 
"BETWEEN", "BETWEEN", "NOT EQUIVALENT TO", "NOT LESS-THAN", "NOT LESS-THAN", 
"NOT LESS-THAN", "NOT GREATER-THAN", "NOT GREATER-THAN", "NOT GREATER-THAN", 
"NEITHER LESS-THAN NOR EQUAL TO", "NEITHER LESS-THAN NOR EQUAL TO", 
"NEITHER LESS-THAN NOR EQUAL TO", "NEITHER GREATER-THAN NOR EQUAL TO", 
"NEITHER GREATER-THAN NOR EQUAL TO", "NEITHER GREATER-THAN NOR EQUAL TO", 
"LESS-THAN OR EQUIVALENT TO", "LESS-THAN OR EQUIVALENT TO", "LESS-THAN OR EQUIVALENT TO", 
"GREATER-THAN OR EQUIVALENT TO", "GREATER-THAN OR EQUIVALENT TO", 
"GREATER-THAN OR EQUIVALENT TO", "NEITHER LESS-THAN NOR EQUIVALENT TO", 
"NEITHER LESS-THAN NOR EQUIVALENT TO", "NEITHER GREATER-THAN NOR EQUIVALENT TO", 
"NEITHER GREATER-THAN NOR EQUIVALENT TO", "LESS-THAN OR GREATER-THAN", 
"LESS-THAN OR GREATER-THAN", "LESS-THAN OR GREATER-THAN", "GREATER-THAN OR LESS-THAN", 
"GREATER-THAN OR LESS-THAN", "GREATER-THAN OR LESS-THAN", "NEITHER LESS-THAN NOR GREATER-THAN", 
"NEITHER LESS-THAN NOR GREATER-THAN", "NEITHER GREATER-THAN NOR LESS-THAN", 
"NEITHER GREATER-THAN NOR LESS-THAN", "PRECEDES", "PRECEDES", 
"PRECEDES", "SUCCEEDS", "SUCCEEDS", "SUCCEEDS", "PRECEDES OR EQUAL TO", 
"PRECEDES OR EQUAL TO", "PRECEDES OR EQUAL TO", "SUCCEEDS OR EQUAL TO", 
"SUCCEEDS OR EQUAL TO", "SUCCEEDS OR EQUAL TO", "PRECEDES OR EQUIVALENT TO", 
"PRECEDES OR EQUIVALENT TO", "PRECEDES OR EQUIVALENT TO", "SUCCEEDS OR EQUIVALENT TO", 
"SUCCEEDS OR EQUIVALENT TO", "SUCCEEDS OR EQUIVALENT TO", "SUCCEEDS OR EQUIVALENT TO with slash", 
"SUCCEEDS OR EQUIVALENT TO with slash", "DOES NOT PRECEDE", "DOES NOT PRECEDE", 
"DOES NOT PRECEDE", "DOES NOT SUCCEED", "DOES NOT SUCCEED", "DOES NOT SUCCEED", 
"SUBSET OF", "SUBSET OF", "SUBSET OF with vertical line", "SUBSET OF with vertical line", 
"SUBSET OF with vertical line", "SUBSET OF with vertical line", 
"SUBSET OF with vertical line", "SUBSET OF with vertical line", 
"SUPERSET OF", "SUPERSET OF", "SUPERSET OF", "SUPERSET OF with vertical line", 
"SUPERSET OF with vertical line", "SUPERSET OF with vertical line", 
"SUPERSET OF with vertical line", "SUPERSET OF with vertical line", 
"SUPERSET OF with vertical line", "NOT A SUBSET OF", "NOT A SUPERSET OF", 
"SUBSET OF OR EQUAL TO", "SUBSET OF OR EQUAL TO", "SUBSET OF OR EQUAL TO", 
"SUPERSET OF OR EQUAL TO", "SUPERSET OF OR EQUAL TO", "SUPERSET OF OR EQUAL TO", 
"NEITHER A SUBSET OF NOR EQUAL TO", "NEITHER A SUBSET OF NOR EQUAL TO", 
"NEITHER A SUBSET OF NOR EQUAL TO", "NEITHER A SUPERSET OF NOR EQUAL TO", 
"NEITHER A SUPERSET OF NOR EQUAL TO", "NEITHER A SUPERSET OF NOR EQUAL TO", 
"SUBSET OF WITH NOT EQUAL TO", "SUBSET OF WITH NOT EQUAL TO", 
"SUBSET OF WITH NOT EQUAL TO - variant with stroke through bottom members", 
"SUBSET OF WITH NOT EQUAL TO - variant with stroke through bottom members", 
"SUBSET OF WITH NOT EQUAL TO - variant with stroke through bottom members", 
"SUBSET OF WITH NOT EQUAL TO - variant with stroke through bottom members", 
"SUPERSET OF WITH NOT EQUAL TO", "SUPERSET OF WITH NOT EQUAL TO", 
"SUPERSET OF WITH NOT EQUAL TO - variant with stroke through bottom members", 
"SUPERSET OF WITH NOT EQUAL TO - variant with stroke through bottom members", 
"SUPERSET OF WITH NOT EQUAL TO - variant with stroke through bottom members", 
"SUPERSET OF WITH NOT EQUAL TO - variant with stroke through bottom members", 
"MULTISET MULTIPLICATION", "MULTISET UNION", "MULTISET UNION", 
"SQUARE IMAGE OF", "SQUARE IMAGE OF", "SQUARE IMAGE OF", "SQUARE IMAGE OF with slash", 
"SQUARE IMAGE OF with slash", "SQUARE ORIGINAL OF", "SQUARE ORIGINAL OF", 
"SQUARE ORIGINAL OF", "SQUARE ORIGINAL OF with slash", "SQUARE ORIGINAL OF with slash", 
"SQUARE IMAGE OF OR EQUAL TO", "SQUARE IMAGE OF OR EQUAL TO", 
"SQUARE IMAGE OF OR EQUAL TO", "SQUARE ORIGINAL OF OR EQUAL TO", 
"SQUARE ORIGINAL OF OR EQUAL TO", "SQUARE ORIGINAL OF OR EQUAL TO", 
"SQUARE CAP", "SQUARE CAP", "SQUARE CAP with serifs", "SQUARE CAP with serifs", 
"SQUARE CUP", "SQUARE CUP", "SQUARE CUP with serifs", "SQUARE CUP with serifs", 
"CIRCLED PLUS", "CIRCLED PLUS", "CIRCLED MINUS", "CIRCLED MINUS", 
"CIRCLED TIMES", "CIRCLED TIMES", "CIRCLED DIVISION SLASH", "CIRCLED DOT OPERATOR", 
"CIRCLED DOT OPERATOR", "CIRCLED RING OPERATOR", "CIRCLED RING OPERATOR", 
"CIRCLED ASTERISK OPERATOR", "CIRCLED ASTERISK OPERATOR", "CIRCLED DASH", 
"CIRCLED DASH", "SQUARED PLUS", "SQUARED PLUS", "SQUARED MINUS", 
"SQUARED MINUS", "SQUARED TIMES", "SQUARED TIMES", "SQUARED DOT OPERATOR", 
"SQUARED DOT OPERATOR", "RIGHT TACK", "RIGHT TACK", "LEFT TACK", 
"LEFT TACK", "DOWN TACK", "DOWN TACK", "UP TACK", "UP TACK", 
"UP TACK", "UP TACK", "MODELS", "TRUE", "TRUE", "FORCES", "TRIPLE VERTICAL BAR RIGHT TURNSTILE", 
"DOUBLE VERTICAL BAR DOUBLE RIGHT TURNSTILE", "DOES NOT PROVE", 
"NOT TRUE", "DOES NOT FORCE", "NEGATED DOUBLE VERTICAL BAR DOUBLE RIGHT TURNSTILE", 
"PRECEDES UNDER RELATION", "NORMAL SUBGROUP OF", "NORMAL SUBGROUP OF", 
"NORMAL SUBGROUP OF", "CONTAINS AS NORMAL SUBGROUP", "CONTAINS AS NORMAL SUBGROUP", 
"CONTAINS AS NORMAL SUBGROUP", "NORMAL SUBGROUP OF OR EQUAL TO", 
"NORMAL SUBGROUP OF OR EQUAL TO", "NORMAL SUBGROUP OF OR EQUAL TO", 
"NORMAL SUBGROUP OF OR EQUAL TO with vertical line", "NORMAL SUBGROUP OF OR EQUAL TO with vertical line", 
"CONTAINS AS NORMAL SUBGROUP OR EQUAL TO", "CONTAINS AS NORMAL SUBGROUP OR EQUAL TO", 
"CONTAINS AS NORMAL SUBGROUP OR EQUAL TO", "CONTAINS AS NORMAL SUBGROUP OR EQUAL TO with vertical line", 
"CONTAINS AS NORMAL SUBGROUP OR EQUAL TO with vertical line", 
"ORIGINAL OF", "IMAGE OF", "MULTIMAP", "MULTIMAP", "HERMITIAN CONJUGATE MATRIX", 
"INTERCALATE", "INTERCALATE", "XOR", "NOR", "RIGHT ANGLE WITH ARC", 
"RIGHT TRIANGLE", "N-ARY LOGICAL AND", "N-ARY LOGICAL AND", "N-ARY LOGICAL AND", 
"N-ARY LOGICAL OR", "N-ARY LOGICAL OR", "N-ARY LOGICAL OR", "N-ARY INTERSECTION", 
"N-ARY INTERSECTION", "N-ARY INTERSECTION", "N-ARY UNION", "N-ARY UNION", 
"N-ARY UNION", "DIAMOND OPERATOR", "DIAMOND OPERATOR", "DIAMOND OPERATOR", 
"DOT OPERATOR", "STAR OPERATOR", "STAR OPERATOR", "DIVISION TIMES", 
"DIVISION TIMES", "BOWTIE", "LEFT NORMAL FACTOR SEMIDIRECT PRODUCT", 
"RIGHT NORMAL FACTOR SEMIDIRECT PRODUCT", "LEFT SEMIDIRECT PRODUCT", 
"LEFT SEMIDIRECT PRODUCT", "RIGHT SEMIDIRECT PRODUCT", "RIGHT SEMIDIRECT PRODUCT", 
"REVERSED TILDE EQUALS", "REVERSED TILDE EQUALS", "CURLY LOGICAL OR", 
"CURLY LOGICAL OR", "CURLY LOGICAL AND", "CURLY LOGICAL AND", 
"DOUBLE SUBSET", "DOUBLE SUBSET", "DOUBLE SUPERSET", "DOUBLE SUPERSET", 
"DOUBLE INTERSECTION", "DOUBLE UNION", "PITCHFORK", "PITCHFORK", 
"EQUAL AND PARALLEL TO", "LESS-THAN WITH DOT", "LESS-THAN WITH DOT", 
"GREATER-THAN WITH DOT", "GREATER-THAN WITH DOT", "VERY MUCH LESS-THAN", 
"VERY MUCH LESS-THAN with slash", "VERY MUCH LESS-THAN with slash", 
"VERY MUCH GREATER-THAN", "VERY MUCH GREATER-THAN", "VERY MUCH GREATER-THAN with slash", 
"VERY MUCH GREATER-THAN with slash", "LESS-THAN EQUAL TO OR GREATER-THAN", 
"LESS-THAN EQUAL TO OR GREATER-THAN", "LESS-THAN EQUAL TO OR GREATER-THAN", 
"LESS-THAN slanted EQUAL TO OR GREATER-THAN", "LESS-THAN slanted EQUAL TO OR GREATER-THAN", 
"GREATER-THAN EQUAL TO OR LESS-THAN", "GREATER-THAN EQUAL TO OR LESS-THAN", 
"GREATER-THAN EQUAL TO OR LESS-THAN", "GREATER-THAN slanted EQUAL TO OR LESS-THAN", 
"GREATER-THAN slanted EQUAL TO OR LESS-THAN", "EQUAL TO OR PRECEDES", 
"EQUAL TO OR PRECEDES", "EQUAL TO OR SUCCEEDS", "EQUAL TO OR SUCCEEDS", 
"DOES NOT PRECEDE OR EQUAL", "DOES NOT PRECEDE OR EQUAL", "DOES NOT SUCCEED OR EQUAL", 
"DOES NOT SUCCEED OR EQUAL", "NOT SQUARE IMAGE OF OR EQUAL TO", 
"NOT SQUARE IMAGE OF OR EQUAL TO", "NOT SQUARE ORIGINAL OF OR EQUAL TO", 
"NOT SQUARE ORIGINAL OF OR EQUAL TO", "LESS-THAN BUT NOT EQUIVALENT TO", 
"GREATER-THAN BUT NOT EQUIVALENT TO", "PRECEDES BUT NOT EQUIVALENT TO", 
"PRECEDES BUT NOT EQUIVALENT TO", "SUCCEEDS BUT NOT EQUIVALENT TO", 
"SUCCEEDS BUT NOT EQUIVALENT TO", "NOT NORMAL SUBGROUP OF", "NOT NORMAL SUBGROUP OF", 
"NOT NORMAL SUBGROUP OF", "DOES NOT CONTAIN AS NORMAL SUBGROUP", 
"DOES NOT CONTAIN AS NORMAL SUBGROUP", "DOES NOT CONTAIN AS NORMAL SUBGROUP", 
"NOT NORMAL SUBGROUP OF OR EQUAL TO", "NOT NORMAL SUBGROUP OF OR EQUAL TO", 
"NOT NORMAL SUBGROUP OF OR EQUAL TO", "DOES NOT CONTAIN AS NORMAL SUBGROUP OR EQUAL", 
"DOES NOT CONTAIN AS NORMAL SUBGROUP OR EQUAL", "DOES NOT CONTAIN AS NORMAL SUBGROUP OR EQUAL", 
"VERTICAL ELLIPSIS", "MIDLINE HORIZONTAL ELLIPSIS", "UP RIGHT DIAGONAL ELLIPSIS", 
"DOWN RIGHT DIAGONAL ELLIPSIS", "ELEMENT OF WITH LONG HORIZONTAL STROKE", 
"ELEMENT OF WITH VERTICAL BAR AT END OF HORIZONTAL STROKE", "SMALL ELEMENT OF WITH VERTICAL BAR AT END OF HORIZONTAL STROKE", 
"ELEMENT OF WITH DOT ABOVE", "ELEMENT OF WITH DOT ABOVE with slash", 
"ELEMENT OF WITH DOT ABOVE with slash", "ELEMENT OF WITH OVERBAR", 
"SMALL ELEMENT OF WITH OVERBAR", "ELEMENT OF WITH TWO HORIZONTAL STROKES", 
"ELEMENT OF WITH TWO HORIZONTAL STROKES with slash", "ELEMENT OF WITH TWO HORIZONTAL STROKES with slash", 
"CONTAINS WITH LONG HORIZONTAL STROKE", "CONTAINS WITH VERTICAL BAR AT END OF HORIZONTAL STROKE", 
"SMALL CONTAINS WITH VERTICAL BAR AT END OF HORIZONTAL STROKE", 
"CONTAINS WITH OVERBAR", "SMALL CONTAINS WITH OVERBAR", "PROJECTIVE", 
"PROJECTIVE", "PERSPECTIVE", "PERSPECTIVE", "LEFT CEILING", "LEFT CEILING", 
"RIGHT CEILING", "RIGHT CEILING", "LEFT FLOOR", "LEFT FLOOR", 
"RIGHT FLOOR", "RIGHT FLOOR", "BOTTOM RIGHT CROP", "BOTTOM LEFT CROP", 
"TOP RIGHT CROP", "TOP LEFT CROP", "REVERSED NOT SIGN", "ARC", 
"SEGMENT", "TELEPHONE RECORDER", "POSITION INDICATOR", "TOP LEFT CORNER", 
"TOP LEFT CORNER", "TOP RIGHT CORNER", "TOP RIGHT CORNER", "BOTTOM LEFT CORNER", 
"BOTTOM LEFT CORNER", "BOTTOM RIGHT CORNER", "BOTTOM RIGHT CORNER", 
"FROWN", "FROWN", "SMILE", "SMILE", "CYLINDRICITY", "ALL AROUND-PROFILE", 
"APL FUNCTIONAL SYMBOL I-BEAM", "APL FUNCTIONAL SYMBOL CIRCLE STILE", 
"APL FUNCTIONAL SYMBOL SLASH BAR", "RIGHT ANGLE WITH DOWNWARDS ZIGZAG ARROW", 
"UPPER LEFT OR LOWER RIGHT CURLY BRACKET SECTION", "UPPER LEFT OR LOWER RIGHT CURLY BRACKET SECTION", 
"UPPER RIGHT OR LOWER LEFT CURLY BRACKET SECTION", "UPPER RIGHT OR LOWER LEFT CURLY BRACKET SECTION", 
"TOP SQUARE BRACKET", "TOP SQUARE BRACKET", "BOTTOM SQUARE BRACKET", 
"BOTTOM SQUARE BRACKET", "BOTTOM SQUARE BRACKET OVER TOP SQUARE BRACKET", 
"TOP PARENTHESIS", "BOTTOM PARENTHESIS", "TOP CURLY BRACKET", 
"BOTTOM CURLY BRACKET", "WHITE TRAPEZIUM", "ELECTRICAL INTERSECTION", 
"OPEN BOX", "CIRCLED LATIN CAPITAL LETTER S", "CIRCLED LATIN CAPITAL LETTER S", 
"BOX DRAWINGS LIGHT HORIZONTAL", "BOX DRAWINGS LIGHT HORIZONTAL", 
"BOX DRAWINGS LIGHT VERTICAL", "BOX DRAWINGS LIGHT DOWN AND RIGHT", 
"BOX DRAWINGS LIGHT DOWN AND LEFT", "BOX DRAWINGS LIGHT UP AND RIGHT", 
"BOX DRAWINGS LIGHT UP AND LEFT", "BOX DRAWINGS LIGHT VERTICAL AND RIGHT", 
"BOX DRAWINGS LIGHT VERTICAL AND LEFT", "BOX DRAWINGS LIGHT DOWN AND HORIZONTAL", 
"BOX DRAWINGS LIGHT UP AND HORIZONTAL", "BOX DRAWINGS LIGHT VERTICAL AND HORIZONTAL", 
"BOX DRAWINGS DOUBLE HORIZONTAL", "BOX DRAWINGS DOUBLE VERTICAL", 
"BOX DRAWINGS DOWN SINGLE AND RIGHT DOUBLE", "BOX DRAWINGS DOWN DOUBLE AND RIGHT SINGLE", 
"BOX DRAWINGS DOUBLE DOWN AND RIGHT", "BOX DRAWINGS DOWN SINGLE AND LEFT DOUBLE", 
"BOX DRAWINGS DOWN DOUBLE AND LEFT SINGLE", "BOX DRAWINGS DOUBLE DOWN AND LEFT", 
"BOX DRAWINGS UP SINGLE AND RIGHT DOUBLE", "BOX DRAWINGS UP DOUBLE AND RIGHT SINGLE", 
"BOX DRAWINGS DOUBLE UP AND RIGHT", "BOX DRAWINGS UP SINGLE AND LEFT DOUBLE", 
"BOX DRAWINGS UP DOUBLE AND LEFT SINGLE", "BOX DRAWINGS DOUBLE UP AND LEFT", 
"BOX DRAWINGS VERTICAL SINGLE AND RIGHT DOUBLE", "BOX DRAWINGS VERTICAL DOUBLE AND RIGHT SINGLE", 
"BOX DRAWINGS DOUBLE VERTICAL AND RIGHT", "BOX DRAWINGS VERTICAL SINGLE AND LEFT DOUBLE", 
"BOX DRAWINGS VERTICAL DOUBLE AND LEFT SINGLE", "BOX DRAWINGS DOUBLE VERTICAL AND LEFT", 
"BOX DRAWINGS DOWN SINGLE AND HORIZONTAL DOUBLE", "BOX DRAWINGS DOWN DOUBLE AND HORIZONTAL SINGLE", 
"BOX DRAWINGS DOUBLE DOWN AND HORIZONTAL", "BOX DRAWINGS UP SINGLE AND HORIZONTAL DOUBLE", 
"BOX DRAWINGS UP DOUBLE AND HORIZONTAL SINGLE", "BOX DRAWINGS DOUBLE UP AND HORIZONTAL", 
"BOX DRAWINGS VERTICAL SINGLE AND HORIZONTAL DOUBLE", "BOX DRAWINGS VERTICAL DOUBLE AND HORIZONTAL SINGLE", 
"BOX DRAWINGS DOUBLE VERTICAL AND HORIZONTAL", "UPPER HALF BLOCK", 
"LOWER HALF BLOCK", "FULL BLOCK", "LIGHT SHADE", "MEDIUM SHADE", 
"DARK SHADE", "WHITE SQUARE", "WHITE SQUARE", "WHITE SQUARE", 
"BLACK SMALL SQUARE", "BLACK SMALL SQUARE", "BLACK SMALL SQUARE", 
"BLACK SMALL SQUARE", "WHITE SMALL SQUARE", "WHITE RECTANGLE", 
"BLACK VERTICAL RECTANGLE", "WHITE PARALLELOGRAM", "WHITE UP-POINTING TRIANGLE", 
"WHITE UP-POINTING TRIANGLE", "BLACK UP-POINTING SMALL TRIANGLE", 
"BLACK UP-POINTING SMALL TRIANGLE", "WHITE UP-POINTING SMALL TRIANGLE", 
"WHITE UP-POINTING SMALL TRIANGLE", "BLACK RIGHT-POINTING SMALL TRIANGLE", 
"BLACK RIGHT-POINTING SMALL TRIANGLE", "WHITE RIGHT-POINTING SMALL TRIANGLE", 
"WHITE RIGHT-POINTING SMALL TRIANGLE", "WHITE DOWN-POINTING TRIANGLE", 
"WHITE DOWN-POINTING TRIANGLE", "BLACK DOWN-POINTING SMALL TRIANGLE", 
"BLACK DOWN-POINTING SMALL TRIANGLE", "WHITE DOWN-POINTING SMALL TRIANGLE", 
"WHITE DOWN-POINTING SMALL TRIANGLE", "BLACK LEFT-POINTING SMALL TRIANGLE", 
"BLACK LEFT-POINTING SMALL TRIANGLE", "WHITE LEFT-POINTING SMALL TRIANGLE", 
"WHITE LEFT-POINTING SMALL TRIANGLE", "LOZENGE", "LOZENGE", "WHITE CIRCLE", 
"WHITE UP-POINTING TRIANGLE WITH DOT", "LARGE CIRCLE", "LARGE CIRCLE", 
"UPPER LEFT TRIANGLE", "UPPER RIGHT TRIANGLE", "LOWER LEFT TRIANGLE", 
"WHITE MEDIUM SQUARE", "BLACK MEDIUM SQUARE", "BLACK STAR", "BLACK STAR", 
"WHITE STAR", "BLACK TELEPHONE", "FEMALE SIGN", "MALE SIGN", 
"BLACK SPADE SUIT", "BLACK SPADE SUIT", "BLACK CLUB SUIT", "BLACK CLUB SUIT", 
"BLACK HEART SUIT", "BLACK HEART SUIT", "BLACK DIAMOND SUIT", 
"BLACK DIAMOND SUIT", "EIGHTH NOTE", "MUSIC FLAT SIGN", "MUSIC NATURAL SIGN", 
"MUSIC NATURAL SIGN", "MUSIC SHARP SIGN", "CHECK MARK", "CHECK MARK", 
"BALLOT X", "MALTESE CROSS", "MALTESE CROSS", "SIX POINTED BLACK STAR", 
"LIGHT VERTICAL BAR", "LIGHT LEFT TORTOISE SHELL BRACKET ORNAMENT", 
"LIGHT RIGHT TORTOISE SHELL BRACKET ORNAMENT", "REVERSE SOLIDUS PRECEDING SUBSET", 
"SUPERSET PRECEDING SOLIDUS", "MATHEMATICAL LEFT WHITE SQUARE BRACKET", 
"MATHEMATICAL LEFT WHITE SQUARE BRACKET", "MATHEMATICAL RIGHT WHITE SQUARE BRACKET", 
"MATHEMATICAL RIGHT WHITE SQUARE BRACKET", "MATHEMATICAL LEFT ANGLE BRACKET", 
"MATHEMATICAL LEFT ANGLE BRACKET", "MATHEMATICAL LEFT ANGLE BRACKET", 
"MATHEMATICAL RIGHT ANGLE BRACKET", "MATHEMATICAL RIGHT ANGLE BRACKET", 
"MATHEMATICAL RIGHT ANGLE BRACKET", "MATHEMATICAL LEFT DOUBLE ANGLE BRACKET", 
"MATHEMATICAL RIGHT DOUBLE ANGLE BRACKET", "MATHEMATICAL LEFT WHITE TORTOISE SHELL BRACKET", 
"MATHEMATICAL RIGHT WHITE TORTOISE SHELL BRACKET", "LONG LEFTWARDS ARROW", 
"LONG LEFTWARDS ARROW", "LONG LEFTWARDS ARROW", "LONG RIGHTWARDS ARROW", 
"LONG RIGHTWARDS ARROW", "LONG RIGHTWARDS ARROW", "LONG LEFT RIGHT ARROW", 
"LONG LEFT RIGHT ARROW", "LONG LEFT RIGHT ARROW", "LONG LEFTWARDS DOUBLE ARROW", 
"LONG LEFTWARDS DOUBLE ARROW", "LONG LEFTWARDS DOUBLE ARROW", 
"LONG RIGHTWARDS DOUBLE ARROW", "LONG RIGHTWARDS DOUBLE ARROW", 
"LONG RIGHTWARDS DOUBLE ARROW", "LONG LEFT RIGHT DOUBLE ARROW", 
"LONG LEFT RIGHT DOUBLE ARROW", "LONG LEFT RIGHT DOUBLE ARROW", 
"LONG RIGHTWARDS ARROW FROM BAR", "LONG RIGHTWARDS ARROW FROM BAR", 
"LONG RIGHTWARDS SQUIGGLE ARROW", "LEFTWARDS DOUBLE ARROW WITH VERTICAL STROKE", 
"RIGHTWARDS DOUBLE ARROW WITH VERTICAL STROKE", "LEFT RIGHT DOUBLE ARROW WITH VERTICAL STROKE", 
"RIGHTWARDS TWO-HEADED ARROW FROM BAR", "LEFTWARDS DOUBLE DASH ARROW", 
"RIGHTWARDS DOUBLE DASH ARROW", "RIGHTWARDS DOUBLE DASH ARROW", 
"LEFTWARDS TRIPLE DASH ARROW", "RIGHTWARDS TRIPLE DASH ARROW", 
"RIGHTWARDS TRIPLE DASH ARROW", "RIGHTWARDS TWO-HEADED TRIPLE DASH ARROW", 
"RIGHTWARDS TWO-HEADED TRIPLE DASH ARROW", "RIGHTWARDS ARROW WITH DOTTED STEM", 
"UPWARDS ARROW TO BAR", "DOWNWARDS ARROW TO BAR", "RIGHTWARDS TWO-HEADED ARROW WITH TAIL", 
"LEFTWARDS ARROW-TAIL", "RIGHTWARDS ARROW-TAIL", "LEFTWARDS DOUBLE ARROW-TAIL", 
"RIGHTWARDS DOUBLE ARROW-TAIL", "LEFTWARDS ARROW TO BLACK DIAMOND", 
"RIGHTWARDS ARROW TO BLACK DIAMOND", "LEFTWARDS ARROW FROM BAR TO BLACK DIAMOND", 
"RIGHTWARDS ARROW FROM BAR TO BLACK DIAMOND", "NORTH WEST ARROW WITH HOOK", 
"NORTH EAST ARROW WITH HOOK", "SOUTH EAST ARROW WITH HOOK", "SOUTH EAST ARROW WITH HOOK", 
"SOUTH WEST ARROW WITH HOOK", "SOUTH WEST ARROW WITH HOOK", "NORTH WEST ARROW AND NORTH EAST ARROW", 
"NORTH EAST ARROW AND SOUTH EAST ARROW", "NORTH EAST ARROW AND SOUTH EAST ARROW", 
"SOUTH EAST ARROW AND SOUTH WEST ARROW", "SOUTH EAST ARROW AND SOUTH WEST ARROW", 
"SOUTH WEST ARROW AND NORTH WEST ARROW", "WAVE ARROW POINTING DIRECTLY RIGHT", 
"WAVE ARROW POINTING DIRECTLY RIGHT with slash", "WAVE ARROW POINTING DIRECTLY RIGHT with slash", 
"ARROW POINTING RIGHTWARDS THEN CURVING DOWNWARDS", "ARROW POINTING DOWNWARDS THEN CURVING LEFTWARDS", 
"ARROW POINTING DOWNWARDS THEN CURVING RIGHTWARDS", "RIGHT-SIDE ARC CLOCKWISE ARROW", 
"LEFT-SIDE ARC ANTICLOCKWISE ARROW", "TOP ARC CLOCKWISE ARROW WITH MINUS", 
"TOP ARC ANTICLOCKWISE ARROW WITH PLUS", "RIGHTWARDS ARROW WITH PLUS BELOW", 
"LEFT RIGHT ARROW THROUGH SMALL CIRCLE", "UPWARDS TWO-HEADED ARROW FROM SMALL CIRCLE", 
"LEFT BARB UP RIGHT BARB DOWN HARPOON", "LEFT BARB DOWN RIGHT BARB UP HARPOON", 
"LEFT BARB UP RIGHT BARB UP HARPOON", "UP BARB RIGHT DOWN BARB RIGHT HARPOON", 
"LEFT BARB DOWN RIGHT BARB DOWN HARPOON", "UP BARB LEFT DOWN BARB LEFT HARPOON", 
"LEFTWARDS HARPOON WITH BARB UP TO BAR", "RIGHTWARDS HARPOON WITH BARB UP TO BAR", 
"UPWARDS HARPOON WITH BARB RIGHT TO BAR", "DOWNWARDS HARPOON WITH BARB RIGHT TO BAR", 
"LEFTWARDS HARPOON WITH BARB DOWN TO BAR", "RIGHTWARDS HARPOON WITH BARB DOWN TO BAR", 
"UPWARDS HARPOON WITH BARB LEFT TO BAR", "DOWNWARDS HARPOON WITH BARB LEFT TO BAR", 
"LEFTWARDS HARPOON WITH BARB UP FROM BAR", "RIGHTWARDS HARPOON WITH BARB UP FROM BAR", 
"UPWARDS HARPOON WITH BARB RIGHT FROM BAR", "DOWNWARDS HARPOON WITH BARB RIGHT FROM BAR", 
"LEFTWARDS HARPOON WITH BARB DOWN FROM BAR", "RIGHTWARDS HARPOON WITH BARB DOWN FROM BAR", 
"UPWARDS HARPOON WITH BARB LEFT FROM BAR", "DOWNWARDS HARPOON WITH BARB LEFT FROM BAR", 
"LEFTWARDS HARPOON WITH BARB UP ABOVE LEFTWARDS HARPOON WITH BARB DOWN", 
"UPWARDS HARPOON WITH BARB LEFT BESIDE UPWARDS HARPOON WITH BARB RIGHT", 
"RIGHTWARDS HARPOON WITH BARB UP ABOVE RIGHTWARDS HARPOON WITH BARB DOWN", 
"DOWNWARDS HARPOON WITH BARB LEFT BESIDE DOWNWARDS HARPOON WITH BARB RIGHT", 
"LEFTWARDS HARPOON WITH BARB UP ABOVE RIGHTWARDS HARPOON WITH BARB UP", 
"LEFTWARDS HARPOON WITH BARB DOWN ABOVE RIGHTWARDS HARPOON WITH BARB DOWN", 
"RIGHTWARDS HARPOON WITH BARB UP ABOVE LEFTWARDS HARPOON WITH BARB UP", 
"RIGHTWARDS HARPOON WITH BARB DOWN ABOVE LEFTWARDS HARPOON WITH BARB DOWN", 
"LEFTWARDS HARPOON WITH BARB UP ABOVE LONG DASH", "LEFTWARDS HARPOON WITH BARB DOWN BELOW LONG DASH", 
"RIGHTWARDS HARPOON WITH BARB UP ABOVE LONG DASH", "RIGHTWARDS HARPOON WITH BARB DOWN BELOW LONG DASH", 
"UPWARDS HARPOON WITH BARB LEFT BESIDE DOWNWARDS HARPOON WITH BARB RIGHT", 
"UPWARDS HARPOON WITH BARB LEFT BESIDE DOWNWARDS HARPOON WITH BARB RIGHT", 
"DOWNWARDS HARPOON WITH BARB LEFT BESIDE UPWARDS HARPOON WITH BARB RIGHT", 
"DOWNWARDS HARPOON WITH BARB LEFT BESIDE UPWARDS HARPOON WITH BARB RIGHT", 
"RIGHT DOUBLE ARROW WITH ROUNDED HEAD", "EQUALS SIGN ABOVE RIGHTWARDS ARROW", 
"TILDE OPERATOR ABOVE RIGHTWARDS ARROW", "LEFTWARDS ARROW ABOVE TILDE OPERATOR", 
"RIGHTWARDS ARROW ABOVE TILDE OPERATOR", "RIGHTWARDS ARROW ABOVE ALMOST EQUAL TO", 
"LESS-THAN ABOVE LEFTWARDS ARROW", "GREATER-THAN ABOVE RIGHTWARDS ARROW", 
"SUBSET ABOVE RIGHTWARDS ARROW", "SUPERSET ABOVE LEFTWARDS ARROW", 
"LEFT FISH TAIL", "RIGHT FISH TAIL", "UP FISH TAIL", "DOWN FISH TAIL", 
"LEFT WHITE PARENTHESIS", "RIGHT WHITE PARENTHESIS", "LEFT SQUARE BRACKET WITH UNDERBAR", 
"RIGHT SQUARE BRACKET WITH UNDERBAR", "LEFT SQUARE BRACKET WITH TICK IN TOP CORNER", 
"RIGHT SQUARE BRACKET WITH TICK IN BOTTOM CORNER", "LEFT SQUARE BRACKET WITH TICK IN BOTTOM CORNER", 
"RIGHT SQUARE BRACKET WITH TICK IN TOP CORNER", "LEFT ANGLE BRACKET WITH DOT", 
"RIGHT ANGLE BRACKET WITH DOT", "LEFT ARC LESS-THAN BRACKET", 
"RIGHT ARC GREATER-THAN BRACKET", "DOUBLE LEFT ARC GREATER-THAN BRACKET", 
"DOUBLE RIGHT ARC LESS-THAN BRACKET", "VERTICAL ZIGZAG LINE", 
"RIGHT ANGLE VARIANT WITH SQUARE", "MEASURED RIGHT ANGLE WITH DOT", 
"ANGLE WITH UNDERBAR", "REVERSED ANGLE WITH UNDERBAR", "OBLIQUE ANGLE OPENING UP", 
"OBLIQUE ANGLE OPENING DOWN", "MEASURED ANGLE WITH OPEN ARM ENDING IN ARROW POINTING UP AND RIGHT", 
"MEASURED ANGLE WITH OPEN ARM ENDING IN ARROW POINTING UP AND LEFT", 
"MEASURED ANGLE WITH OPEN ARM ENDING IN ARROW POINTING DOWN AND RIGHT", 
"MEASURED ANGLE WITH OPEN ARM ENDING IN ARROW POINTING DOWN AND LEFT", 
"MEASURED ANGLE WITH OPEN ARM ENDING IN ARROW POINTING RIGHT AND UP", 
"MEASURED ANGLE WITH OPEN ARM ENDING IN ARROW POINTING LEFT AND UP", 
"MEASURED ANGLE WITH OPEN ARM ENDING IN ARROW POINTING RIGHT AND DOWN", 
"MEASURED ANGLE WITH OPEN ARM ENDING IN ARROW POINTING LEFT AND DOWN", 
"REVERSED EMPTY SET", "EMPTY SET WITH OVERBAR", "EMPTY SET WITH SMALL CIRCLE ABOVE", 
"EMPTY SET WITH RIGHT ARROW ABOVE", "EMPTY SET WITH LEFT ARROW ABOVE", 
"CIRCLE WITH HORIZONTAL BAR", "CIRCLED VERTICAL BAR", "CIRCLED PARALLEL", 
"CIRCLED PERPENDICULAR", "CIRCLE WITH SUPERIMPOSED X", "CIRCLED ANTICLOCKWISE-ROTATED DIVISION SIGN", 
"CIRCLED WHITE BULLET", "CIRCLED BULLET", "CIRCLED LESS-THAN", 
"CIRCLED GREATER-THAN", "CIRCLE WITH SMALL CIRCLE TO THE RIGHT", 
"CIRCLE WITH TWO HORIZONTAL STROKES TO THE RIGHT", "SQUARED RISING DIAGONAL SLASH", 
"SQUARED FALLING DIAGONAL SLASH", "TWO JOINED SQUARES", "TRIANGLE WITH SERIFS AT BOTTOM", 
"RIGHT TRIANGLE ABOVE LEFT TRIANGLE", "LEFT TRIANGLE BESIDE VERTICAL BAR", 
"LEFT TRIANGLE BESIDE VERTICAL BAR with slash", "LEFT TRIANGLE BESIDE VERTICAL BAR with slash", 
"VERTICAL BAR BESIDE RIGHT TRIANGLE", "VERTICAL BAR BESIDE RIGHT TRIANGLE with slash", 
"VERTICAL BAR BESIDE RIGHT TRIANGLE with slash", "INCOMPLETE INFINITY", 
"TIE OVER INFINITY", "INFINITY NEGATED WITH VERTICAL BAR", "EQUALS SIGN AND SLANTED PARALLEL", 
"EQUALS SIGN AND SLANTED PARALLEL WITH TILDE ABOVE", "IDENTICAL TO AND SLANTED PARALLEL", 
"BLACK LOZENGE", "BLACK LOZENGE", "RULE-DELAYED", "SOLIDUS WITH OVERBAR", 
"N-ARY CIRCLED DOT OPERATOR", "N-ARY CIRCLED DOT OPERATOR", "N-ARY CIRCLED PLUS OPERATOR", 
"N-ARY CIRCLED PLUS OPERATOR", "N-ARY CIRCLED TIMES OPERATOR", 
"N-ARY CIRCLED TIMES OPERATOR", "N-ARY UNION OPERATOR WITH PLUS", 
"N-ARY UNION OPERATOR WITH PLUS", "N-ARY SQUARE UNION OPERATOR", 
"N-ARY SQUARE UNION OPERATOR", "QUADRUPLE INTEGRAL OPERATOR", 
"QUADRUPLE INTEGRAL OPERATOR", "FINITE PART INTEGRAL", "CIRCULATION FUNCTION", 
"ANTICLOCKWISE INTEGRATION", "LINE INTEGRATION WITH RECTANGULAR PATH AROUND POLE", 
"LINE INTEGRATION WITH SEMICIRCULAR PATH AROUND POLE", "LINE INTEGRATION NOT INCLUDING THE POLE", 
"INTEGRAL AROUND A POINT OPERATOR", "QUATERNION INTEGRAL OPERATOR", 
"INTEGRAL WITH LEFTWARDS ARROW WITH HOOK", "PLUS SIGN WITH SMALL CIRCLE ABOVE", 
"PLUS SIGN WITH CIRCUMFLEX ACCENT ABOVE", "PLUS SIGN WITH TILDE ABOVE", 
"PLUS SIGN WITH DOT BELOW", "PLUS SIGN WITH TILDE BELOW", "PLUS SIGN WITH SUBSCRIPT TWO", 
"MINUS SIGN WITH COMMA ABOVE", "MINUS SIGN WITH DOT BELOW", "PLUS SIGN IN LEFT HALF CIRCLE", 
"PLUS SIGN IN RIGHT HALF CIRCLE", "VECTOR OR CROSS PRODUCT", 
"MULTIPLICATION SIGN WITH DOT ABOVE", "MULTIPLICATION SIGN WITH UNDERBAR", 
"SMASH PRODUCT", "MULTIPLICATION SIGN IN LEFT HALF CIRCLE", "MULTIPLICATION SIGN IN RIGHT HALF CIRCLE", 
"CIRCLED MULTIPLICATION SIGN WITH CIRCUMFLEX ACCENT", "MULTIPLICATION SIGN IN DOUBLE CIRCLE", 
"CIRCLED DIVISION SIGN", "PLUS SIGN IN TRIANGLE", "MINUS SIGN IN TRIANGLE", 
"MULTIPLICATION SIGN IN TRIANGLE", "INTERIOR PRODUCT", "INTERIOR PRODUCT", 
"AMALGAMATION OR COPRODUCT", "INTERSECTION WITH DOT", "UNION WITH OVERBAR", 
"INTERSECTION WITH OVERBAR", "INTERSECTION WITH LOGICAL AND", 
"UNION WITH LOGICAL OR", "UNION ABOVE INTERSECTION", "INTERSECTION ABOVE UNION", 
"UNION ABOVE BAR ABOVE INTERSECTION", "INTERSECTION ABOVE BAR ABOVE UNION", 
"UNION BESIDE AND JOINED WITH UNION", "INTERSECTION BESIDE AND JOINED WITH INTERSECTION", 
"CLOSED UNION WITH SERIFS", "CLOSED INTERSECTION WITH SERIFS", 
"CLOSED UNION WITH SERIFS AND SMASH PRODUCT", "DOUBLE LOGICAL AND", 
"DOUBLE LOGICAL OR", "TWO INTERSECTING LOGICAL AND", "TWO INTERSECTING LOGICAL OR", 
"SLOPING LARGE OR", "SLOPING LARGE AND", "LOGICAL AND WITH MIDDLE STEM", 
"LOGICAL OR WITH MIDDLE STEM", "LOGICAL AND WITH HORIZONTAL DASH", 
"LOGICAL OR WITH HORIZONTAL DASH", "LOGICAL AND WITH UNDERBAR", 
"EQUALS SIGN WITH DOT BELOW", "TILDE OPERATOR WITH DOT ABOVE", 
"CONGRUENT WITH DOT ABOVE", "CONGRUENT WITH DOT ABOVE with slash", 
"CONGRUENT WITH DOT ABOVE with slash", "EQUALS WITH ASTERISK", 
"ALMOST EQUAL TO WITH CIRCUMFLEX ACCENT", "APPROXIMATELY EQUAL OR EQUAL TO", 
"APPROXIMATELY EQUAL OR EQUAL TO with slash", "APPROXIMATELY EQUAL OR EQUAL TO with slash", 
"EQUALS SIGN ABOVE PLUS SIGN", "PLUS SIGN ABOVE EQUALS SIGN", 
"EQUALS SIGN ABOVE TILDE OPERATOR", "DOUBLE COLON EQUAL", "TWO CONSECUTIVE EQUALS SIGNS", 
"EQUALS SIGN WITH TWO DOTS ABOVE AND TWO DOTS BELOW", "EQUALS SIGN WITH TWO DOTS ABOVE AND TWO DOTS BELOW", 
"EQUIVALENT WITH FOUR DOTS ABOVE", "LESS-THAN WITH CIRCLE INSIDE", 
"GREATER-THAN WITH CIRCLE INSIDE", "LESS-THAN WITH QUESTION MARK ABOVE", 
"GREATER-THAN WITH QUESTION MARK ABOVE", "LESS-THAN OR SLANTED EQUAL TO", 
"LESS-THAN OR SLANTED EQUAL TO", "LESS-THAN OR SLANTED EQUAL TO", 
"LESS-THAN OR SLANTED EQUAL TO with slash", "LESS-THAN OR SLANTED EQUAL TO with slash", 
"LESS-THAN OR SLANTED EQUAL TO with slash", "LESS-THAN OR SLANTED EQUAL TO with slash", 
"LESS-THAN OR SLANTED EQUAL TO with slash", "LESS-THAN OR SLANTED EQUAL TO with slash", 
"GREATER-THAN OR SLANTED EQUAL TO", "GREATER-THAN OR SLANTED EQUAL TO", 
"GREATER-THAN OR SLANTED EQUAL TO", "GREATER-THAN OR SLANTED EQUAL TO with slash", 
"GREATER-THAN OR SLANTED EQUAL TO with slash", "GREATER-THAN OR SLANTED EQUAL TO with slash", 
"GREATER-THAN OR SLANTED EQUAL TO with slash", "GREATER-THAN OR SLANTED EQUAL TO with slash", 
"GREATER-THAN OR SLANTED EQUAL TO with slash", "LESS-THAN OR SLANTED EQUAL TO WITH DOT INSIDE", 
"GREATER-THAN OR SLANTED EQUAL TO WITH DOT INSIDE", "LESS-THAN OR SLANTED EQUAL TO WITH DOT ABOVE", 
"GREATER-THAN OR SLANTED EQUAL TO WITH DOT ABOVE", "LESS-THAN OR SLANTED EQUAL TO WITH DOT ABOVE RIGHT", 
"GREATER-THAN OR SLANTED EQUAL TO WITH DOT ABOVE LEFT", "LESS-THAN OR APPROXIMATE", 
"LESS-THAN OR APPROXIMATE", "GREATER-THAN OR APPROXIMATE", "GREATER-THAN OR APPROXIMATE", 
"LESS-THAN AND SINGLE-LINE NOT EQUAL TO", "LESS-THAN AND SINGLE-LINE NOT EQUAL TO", 
"GREATER-THAN AND SINGLE-LINE NOT EQUAL TO", "GREATER-THAN AND SINGLE-LINE NOT EQUAL TO", 
"LESS-THAN AND NOT APPROXIMATE", "LESS-THAN AND NOT APPROXIMATE", 
"GREATER-THAN AND NOT APPROXIMATE", "GREATER-THAN AND NOT APPROXIMATE", 
"LESS-THAN ABOVE DOUBLE-LINE EQUAL ABOVE GREATER-THAN", "LESS-THAN ABOVE DOUBLE-LINE EQUAL ABOVE GREATER-THAN", 
"GREATER-THAN ABOVE DOUBLE-LINE EQUAL ABOVE LESS-THAN", "GREATER-THAN ABOVE DOUBLE-LINE EQUAL ABOVE LESS-THAN", 
"LESS-THAN ABOVE SIMILAR OR EQUAL", "GREATER-THAN ABOVE SIMILAR OR EQUAL", 
"LESS-THAN ABOVE SIMILAR ABOVE GREATER-THAN", "GREATER-THAN ABOVE SIMILAR ABOVE LESS-THAN", 
"LESS-THAN ABOVE GREATER-THAN ABOVE DOUBLE-LINE EQUAL", "GREATER-THAN ABOVE LESS-THAN ABOVE DOUBLE-LINE EQUAL", 
"LESS-THAN ABOVE SLANTED EQUAL ABOVE GREATER-THAN ABOVE SLANTED EQUAL", 
"GREATER-THAN ABOVE SLANTED EQUAL ABOVE LESS-THAN ABOVE SLANTED EQUAL", 
"SLANTED EQUAL TO OR LESS-THAN", "SLANTED EQUAL TO OR LESS-THAN", 
"SLANTED EQUAL TO OR GREATER-THAN", "SLANTED EQUAL TO OR GREATER-THAN", 
"SLANTED EQUAL TO OR LESS-THAN WITH DOT INSIDE", "SLANTED EQUAL TO OR GREATER-THAN WITH DOT INSIDE", 
"DOUBLE-LINE EQUAL TO OR LESS-THAN", "DOUBLE-LINE EQUAL TO OR GREATER-THAN", 
"SIMILAR OR LESS-THAN", "SIMILAR OR GREATER-THAN", "SIMILAR ABOVE LESS-THAN ABOVE EQUALS SIGN", 
"SIMILAR ABOVE GREATER-THAN ABOVE EQUALS SIGN", "DOUBLE NESTED LESS-THAN", 
"DOUBLE NESTED LESS-THAN with slash", "DOUBLE NESTED LESS-THAN with slash", 
"DOUBLE NESTED GREATER-THAN", "DOUBLE NESTED GREATER-THAN with slash", 
"DOUBLE NESTED GREATER-THAN with slash", "GREATER-THAN OVERLAPPING LESS-THAN", 
"GREATER-THAN BESIDE LESS-THAN", "LESS-THAN CLOSED BY CURVE", 
"GREATER-THAN CLOSED BY CURVE", "LESS-THAN CLOSED BY CURVE ABOVE SLANTED EQUAL", 
"GREATER-THAN CLOSED BY CURVE ABOVE SLANTED EQUAL", "SMALLER THAN", 
"LARGER THAN", "SMALLER THAN OR EQUAL TO", "SMALLER THAN OR slanted EQUAL", 
"SMALLER THAN OR slanted EQUAL", "LARGER THAN OR EQUAL TO", "LARGER THAN OR slanted EQUAL", 
"LARGER THAN OR slanted EQUAL", "EQUALS SIGN WITH BUMPY ABOVE", 
"PRECEDES ABOVE SINGLE-LINE EQUALS SIGN", "PRECEDES ABOVE SINGLE-LINE EQUALS SIGN", 
"PRECEDES ABOVE SINGLE-LINE EQUALS SIGN", "PRECEDES ABOVE SINGLE-LINE EQUALS SIGN with slash", 
"PRECEDES ABOVE SINGLE-LINE EQUALS SIGN with slash", "PRECEDES ABOVE SINGLE-LINE EQUALS SIGN with slash", 
"PRECEDES ABOVE SINGLE-LINE EQUALS SIGN with slash", "PRECEDES ABOVE SINGLE-LINE EQUALS SIGN with slash", 
"PRECEDES ABOVE SINGLE-LINE EQUALS SIGN with slash", "SUCCEEDS ABOVE SINGLE-LINE EQUALS SIGN", 
"SUCCEEDS ABOVE SINGLE-LINE EQUALS SIGN", "SUCCEEDS ABOVE SINGLE-LINE EQUALS SIGN", 
"SUCCEEDS ABOVE SINGLE-LINE EQUALS SIGN with slash", "SUCCEEDS ABOVE SINGLE-LINE EQUALS SIGN with slash", 
"SUCCEEDS ABOVE SINGLE-LINE EQUALS SIGN with slash", "SUCCEEDS ABOVE SINGLE-LINE EQUALS SIGN with slash", 
"SUCCEEDS ABOVE SINGLE-LINE EQUALS SIGN with slash", "SUCCEEDS ABOVE SINGLE-LINE EQUALS SIGN with slash", 
"PRECEDES ABOVE EQUALS SIGN", "SUCCEEDS ABOVE EQUALS SIGN", "PRECEDES ABOVE NOT EQUAL TO", 
"PRECEDES ABOVE NOT EQUAL TO", "SUCCEEDS ABOVE NOT EQUAL TO", 
"SUCCEEDS ABOVE NOT EQUAL TO", "PRECEDES ABOVE ALMOST EQUAL TO", 
"PRECEDES ABOVE ALMOST EQUAL TO", "SUCCEEDS ABOVE ALMOST EQUAL TO", 
"SUCCEEDS ABOVE ALMOST EQUAL TO", "PRECEDES ABOVE NOT ALMOST EQUAL TO", 
"PRECEDES ABOVE NOT ALMOST EQUAL TO", "SUCCEEDS ABOVE NOT ALMOST EQUAL TO", 
"SUCCEEDS ABOVE NOT ALMOST EQUAL TO", "DOUBLE PRECEDES", "DOUBLE SUCCEEDS", 
"SUBSET WITH DOT", "SUPERSET WITH DOT", "SUBSET WITH PLUS SIGN BELOW", 
"SUPERSET WITH PLUS SIGN BELOW", "SUBSET WITH MULTIPLICATION SIGN BELOW", 
"SUPERSET WITH MULTIPLICATION SIGN BELOW", "SUBSET OF OR EQUAL TO WITH DOT ABOVE", 
"SUPERSET OF OR EQUAL TO WITH DOT ABOVE", "SUBSET OF ABOVE EQUALS SIGN", 
"SUBSET OF ABOVE EQUALS SIGN", "SUBSET OF ABOVE EQUALS SIGN with slash", 
"SUBSET OF ABOVE EQUALS SIGN with slash", "SUBSET OF ABOVE EQUALS SIGN with slash", 
"SUBSET OF ABOVE EQUALS SIGN with slash", "SUPERSET OF ABOVE EQUALS SIGN", 
"SUPERSET OF ABOVE EQUALS SIGN", "SUPERSET OF ABOVE EQUALS SIGN with slash", 
"SUPERSET OF ABOVE EQUALS SIGN with slash", "SUPERSET OF ABOVE EQUALS SIGN with slash", 
"SUPERSET OF ABOVE EQUALS SIGN with slash", "SUBSET OF ABOVE TILDE OPERATOR", 
"SUPERSET OF ABOVE TILDE OPERATOR", "SUBSET OF ABOVE NOT EQUAL TO", 
"SUBSET OF ABOVE NOT EQUAL TO", "SUBSET OF ABOVE NOT EQUAL TO - variant with stroke through bottom members", 
"SUBSET OF ABOVE NOT EQUAL TO - variant with stroke through bottom members", 
"SUBSET OF ABOVE NOT EQUAL TO - variant with stroke through bottom members", 
"SUBSET OF ABOVE NOT EQUAL TO - variant with stroke through bottom members", 
"SUPERSET OF ABOVE NOT EQUAL TO", "SUPERSET OF ABOVE NOT EQUAL TO", 
"SUPERSET OF ABOVE NOT EQUAL TO - variant with stroke through bottom members", 
"SUPERSET OF ABOVE NOT EQUAL TO - variant with stroke through bottom members", 
"SUPERSET OF ABOVE NOT EQUAL TO - variant with stroke through bottom members", 
"SUPERSET OF ABOVE NOT EQUAL TO - variant with stroke through bottom members", 
"CLOSED SUBSET", "CLOSED SUPERSET", "CLOSED SUBSET OR EQUAL TO", 
"CLOSED SUPERSET OR EQUAL TO", "SUBSET ABOVE SUPERSET", "SUPERSET ABOVE SUBSET", 
"SUBSET ABOVE SUBSET", "SUPERSET ABOVE SUPERSET", "SUPERSET BESIDE SUBSET", 
"SUPERSET BESIDE AND JOINED BY DASH WITH SUBSET", "ELEMENT OF OPENING DOWNWARDS", 
"PITCHFORK WITH TEE TOP", "TRANSVERSAL INTERSECTION", "VERTICAL BAR DOUBLE LEFT TURNSTILE", 
"VERTICAL BAR DOUBLE LEFT TURNSTILE", "LONG DASH FROM LEFT MEMBER OF DOUBLE VERTICAL", 
"SHORT DOWN TACK WITH OVERBAR", "SHORT UP TACK WITH UNDERBAR", 
"SHORT UP TACK ABOVE SHORT DOWN TACK", "DOUBLE UP TACK", "DOUBLE STROKE NOT SIGN", 
"REVERSED DOUBLE STROKE NOT SIGN", "DOES NOT DIVIDE WITH REVERSED NEGATION SLASH", 
"VERTICAL LINE WITH CIRCLE ABOVE", "VERTICAL LINE WITH CIRCLE BELOW", 
"DOWN TACK WITH CIRCLE BELOW", "PARALLEL WITH HORIZONTAL STROKE", 
"PARALLEL WITH TILDE OPERATOR", "DOUBLE SOLIDUS OPERATOR", "DOUBLE SOLIDUS OPERATOR with reverse slash", 
"DOUBLE SOLIDUS OPERATOR with reverse slash", "LATIN SMALL LIGATURE FF", 
"LATIN SMALL LIGATURE FI", "LATIN SMALL LIGATURE FL", "LATIN SMALL LIGATURE FFI", 
"LATIN SMALL LIGATURE FFL", "MATHEMATICAL SCRIPT CAPITAL A", 
"MATHEMATICAL SCRIPT CAPITAL C", "MATHEMATICAL SCRIPT CAPITAL D", 
"MATHEMATICAL SCRIPT CAPITAL G", "MATHEMATICAL SCRIPT CAPITAL J", 
"MATHEMATICAL SCRIPT CAPITAL K", "MATHEMATICAL SCRIPT CAPITAL N", 
"MATHEMATICAL SCRIPT CAPITAL O", "MATHEMATICAL SCRIPT CAPITAL P", 
"MATHEMATICAL SCRIPT CAPITAL Q", "MATHEMATICAL SCRIPT CAPITAL S", 
"MATHEMATICAL SCRIPT CAPITAL T", "MATHEMATICAL SCRIPT CAPITAL U", 
"MATHEMATICAL SCRIPT CAPITAL V", "MATHEMATICAL SCRIPT CAPITAL W", 
"MATHEMATICAL SCRIPT CAPITAL X", "MATHEMATICAL SCRIPT CAPITAL Y", 
"MATHEMATICAL SCRIPT CAPITAL Z", "MATHEMATICAL SCRIPT SMALL A", 
"MATHEMATICAL SCRIPT SMALL B", "MATHEMATICAL SCRIPT SMALL C", 
"MATHEMATICAL SCRIPT SMALL D", "MATHEMATICAL SCRIPT SMALL F", 
"MATHEMATICAL SCRIPT SMALL H", "MATHEMATICAL SCRIPT SMALL I", 
"MATHEMATICAL SCRIPT SMALL J", "MATHEMATICAL SCRIPT SMALL K", 
"MATHEMATICAL SCRIPT SMALL L", "MATHEMATICAL SCRIPT SMALL M", 
"MATHEMATICAL SCRIPT SMALL N", "MATHEMATICAL SCRIPT SMALL P", 
"MATHEMATICAL SCRIPT SMALL Q", "MATHEMATICAL SCRIPT SMALL R", 
"MATHEMATICAL SCRIPT SMALL S", "MATHEMATICAL SCRIPT SMALL T", 
"MATHEMATICAL SCRIPT SMALL U", "MATHEMATICAL SCRIPT SMALL V", 
"MATHEMATICAL SCRIPT SMALL W", "MATHEMATICAL SCRIPT SMALL X", 
"MATHEMATICAL SCRIPT SMALL Y", "MATHEMATICAL SCRIPT SMALL Z", 
"MATHEMATICAL FRAKTUR CAPITAL A", "MATHEMATICAL FRAKTUR CAPITAL B", 
"MATHEMATICAL FRAKTUR CAPITAL D", "MATHEMATICAL FRAKTUR CAPITAL E", 
"MATHEMATICAL FRAKTUR CAPITAL F", "MATHEMATICAL FRAKTUR CAPITAL G", 
"MATHEMATICAL FRAKTUR CAPITAL J", "MATHEMATICAL FRAKTUR CAPITAL K", 
"MATHEMATICAL FRAKTUR CAPITAL L", "MATHEMATICAL FRAKTUR CAPITAL M", 
"MATHEMATICAL FRAKTUR CAPITAL N", "MATHEMATICAL FRAKTUR CAPITAL O", 
"MATHEMATICAL FRAKTUR CAPITAL P", "MATHEMATICAL FRAKTUR CAPITAL Q", 
"MATHEMATICAL FRAKTUR CAPITAL S", "MATHEMATICAL FRAKTUR CAPITAL T", 
"MATHEMATICAL FRAKTUR CAPITAL U", "MATHEMATICAL FRAKTUR CAPITAL V", 
"MATHEMATICAL FRAKTUR CAPITAL W", "MATHEMATICAL FRAKTUR CAPITAL X", 
"MATHEMATICAL FRAKTUR CAPITAL Y", "MATHEMATICAL FRAKTUR SMALL A", 
"MATHEMATICAL FRAKTUR SMALL B", "MATHEMATICAL FRAKTUR SMALL C", 
"MATHEMATICAL FRAKTUR SMALL D", "MATHEMATICAL FRAKTUR SMALL E", 
"MATHEMATICAL FRAKTUR SMALL F", "MATHEMATICAL FRAKTUR SMALL G", 
"MATHEMATICAL FRAKTUR SMALL H", "MATHEMATICAL FRAKTUR SMALL I", 
"MATHEMATICAL FRAKTUR SMALL J", "MATHEMATICAL FRAKTUR SMALL K", 
"MATHEMATICAL FRAKTUR SMALL L", "MATHEMATICAL FRAKTUR SMALL M", 
"MATHEMATICAL FRAKTUR SMALL N", "MATHEMATICAL FRAKTUR SMALL O", 
"MATHEMATICAL FRAKTUR SMALL P", "MATHEMATICAL FRAKTUR SMALL Q", 
"MATHEMATICAL FRAKTUR SMALL R", "MATHEMATICAL FRAKTUR SMALL S", 
"MATHEMATICAL FRAKTUR SMALL T", "MATHEMATICAL FRAKTUR SMALL U", 
"MATHEMATICAL FRAKTUR SMALL V", "MATHEMATICAL FRAKTUR SMALL W", 
"MATHEMATICAL FRAKTUR SMALL X", "MATHEMATICAL FRAKTUR SMALL Y", 
"MATHEMATICAL FRAKTUR SMALL Z", "MATHEMATICAL DOUBLE-STRUCK CAPITAL A", 
"MATHEMATICAL DOUBLE-STRUCK CAPITAL B", "MATHEMATICAL DOUBLE-STRUCK CAPITAL D", 
"MATHEMATICAL DOUBLE-STRUCK CAPITAL E", "MATHEMATICAL DOUBLE-STRUCK CAPITAL F", 
"MATHEMATICAL DOUBLE-STRUCK CAPITAL G", "MATHEMATICAL DOUBLE-STRUCK CAPITAL I", 
"MATHEMATICAL DOUBLE-STRUCK CAPITAL J", "MATHEMATICAL DOUBLE-STRUCK CAPITAL K", 
"MATHEMATICAL DOUBLE-STRUCK CAPITAL L", "MATHEMATICAL DOUBLE-STRUCK CAPITAL M", 
"MATHEMATICAL DOUBLE-STRUCK CAPITAL O", "MATHEMATICAL DOUBLE-STRUCK CAPITAL S", 
"MATHEMATICAL DOUBLE-STRUCK CAPITAL T", "MATHEMATICAL DOUBLE-STRUCK CAPITAL U", 
"MATHEMATICAL DOUBLE-STRUCK CAPITAL V", "MATHEMATICAL DOUBLE-STRUCK CAPITAL W", 
"MATHEMATICAL DOUBLE-STRUCK CAPITAL X", "MATHEMATICAL DOUBLE-STRUCK CAPITAL Y", 
"MATHEMATICAL DOUBLE-STRUCK SMALL A", "MATHEMATICAL DOUBLE-STRUCK SMALL B", 
"MATHEMATICAL DOUBLE-STRUCK SMALL C", "MATHEMATICAL DOUBLE-STRUCK SMALL D", 
"MATHEMATICAL DOUBLE-STRUCK SMALL E", "MATHEMATICAL DOUBLE-STRUCK SMALL F", 
"MATHEMATICAL DOUBLE-STRUCK SMALL G", "MATHEMATICAL DOUBLE-STRUCK SMALL H", 
"MATHEMATICAL DOUBLE-STRUCK SMALL I", "MATHEMATICAL DOUBLE-STRUCK SMALL J", 
"MATHEMATICAL DOUBLE-STRUCK SMALL K", "MATHEMATICAL DOUBLE-STRUCK SMALL L", 
"MATHEMATICAL DOUBLE-STRUCK SMALL M", "MATHEMATICAL DOUBLE-STRUCK SMALL N", 
"MATHEMATICAL DOUBLE-STRUCK SMALL O", "MATHEMATICAL DOUBLE-STRUCK SMALL P", 
"MATHEMATICAL DOUBLE-STRUCK SMALL Q", "MATHEMATICAL DOUBLE-STRUCK SMALL R", 
"MATHEMATICAL DOUBLE-STRUCK SMALL S", "MATHEMATICAL DOUBLE-STRUCK SMALL T", 
"MATHEMATICAL DOUBLE-STRUCK SMALL U", "MATHEMATICAL DOUBLE-STRUCK SMALL V", 
"MATHEMATICAL DOUBLE-STRUCK SMALL W", "MATHEMATICAL DOUBLE-STRUCK SMALL X", 
"MATHEMATICAL DOUBLE-STRUCK SMALL Y", "MATHEMATICAL DOUBLE-STRUCK SMALL Z", 
"MATHEMATICAL BOLD CAPITAL GAMMA", "MATHEMATICAL BOLD CAPITAL DELTA", 
"MATHEMATICAL BOLD CAPITAL THETA", "MATHEMATICAL BOLD CAPITAL LAMDA", 
"MATHEMATICAL BOLD CAPITAL XI", "MATHEMATICAL BOLD CAPITAL PI", 
"MATHEMATICAL BOLD CAPITAL SIGMA", "MATHEMATICAL BOLD CAPITAL UPSILON", 
"MATHEMATICAL BOLD CAPITAL PHI", "MATHEMATICAL BOLD CAPITAL PSI", 
"MATHEMATICAL BOLD CAPITAL OMEGA", "MATHEMATICAL BOLD SMALL ALPHA", 
"MATHEMATICAL BOLD SMALL BETA", "MATHEMATICAL BOLD SMALL GAMMA", 
"MATHEMATICAL BOLD SMALL DELTA", "MATHEMATICAL BOLD SMALL EPSILON", 
"MATHEMATICAL BOLD SMALL ZETA", "MATHEMATICAL BOLD SMALL ETA", 
"MATHEMATICAL BOLD SMALL THETA", "MATHEMATICAL BOLD SMALL IOTA", 
"MATHEMATICAL BOLD SMALL KAPPA", "MATHEMATICAL BOLD SMALL LAMDA", 
"MATHEMATICAL BOLD SMALL MU", "MATHEMATICAL BOLD SMALL NU", "MATHEMATICAL BOLD SMALL XI", 
"MATHEMATICAL BOLD SMALL PI", "MATHEMATICAL BOLD SMALL RHO", 
"MATHEMATICAL BOLD SMALL FINAL SIGMA", "MATHEMATICAL BOLD SMALL SIGMA", 
"MATHEMATICAL BOLD SMALL TAU", "MATHEMATICAL BOLD SMALL UPSILON", 
"MATHEMATICAL BOLD SMALL PHI", "MATHEMATICAL BOLD SMALL CHI", 
"MATHEMATICAL BOLD SMALL PSI", "MATHEMATICAL BOLD SMALL OMEGA", 
"MATHEMATICAL BOLD EPSILON SYMBOL", "MATHEMATICAL BOLD THETA SYMBOL", 
"MATHEMATICAL BOLD KAPPA SYMBOL", "MATHEMATICAL BOLD PHI SYMBOL", 
"MATHEMATICAL BOLD RHO SYMBOL", "MATHEMATICAL BOLD PI SYMBOL", 
"MATHEMATICAL BOLD CAPITAL DIGAMMA", "MATHEMATICAL BOLD SMALL DIGAMMA"
), unicode = c("U+0009", "U+000A", "U+0021", "U+0022", "U+0022", 
"U+0023", "U+0024", "U+0025", "U+0026", "U+0026", "U+0027", "U+0028", 
"U+0029", "U+002A", "U+002A", "U+002B", "U+002C", "U+002E", "U+002F", 
"U+003A", "U+003B", "U+003C", "U+003C", "U+003C", "U+20D2", "U+003D", 
"U+003D", "U+20E5", "U+003E", "U+003E", "U+003E", "U+20D2", "U+003F", 
"U+0040", "U+005B", "U+005B", "U+005C", "U+005D", "U+005D", "U+005E", 
"U+005F", "U+005F", "U+0060", "U+0060", "U+0066", "U+006A", "U+007B", 
"U+007B", "U+007C", "U+007C", "U+007C", "U+007D", "U+007D", "U+00A0", 
"U+00A0", "U+00A1", "U+00A2", "U+00A3", "U+00A4", "U+00A5", "U+00A6", 
"U+00A7", "U+00A8", "U+00A8", "U+00A8", "U+00A8", "U+00A9", "U+00A9", 
"U+00AA", "U+00AB", "U+00AC", "U+00AD", "U+00AE", "U+00AE", "U+00AE", 
"U+00AF", "U+00AF", "U+00B0", "U+00B1", "U+00B1", "U+00B1", "U+00B2", 
"U+00B3", "U+00B4", "U+00B4", "U+00B5", "U+00B6", "U+00B7", "U+00B7", 
"U+00B7", "U+00B8", "U+00B8", "U+00B9", "U+00BA", "U+00BB", "U+00BC", 
"U+00BD", "U+00BD", "U+00BE", "U+00BF", "U+00C0", "U+00C1", "U+00C2", 
"U+00C3", "U+00C4", "U+00C5", "U+00C5", "U+00C6", "U+00C7", "U+00C8", 
"U+00C9", "U+00CA", "U+00CB", "U+00CC", "U+00CD", "U+00CE", "U+00CF", 
"U+00D0", "U+00D1", "U+00D2", "U+00D3", "U+00D4", "U+00D5", "U+00D6", 
"U+00D7", "U+00D8", "U+00D9", "U+00DA", "U+00DB", "U+00DC", "U+00DD", 
"U+00DE", "U+00DF", "U+00E0", "U+00E1", "U+00E2", "U+00E3", "U+00E4", 
"U+00E5", "U+00E6", "U+00E7", "U+00E8", "U+00E9", "U+00EA", "U+00EB", 
"U+00EC", "U+00ED", "U+00EE", "U+00EF", "U+00F0", "U+00F1", "U+00F2", 
"U+00F3", "U+00F4", "U+00F5", "U+00F6", "U+00F7", "U+00F7", "U+00F8", 
"U+00F9", "U+00FA", "U+00FB", "U+00FC", "U+00FD", "U+00FE", "U+00FF", 
"U+0100", "U+0101", "U+0102", "U+0103", "U+0104", "U+0105", "U+0106", 
"U+0107", "U+0108", "U+0109", "U+010A", "U+010B", "U+010C", "U+010D", 
"U+010E", "U+010F", "U+0110", "U+0111", "U+0112", "U+0113", "U+0116", 
"U+0117", "U+0118", "U+0119", "U+011A", "U+011B", "U+011C", "U+011D", 
"U+011E", "U+011F", "U+0120", "U+0121", "U+0122", "U+0124", "U+0125", 
"U+0126", "U+0127", "U+0128", "U+0129", "U+012A", "U+012B", "U+012E", 
"U+012F", "U+0130", "U+0131", "U+0131", "U+0132", "U+0133", "U+0134", 
"U+0135", "U+0136", "U+0137", "U+0138", "U+0139", "U+013A", "U+013B", 
"U+013C", "U+013D", "U+013E", "U+013F", "U+0140", "U+0141", "U+0142", 
"U+0143", "U+0144", "U+0145", "U+0146", "U+0147", "U+0148", "U+0149", 
"U+014A", "U+014B", "U+014C", "U+014D", "U+0150", "U+0151", "U+0152", 
"U+0153", "U+0154", "U+0155", "U+0156", "U+0157", "U+0158", "U+0159", 
"U+015A", "U+015B", "U+015C", "U+015D", "U+015E", "U+015F", "U+0160", 
"U+0161", "U+0162", "U+0163", "U+0164", "U+0165", "U+0166", "U+0167", 
"U+0168", "U+0169", "U+016A", "U+016B", "U+016C", "U+016D", "U+016E", 
"U+016F", "U+0170", "U+0171", "U+0172", "U+0173", "U+0174", "U+0175", 
"U+0176", "U+0177", "U+0178", "U+0179", "U+017A", "U+017B", "U+017C", 
"U+017D", "U+017E", "U+0192", "U+01B5", "U+01F5", "U+0237", "U+02C6", 
"U+02C7", "U+02C7", "U+02D8", "U+02D8", "U+02D9", "U+02D9", "U+02DA", 
"U+02DB", "U+02DC", "U+02DC", "U+02DD", "U+02DD", "U+0311", "U+0386", 
"U+0388", "U+0389", "U+038A", "U+038C", "U+038E", "U+038F", "U+0390", 
"U+0391", "U+0391", "U+0392", "U+0392", "U+0393", "U+0393", "U+0394", 
"U+0394", "U+0395", "U+0395", "U+0396", "U+0396", "U+0397", "U+0397", 
"U+0398", "U+0398", "U+0399", "U+0399", "U+039A", "U+039A", "U+039B", 
"U+039B", "U+039C", "U+039C", "U+039D", "U+039D", "U+039E", "U+039E", 
"U+039F", "U+039F", "U+03A0", "U+03A0", "U+03A1", "U+03A1", "U+03A3", 
"U+03A3", "U+03A4", "U+03A4", "U+03A5", "U+03A5", "U+03A6", "U+03A6", 
"U+03A7", "U+03A7", "U+03A8", "U+03A8", "U+03A9", "U+03A9", "U+03A9", 
"U+03AA", "U+03AB", "U+03AC", "U+03AD", "U+03AE", "U+03AF", "U+03B0", 
"U+03B1", "U+03B1", "U+03B2", "U+03B2", "U+03B3", "U+03B3", "U+03B4", 
"U+03B4", "U+03B5", "U+03B5", "U+03B5", "U+03B6", "U+03B6", "U+03B7", 
"U+03B7", "U+03B8", "U+03B8", "U+03B9", "U+03B9", "U+03BA", "U+03BA", 
"U+03BB", "U+03BB", "U+03BC", "U+03BC", "U+03BD", "U+03BD", "U+03BE", 
"U+03BE", "U+03BF", "U+03BF", "U+03C0", "U+03C0", "U+03C1", "U+03C1", 
"U+03C2", "U+03C2", "U+03C2", "U+03C2", "U+03C3", "U+03C3", "U+03C4", 
"U+03C4", "U+03C5", "U+03C5", "U+03C5", "U+03C6", "U+03C6", "U+03C7", 
"U+03C7", "U+03C8", "U+03C8", "U+03C9", "U+03C9", "U+03CA", "U+03CB", 
"U+03CC", "U+03CD", "U+03CE", "U+03D1", "U+03D1", "U+03D1", "U+03D2", 
"U+03D2", "U+03D5", "U+03D5", "U+03D5", "U+03D6", "U+03D6", "U+03DC", 
"U+03DD", "U+03DD", "U+03F0", "U+03F0", "U+03F1", "U+03F1", "U+03F5", 
"U+03F5", "U+03F5", "U+03F6", "U+03F6", "U+0401", "U+0402", "U+0403", 
"U+0404", "U+0405", "U+0406", "U+0407", "U+0408", "U+0409", "U+040A", 
"U+040B", "U+040C", "U+040E", "U+040F", "U+0410", "U+0411", "U+0412", 
"U+0413", "U+0414", "U+0415", "U+0416", "U+0417", "U+0418", "U+0419", 
"U+041A", "U+041B", "U+041C", "U+041D", "U+041E", "U+041F", "U+0420", 
"U+0421", "U+0422", "U+0423", "U+0424", "U+0425", "U+0426", "U+0427", 
"U+0428", "U+0429", "U+042A", "U+042B", "U+042C", "U+042D", "U+042E", 
"U+042F", "U+0430", "U+0431", "U+0432", "U+0433", "U+0434", "U+0435", 
"U+0436", "U+0437", "U+0438", "U+0439", "U+043A", "U+043B", "U+043C", 
"U+043D", "U+043E", "U+043F", "U+0440", "U+0441", "U+0442", "U+0443", 
"U+0444", "U+0445", "U+0446", "U+0447", "U+0448", "U+0449", "U+044A", 
"U+044B", "U+044C", "U+044D", "U+044E", "U+044F", "U+0451", "U+0452", 
"U+0453", "U+0454", "U+0455", "U+0456", "U+0457", "U+0458", "U+0459", 
"U+045A", "U+045B", "U+045C", "U+045E", "U+045F", "U+2002", "U+2003", 
"U+2004", "U+2005", "U+2007", "U+2008", "U+2009", "U+2009", "U+200A", 
"U+200A", "U+200B", "U+200B", "U+200B", "U+200B", "U+200B", "U+200C", 
"U+200D", "U+200E", "U+200F", "U+2010", "U+2010", "U+2013", "U+2014", 
"U+2015", "U+2016", "U+2016", "U+2018", "U+2018", "U+2019", "U+2019", 
"U+2019", "U+201A", "U+201A", "U+201C", "U+201C", "U+201D", "U+201D", 
"U+201D", "U+201E", "U+201E", "U+2020", "U+2021", "U+2021", "U+2022", 
"U+2022", "U+2025", "U+2026", "U+2026", "U+2030", "U+2031", "U+2032", 
"U+2033", "U+2034", "U+2035", "U+2035", "U+2039", "U+203A", "U+203E", 
"U+203E", "U+2041", "U+2043", "U+2044", "U+204F", "U+2057", "U+205F", 
"U+205F", "U+200A", "U+2060", "U+2061", "U+2061", "U+2062", "U+2062", 
"U+2063", "U+2063", "U+20AC", "U+20DB", "U+20DB", "U+20DC", "U+2102", 
"U+2102", "U+2105", "U+210A", "U+210B", "U+210B", "U+210B", "U+210C", 
"U+210C", "U+210D", "U+210D", "U+210E", "U+210F", "U+210F", "U+210F", 
"U+210F", "U+2110", "U+2110", "U+2111", "U+2111", "U+2111", "U+2111", 
"U+2112", "U+2112", "U+2112", "U+2113", "U+2115", "U+2115", "U+2116", 
"U+2117", "U+2118", "U+2118", "U+2119", "U+2119", "U+211A", "U+211A", 
"U+211B", "U+211B", "U+211C", "U+211C", "U+211C", "U+211C", "U+211D", 
"U+211D", "U+211E", "U+2122", "U+2122", "U+2124", "U+2124", "U+2127", 
"U+2128", "U+2128", "U+2129", "U+212C", "U+212C", "U+212C", "U+212D", 
"U+212D", "U+212F", "U+2130", "U+2130", "U+2131", "U+2131", "U+2133", 
"U+2133", "U+2133", "U+2134", "U+2134", "U+2134", "U+2135", "U+2135", 
"U+2136", "U+2137", "U+2138", "U+2145", "U+2145", "U+2146", "U+2146", 
"U+2147", "U+2147", "U+2147", "U+2148", "U+2148", "U+2153", "U+2154", 
"U+2155", "U+2156", "U+2157", "U+2158", "U+2159", "U+215A", "U+215B", 
"U+215C", "U+215D", "U+215E", "U+2190", "U+2190", "U+2190", "U+2190", 
"U+2190", "U+2191", "U+2191", "U+2191", "U+2191", "U+2192", "U+2192", 
"U+2192", "U+2192", "U+2192", "U+2193", "U+2193", "U+2193", "U+2193", 
"U+2194", "U+2194", "U+2194", "U+2195", "U+2195", "U+2195", "U+2196", 
"U+2196", "U+2196", "U+2197", "U+2197", "U+2197", "U+2198", "U+2198", 
"U+2198", "U+2199", "U+2199", "U+2199", "U+219A", "U+219A", "U+219B", 
"U+219B", "U+219D", "U+219D", "U+219D", "U+0338", "U+219E", "U+219E", 
"U+219F", "U+21A0", "U+21A0", "U+21A1", "U+21A2", "U+21A2", "U+21A3", 
"U+21A3", "U+21A4", "U+21A4", "U+21A5", "U+21A5", "U+21A6", "U+21A6", 
"U+21A6", "U+21A7", "U+21A7", "U+21A9", "U+21A9", "U+21AA", "U+21AA", 
"U+21AB", "U+21AB", "U+21AC", "U+21AC", "U+21AD", "U+21AD", "U+21AE", 
"U+21AE", "U+21B0", "U+21B0", "U+21B1", "U+21B1", "U+21B2", "U+21B3", 
"U+21B5", "U+21B6", "U+21B6", "U+21B7", "U+21B7", "U+21BA", "U+21BA", 
"U+21BB", "U+21BB", "U+21BC", "U+21BC", "U+21BC", "U+21BD", "U+21BD", 
"U+21BD", "U+21BE", "U+21BE", "U+21BE", "U+21BF", "U+21BF", "U+21BF", 
"U+21C0", "U+21C0", "U+21C0", "U+21C1", "U+21C1", "U+21C1", "U+21C2", 
"U+21C2", "U+21C2", "U+21C3", "U+21C3", "U+21C3", "U+21C4", "U+21C4", 
"U+21C4", "U+21C5", "U+21C5", "U+21C6", "U+21C6", "U+21C6", "U+21C7", 
"U+21C7", "U+21C8", "U+21C8", "U+21C9", "U+21C9", "U+21CA", "U+21CA", 
"U+21CB", "U+21CB", "U+21CB", "U+21CC", "U+21CC", "U+21CC", "U+21CD", 
"U+21CD", "U+21CE", "U+21CE", "U+21CF", "U+21CF", "U+21D0", "U+21D0", 
"U+21D0", "U+21D1", "U+21D1", "U+21D1", "U+21D2", "U+21D2", "U+21D2", 
"U+21D2", "U+21D3", "U+21D3", "U+21D3", "U+21D4", "U+21D4", "U+21D4", 
"U+21D4", "U+21D5", "U+21D5", "U+21D5", "U+21D6", "U+21D7", "U+21D8", 
"U+21D9", "U+21DA", "U+21DA", "U+21DB", "U+21DB", "U+21DD", "U+21E4", 
"U+21E4", "U+21E5", "U+21E5", "U+21F5", "U+21F5", "U+21FD", "U+21FE", 
"U+21FF", "U+2200", "U+2200", "U+2201", "U+2201", "U+2202", "U+2202", 
"U+2202", "U+0338", "U+2203", "U+2203", "U+2204", "U+2204", "U+2204", 
"U+2205", "U+2205", "U+2205", "U+2205", "U+2207", "U+2207", "U+2208", 
"U+2208", "U+2208", "U+2208", "U+2209", "U+2209", "U+2209", "U+220B", 
"U+220B", "U+220B", "U+220B", "U+220C", "U+220C", "U+220C", "U+220F", 
"U+220F", "U+2210", "U+2210", "U+2211", "U+2211", "U+2212", "U+2213", 
"U+2213", "U+2213", "U+2214", "U+2214", "U+2216", "U+2216", "U+2216", 
"U+2216", "U+2216", "U+2217", "U+2218", "U+2218", "U+221A", "U+221A", 
"U+221D", "U+221D", "U+221D", "U+221D", "U+221D", "U+221E", "U+221F", 
"U+2220", "U+2220", "U+2220", "U+20D2", "U+2221", "U+2221", "U+2222", 
"U+2223", "U+2223", "U+2223", "U+2223", "U+2224", "U+2224", "U+2224", 
"U+2224", "U+2225", "U+2225", "U+2225", "U+2225", "U+2225", "U+2226", 
"U+2226", "U+2226", "U+2226", "U+2226", "U+2227", "U+2227", "U+2228", 
"U+2228", "U+2229", "U+2229", "U+FE00", "U+222A", "U+222A", "U+FE00", 
"U+222B", "U+222B", "U+222C", "U+222D", "U+222D", "U+222E", "U+222E", 
"U+222E", "U+222F", "U+222F", "U+2230", "U+2231", "U+2232", "U+2232", 
"U+2233", "U+2233", "U+2234", "U+2234", "U+2234", "U+2235", "U+2235", 
"U+2235", "U+2236", "U+2237", "U+2237", "U+2238", "U+2238", "U+223A", 
"U+223B", "U+223C", "U+223C", "U+223C", "U+223C", "U+223C", "U+20D2", 
"U+223D", "U+223D", "U+223D", "U+0331", "U+223E", "U+223E", "U+223E", 
"U+0333", "U+223F", "U+2240", "U+2240", "U+2240", "U+2241", "U+2241", 
"U+2242", "U+2242", "U+2242", "U+2242", "U+2242", "U+0338", "U+0338", 
"U+2243", "U+2243", "U+2243", "U+2244", "U+2244", "U+2244", "U+2245", 
"U+2245", "U+2246", "U+2247", "U+2247", "U+2248", "U+2248", "U+2248", 
"U+2248", "U+2248", "U+2248", "U+2249", "U+2249", "U+2249", "U+224A", 
"U+224A", "U+224B", "U+224B", "U+0338", "U+224C", "U+224C", "U+224D", 
"U+224D", "U+224D", "U+20D2", "U+224E", "U+224E", "U+224E", "U+224E", 
"U+224E", "U+0338", "U+0338", "U+224F", "U+224F", "U+224F", "U+224F", 
"U+224F", "U+0338", "U+0338", "U+2250", "U+2250", "U+2250", "U+2250", 
"U+0338", "U+2251", "U+2251", "U+2252", "U+2252", "U+2253", "U+2253", 
"U+2254", "U+2254", "U+2254", "U+2255", "U+2255", "U+2256", "U+2256", 
"U+2257", "U+2257", "U+2259", "U+225A", "U+225C", "U+225C", "U+225F", 
"U+225F", "U+2260", "U+2260", "U+2261", "U+2261", "U+2261", "U+20E5", 
"U+2262", "U+2262", "U+2264", "U+2264", "U+2264", "U+20D2", "U+2265", 
"U+2265", "U+2265", "U+2265", "U+20D2", "U+2266", "U+2266", "U+2266", 
"U+2266", "U+2266", "U+0338", "U+0338", "U+2267", "U+2267", "U+2267", 
"U+2267", "U+2267", "U+2267", "U+0338", "U+0338", "U+0338", "U+2268", 
"U+2268", "U+2268", "U+2268", "U+FE00", "U+FE00", "U+2269", "U+2269", 
"U+2269", "U+2269", "U+FE00", "U+FE00", "U+226A", "U+226A", "U+226A", 
"U+226A", "U+226A", "U+0338", "U+0338", "U+226A", "U+20D2", "U+226B", 
"U+226B", "U+226B", "U+226B", "U+226B", "U+0338", "U+0338", "U+226B", 
"U+20D2", "U+226C", "U+226C", "U+226D", "U+226E", "U+226E", "U+226E", 
"U+226F", "U+226F", "U+226F", "U+2270", "U+2270", "U+2270", "U+2271", 
"U+2271", "U+2271", "U+2272", "U+2272", "U+2272", "U+2273", "U+2273", 
"U+2273", "U+2274", "U+2274", "U+2275", "U+2275", "U+2276", "U+2276", 
"U+2276", "U+2277", "U+2277", "U+2277", "U+2278", "U+2278", "U+2279", 
"U+2279", "U+227A", "U+227A", "U+227A", "U+227B", "U+227B", "U+227B", 
"U+227C", "U+227C", "U+227C", "U+227D", "U+227D", "U+227D", "U+227E", 
"U+227E", "U+227E", "U+227F", "U+227F", "U+227F", "U+227F", "U+0338", 
"U+2280", "U+2280", "U+2280", "U+2281", "U+2281", "U+2281", "U+2282", 
"U+2282", "U+2282", "U+2282", "U+2282", "U+20D2", "U+20D2", "U+20D2", 
"U+2283", "U+2283", "U+2283", "U+2283", "U+2283", "U+2283", "U+20D2", 
"U+20D2", "U+20D2", "U+2284", "U+2285", "U+2286", "U+2286", "U+2286", 
"U+2287", "U+2287", "U+2287", "U+2288", "U+2288", "U+2288", "U+2289", 
"U+2289", "U+2289", "U+228A", "U+228A", "U+228A", "U+228A", "U+FE00", 
"U+FE00", "U+228B", "U+228B", "U+228B", "U+228B", "U+FE00", "U+FE00", 
"U+228D", "U+228E", "U+228E", "U+228F", "U+228F", "U+228F", "U+228F", 
"U+0338", "U+2290", "U+2290", "U+2290", "U+2290", "U+0338", "U+2291", 
"U+2291", "U+2291", "U+2292", "U+2292", "U+2292", "U+2293", "U+2293", 
"U+2293", "U+FE00", "U+2294", "U+2294", "U+2294", "U+FE00", "U+2295", 
"U+2295", "U+2296", "U+2296", "U+2297", "U+2297", "U+2298", "U+2299", 
"U+2299", "U+229A", "U+229A", "U+229B", "U+229B", "U+229D", "U+229D", 
"U+229E", "U+229E", "U+229F", "U+229F", "U+22A0", "U+22A0", "U+22A1", 
"U+22A1", "U+22A2", "U+22A2", "U+22A3", "U+22A3", "U+22A4", "U+22A4", 
"U+22A5", "U+22A5", "U+22A5", "U+22A5", "U+22A7", "U+22A8", "U+22A8", 
"U+22A9", "U+22AA", "U+22AB", "U+22AC", "U+22AD", "U+22AE", "U+22AF", 
"U+22B0", "U+22B2", "U+22B2", "U+22B2", "U+22B3", "U+22B3", "U+22B3", 
"U+22B4", "U+22B4", "U+22B4", "U+22B4", "U+20D2", "U+22B5", "U+22B5", 
"U+22B5", "U+22B5", "U+20D2", "U+22B6", "U+22B7", "U+22B8", "U+22B8", 
"U+22B9", "U+22BA", "U+22BA", "U+22BB", "U+22BD", "U+22BE", "U+22BF", 
"U+22C0", "U+22C0", "U+22C0", "U+22C1", "U+22C1", "U+22C1", "U+22C2", 
"U+22C2", "U+22C2", "U+22C3", "U+22C3", "U+22C3", "U+22C4", "U+22C4", 
"U+22C4", "U+22C5", "U+22C6", "U+22C6", "U+22C7", "U+22C7", "U+22C8", 
"U+22C9", "U+22CA", "U+22CB", "U+22CB", "U+22CC", "U+22CC", "U+22CD", 
"U+22CD", "U+22CE", "U+22CE", "U+22CF", "U+22CF", "U+22D0", "U+22D0", 
"U+22D1", "U+22D1", "U+22D2", "U+22D3", "U+22D4", "U+22D4", "U+22D5", 
"U+22D6", "U+22D6", "U+22D7", "U+22D7", "U+22D8", "U+22D8", "U+0338", 
"U+22D9", "U+22D9", "U+22D9", "U+0338", "U+22DA", "U+22DA", "U+22DA", 
"U+22DA", "U+FE00", "U+22DB", "U+22DB", "U+22DB", "U+22DB", "U+FE00", 
"U+22DE", "U+22DE", "U+22DF", "U+22DF", "U+22E0", "U+22E0", "U+22E1", 
"U+22E1", "U+22E2", "U+22E2", "U+22E3", "U+22E3", "U+22E6", "U+22E7", 
"U+22E8", "U+22E8", "U+22E9", "U+22E9", "U+22EA", "U+22EA", "U+22EA", 
"U+22EB", "U+22EB", "U+22EB", "U+22EC", "U+22EC", "U+22EC", "U+22ED", 
"U+22ED", "U+22ED", "U+22EE", "U+22EF", "U+22F0", "U+22F1", "U+22F2", 
"U+22F3", "U+22F4", "U+22F5", "U+22F5", "U+0338", "U+22F6", "U+22F7", 
"U+22F9", "U+22F9", "U+0338", "U+22FA", "U+22FB", "U+22FC", "U+22FD", 
"U+22FE", "U+2305", "U+2305", "U+2306", "U+2306", "U+2308", "U+2308", 
"U+2309", "U+2309", "U+230A", "U+230A", "U+230B", "U+230B", "U+230C", 
"U+230D", "U+230E", "U+230F", "U+2310", "U+2312", "U+2313", "U+2315", 
"U+2316", "U+231C", "U+231C", "U+231D", "U+231D", "U+231E", "U+231E", 
"U+231F", "U+231F", "U+2322", "U+2322", "U+2323", "U+2323", "U+232D", 
"U+232E", "U+2336", "U+233D", "U+233F", "U+237C", "U+23B0", "U+23B0", 
"U+23B1", "U+23B1", "U+23B4", "U+23B4", "U+23B5", "U+23B5", "U+23B6", 
"U+23DC", "U+23DD", "U+23DE", "U+23DF", "U+23E2", "U+23E7", "U+2423", 
"U+24C8", "U+24C8", "U+2500", "U+2500", "U+2502", "U+250C", "U+2510", 
"U+2514", "U+2518", "U+251C", "U+2524", "U+252C", "U+2534", "U+253C", 
"U+2550", "U+2551", "U+2552", "U+2553", "U+2554", "U+2555", "U+2556", 
"U+2557", "U+2558", "U+2559", "U+255A", "U+255B", "U+255C", "U+255D", 
"U+255E", "U+255F", "U+2560", "U+2561", "U+2562", "U+2563", "U+2564", 
"U+2565", "U+2566", "U+2567", "U+2568", "U+2569", "U+256A", "U+256B", 
"U+256C", "U+2580", "U+2584", "U+2588", "U+2591", "U+2592", "U+2593", 
"U+25A1", "U+25A1", "U+25A1", "U+25AA", "U+25AA", "U+25AA", "U+25AA", 
"U+25AB", "U+25AD", "U+25AE", "U+25B1", "U+25B3", "U+25B3", "U+25B4", 
"U+25B4", "U+25B5", "U+25B5", "U+25B8", "U+25B8", "U+25B9", "U+25B9", 
"U+25BD", "U+25BD", "U+25BE", "U+25BE", "U+25BF", "U+25BF", "U+25C2", 
"U+25C2", "U+25C3", "U+25C3", "U+25CA", "U+25CA", "U+25CB", "U+25EC", 
"U+25EF", "U+25EF", "U+25F8", "U+25F9", "U+25FA", "U+25FB", "U+25FC", 
"U+2605", "U+2605", "U+2606", "U+260E", "U+2640", "U+2642", "U+2660", 
"U+2660", "U+2663", "U+2663", "U+2665", "U+2665", "U+2666", "U+2666", 
"U+266A", "U+266D", "U+266E", "U+266E", "U+266F", "U+2713", "U+2713", 
"U+2717", "U+2720", "U+2720", "U+2736", "U+2758", "U+2772", "U+2773", 
"U+27C8", "U+27C9", "U+27E6", "U+27E6", "U+27E7", "U+27E7", "U+27E8", 
"U+27E8", "U+27E8", "U+27E9", "U+27E9", "U+27E9", "U+27EA", "U+27EB", 
"U+27EC", "U+27ED", "U+27F5", "U+27F5", "U+27F5", "U+27F6", "U+27F6", 
"U+27F6", "U+27F7", "U+27F7", "U+27F7", "U+27F8", "U+27F8", "U+27F8", 
"U+27F9", "U+27F9", "U+27F9", "U+27FA", "U+27FA", "U+27FA", "U+27FC", 
"U+27FC", "U+27FF", "U+2902", "U+2903", "U+2904", "U+2905", "U+290C", 
"U+290D", "U+290D", "U+290E", "U+290F", "U+290F", "U+2910", "U+2910", 
"U+2911", "U+2912", "U+2913", "U+2916", "U+2919", "U+291A", "U+291B", 
"U+291C", "U+291D", "U+291E", "U+291F", "U+2920", "U+2923", "U+2924", 
"U+2925", "U+2925", "U+2926", "U+2926", "U+2927", "U+2928", "U+2928", 
"U+2929", "U+2929", "U+292A", "U+2933", "U+2933", "U+0338", "U+2935", 
"U+2936", "U+2937", "U+2938", "U+2939", "U+293C", "U+293D", "U+2945", 
"U+2948", "U+2949", "U+294A", "U+294B", "U+294E", "U+294F", "U+2950", 
"U+2951", "U+2952", "U+2953", "U+2954", "U+2955", "U+2956", "U+2957", 
"U+2958", "U+2959", "U+295A", "U+295B", "U+295C", "U+295D", "U+295E", 
"U+295F", "U+2960", "U+2961", "U+2962", "U+2963", "U+2964", "U+2965", 
"U+2966", "U+2967", "U+2968", "U+2969", "U+296A", "U+296B", "U+296C", 
"U+296D", "U+296E", "U+296E", "U+296F", "U+296F", "U+2970", "U+2971", 
"U+2972", "U+2973", "U+2974", "U+2975", "U+2976", "U+2978", "U+2979", 
"U+297B", "U+297C", "U+297D", "U+297E", "U+297F", "U+2985", "U+2986", 
"U+298B", "U+298C", "U+298D", "U+298E", "U+298F", "U+2990", "U+2991", 
"U+2992", "U+2993", "U+2994", "U+2995", "U+2996", "U+299A", "U+299C", 
"U+299D", "U+29A4", "U+29A5", "U+29A6", "U+29A7", "U+29A8", "U+29A9", 
"U+29AA", "U+29AB", "U+29AC", "U+29AD", "U+29AE", "U+29AF", "U+29B0", 
"U+29B1", "U+29B2", "U+29B3", "U+29B4", "U+29B5", "U+29B6", "U+29B7", 
"U+29B9", "U+29BB", "U+29BC", "U+29BE", "U+29BF", "U+29C0", "U+29C1", 
"U+29C2", "U+29C3", "U+29C4", "U+29C5", "U+29C9", "U+29CD", "U+29CE", 
"U+29CF", "U+29CF", "U+0338", "U+29D0", "U+29D0", "U+0338", "U+29DC", 
"U+29DD", "U+29DE", "U+29E3", "U+29E4", "U+29E5", "U+29EB", "U+29EB", 
"U+29F4", "U+29F6", "U+2A00", "U+2A00", "U+2A01", "U+2A01", "U+2A02", 
"U+2A02", "U+2A04", "U+2A04", "U+2A06", "U+2A06", "U+2A0C", "U+2A0C", 
"U+2A0D", "U+2A10", "U+2A11", "U+2A12", "U+2A13", "U+2A14", "U+2A15", 
"U+2A16", "U+2A17", "U+2A22", "U+2A23", "U+2A24", "U+2A25", "U+2A26", 
"U+2A27", "U+2A29", "U+2A2A", "U+2A2D", "U+2A2E", "U+2A2F", "U+2A30", 
"U+2A31", "U+2A33", "U+2A34", "U+2A35", "U+2A36", "U+2A37", "U+2A38", 
"U+2A39", "U+2A3A", "U+2A3B", "U+2A3C", "U+2A3C", "U+2A3F", "U+2A40", 
"U+2A42", "U+2A43", "U+2A44", "U+2A45", "U+2A46", "U+2A47", "U+2A48", 
"U+2A49", "U+2A4A", "U+2A4B", "U+2A4C", "U+2A4D", "U+2A50", "U+2A53", 
"U+2A54", "U+2A55", "U+2A56", "U+2A57", "U+2A58", "U+2A5A", "U+2A5B", 
"U+2A5C", "U+2A5D", "U+2A5F", "U+2A66", "U+2A6A", "U+2A6D", "U+2A6D", 
"U+0338", "U+2A6E", "U+2A6F", "U+2A70", "U+2A70", "U+0338", "U+2A71", 
"U+2A72", "U+2A73", "U+2A74", "U+2A75", "U+2A77", "U+2A77", "U+2A78", 
"U+2A79", "U+2A7A", "U+2A7B", "U+2A7C", "U+2A7D", "U+2A7D", "U+2A7D", 
"U+2A7D", "U+2A7D", "U+2A7D", "U+0338", "U+0338", "U+0338", "U+2A7E", 
"U+2A7E", "U+2A7E", "U+2A7E", "U+2A7E", "U+2A7E", "U+0338", "U+0338", 
"U+0338", "U+2A7F", "U+2A80", "U+2A81", "U+2A82", "U+2A83", "U+2A84", 
"U+2A85", "U+2A85", "U+2A86", "U+2A86", "U+2A87", "U+2A87", "U+2A88", 
"U+2A88", "U+2A89", "U+2A89", "U+2A8A", "U+2A8A", "U+2A8B", "U+2A8B", 
"U+2A8C", "U+2A8C", "U+2A8D", "U+2A8E", "U+2A8F", "U+2A90", "U+2A91", 
"U+2A92", "U+2A93", "U+2A94", "U+2A95", "U+2A95", "U+2A96", "U+2A96", 
"U+2A97", "U+2A98", "U+2A99", "U+2A9A", "U+2A9D", "U+2A9E", "U+2A9F", 
"U+2AA0", "U+2AA1", "U+2AA1", "U+0338", "U+2AA2", "U+2AA2", "U+0338", 
"U+2AA4", "U+2AA5", "U+2AA6", "U+2AA7", "U+2AA8", "U+2AA9", "U+2AAA", 
"U+2AAB", "U+2AAC", "U+2AAC", "U+FE00", "U+2AAD", "U+2AAD", "U+FE00", 
"U+2AAE", "U+2AAF", "U+2AAF", "U+2AAF", "U+2AAF", "U+2AAF", "U+2AAF", 
"U+0338", "U+0338", "U+0338", "U+2AB0", "U+2AB0", "U+2AB0", "U+2AB0", 
"U+2AB0", "U+2AB0", "U+0338", "U+0338", "U+0338", "U+2AB3", "U+2AB4", 
"U+2AB5", "U+2AB5", "U+2AB6", "U+2AB6", "U+2AB7", "U+2AB7", "U+2AB8", 
"U+2AB8", "U+2AB9", "U+2AB9", "U+2ABA", "U+2ABA", "U+2ABB", "U+2ABC", 
"U+2ABD", "U+2ABE", "U+2ABF", "U+2AC0", "U+2AC1", "U+2AC2", "U+2AC3", 
"U+2AC4", "U+2AC5", "U+2AC5", "U+2AC5", "U+2AC5", "U+0338", "U+0338", 
"U+2AC6", "U+2AC6", "U+2AC6", "U+2AC6", "U+0338", "U+0338", "U+2AC7", 
"U+2AC8", "U+2ACB", "U+2ACB", "U+2ACB", "U+2ACB", "U+FE00", "U+FE00", 
"U+2ACC", "U+2ACC", "U+2ACC", "U+2ACC", "U+FE00", "U+FE00", "U+2ACF", 
"U+2AD0", "U+2AD1", "U+2AD2", "U+2AD3", "U+2AD4", "U+2AD5", "U+2AD6", 
"U+2AD7", "U+2AD8", "U+2AD9", "U+2ADA", "U+2ADB", "U+2AE4", "U+2AE4", 
"U+2AE6", "U+2AE7", "U+2AE8", "U+2AE9", "U+2AEB", "U+2AEC", "U+2AED", 
"U+2AEE", "U+2AEF", "U+2AF0", "U+2AF1", "U+2AF2", "U+2AF3", "U+2AFD", 
"U+2AFD", "U+20E5", "U+FB00", "U+FB01", "U+FB02", "U+FB03", "U+FB04", 
"U+1D49C", "U+1D49E", "U+1D49F", "U+1D4A2", "U+1D4A5", "U+1D4A6", 
"U+1D4A9", "U+1D4AA", "U+1D4AB", "U+1D4AC", "U+1D4AE", "U+1D4AF", 
"U+1D4B0", "U+1D4B1", "U+1D4B2", "U+1D4B3", "U+1D4B4", "U+1D4B5", 
"U+1D4B6", "U+1D4B7", "U+1D4B8", "U+1D4B9", "U+1D4BB", "U+1D4BD", 
"U+1D4BE", "U+1D4BF", "U+1D4C0", "U+1D4C1", "U+1D4C2", "U+1D4C3", 
"U+1D4C5", "U+1D4C6", "U+1D4C7", "U+1D4C8", "U+1D4C9", "U+1D4CA", 
"U+1D4CB", "U+1D4CC", "U+1D4CD", "U+1D4CE", "U+1D4CF", "U+1D504", 
"U+1D505", "U+1D507", "U+1D508", "U+1D509", "U+1D50A", "U+1D50D", 
"U+1D50E", "U+1D50F", "U+1D510", "U+1D511", "U+1D512", "U+1D513", 
"U+1D514", "U+1D516", "U+1D517", "U+1D518", "U+1D519", "U+1D51A", 
"U+1D51B", "U+1D51C", "U+1D51E", "U+1D51F", "U+1D520", "U+1D521", 
"U+1D522", "U+1D523", "U+1D524", "U+1D525", "U+1D526", "U+1D527", 
"U+1D528", "U+1D529", "U+1D52A", "U+1D52B", "U+1D52C", "U+1D52D", 
"U+1D52E", "U+1D52F", "U+1D530", "U+1D531", "U+1D532", "U+1D533", 
"U+1D534", "U+1D535", "U+1D536", "U+1D537", "U+1D538", "U+1D539", 
"U+1D53B", "U+1D53C", "U+1D53D", "U+1D53E", "U+1D540", "U+1D541", 
"U+1D542", "U+1D543", "U+1D544", "U+1D546", "U+1D54A", "U+1D54B", 
"U+1D54C", "U+1D54D", "U+1D54E", "U+1D54F", "U+1D550", "U+1D552", 
"U+1D553", "U+1D554", "U+1D555", "U+1D556", "U+1D557", "U+1D558", 
"U+1D559", "U+1D55A", "U+1D55B", "U+1D55C", "U+1D55D", "U+1D55E", 
"U+1D55F", "U+1D560", "U+1D561", "U+1D562", "U+1D563", "U+1D564", 
"U+1D565", "U+1D566", "U+1D567", "U+1D568", "U+1D569", "U+1D56A", 
"U+1D56B", "U+1D6AA", "U+1D6AB", "U+1D6AF", "U+1D6B2", "U+1D6B5", 
"U+1D6B7", "U+1D6BA", "U+1D6BC", "U+1D6BD", "U+1D6BF", "U+1D6C0", 
"U+1D6C2", "U+1D6C3", "U+1D6C4", "U+1D6C5", "U+1D6C6", "U+1D6C7", 
"U+1D6C8", "U+1D6C9", "U+1D6CA", "U+1D6CB", "U+1D6CC", "U+1D6CD", 
"U+1D6CE", "U+1D6CF", "U+1D6D1", "U+1D6D2", "U+1D6D3", "U+1D6D4", 
"U+1D6D5", "U+1D6D6", "U+1D6D7", "U+1D6D8", "U+1D6D9", "U+1D6DA", 
"U+1D6DC", "U+1D6DD", "U+1D6DE", "U+1D6DF", "U+1D6E0", "U+1D6E1", 
"U+1D7CA", "U+1D7CB"), name = c("&Tab;", "&NewLine;", "&excl;", 
"&quot;", "&QUOT;", "&num;", "&dollar;", "&percnt;", "&amp;", 
"&AMP;", "&apos;", "&lpar;", "&rpar;", "&ast;", "&midast;", "&plus;", 
"&comma;", "&period;", "&sol;", "&colon;", "&semi;", "&lt;", 
"&LT;", "&nvlt;", "&nvlt;", "&equals;", "&bne;", "&bne;", "&gt;", 
"&GT;", "&nvgt;", "&nvgt;", "&quest;", "&commat;", "&lsqb;", 
"&lbrack;", "&bsol;", "&rsqb;", "&rbrack;", "&Hat;", "&lowbar;", 
"&UnderBar;", "&grave;", "&DiacriticalGrave;", "&fjlig;", "&fjlig;", 
"&lcub;", "&lbrace;", "&verbar;", "&vert;", "&VerticalLine;", 
"&rcub;", "&rbrace;", "&nbsp;", "&NonBreakingSpace;", "&iexcl;", 
"&cent;", "&pound;", "&curren;", "&yen;", "&brvbar;", "&sect;", 
"&Dot;", "&die;", "&DoubleDot;", "&uml;", "&copy;", "&COPY;", 
"&ordf;", "&laquo;", "&not;", "&shy;", "&reg;", "&circledR;", 
"&REG;", "&macr;", "&strns;", "&deg;", "&plusmn;", "&pm;", "&PlusMinus;", 
"&sup2;", "&sup3;", "&acute;", "&DiacriticalAcute;", "&micro;", 
"&para;", "&middot;", "&centerdot;", "&CenterDot;", "&cedil;", 
"&Cedilla;", "&sup1;", "&ordm;", "&raquo;", "&frac14;", "&frac12;", 
"&half;", "&frac34;", "&iquest;", "&Agrave;", "&Aacute;", "&Acirc;", 
"&Atilde;", "&Auml;", "&Aring;", "&angst;", "&AElig;", "&Ccedil;", 
"&Egrave;", "&Eacute;", "&Ecirc;", "&Euml;", "&Igrave;", "&Iacute;", 
"&Icirc;", "&Iuml;", "&ETH;", "&Ntilde;", "&Ograve;", "&Oacute;", 
"&Ocirc;", "&Otilde;", "&Ouml;", "&times;", "&Oslash;", "&Ugrave;", 
"&Uacute;", "&Ucirc;", "&Uuml;", "&Yacute;", "&THORN;", "&szlig;", 
"&agrave;", "&aacute;", "&acirc;", "&atilde;", "&auml;", "&aring;", 
"&aelig;", "&ccedil;", "&egrave;", "&eacute;", "&ecirc;", "&euml;", 
"&igrave;", "&iacute;", "&icirc;", "&iuml;", "&eth;", "&ntilde;", 
"&ograve;", "&oacute;", "&ocirc;", "&otilde;", "&ouml;", "&divide;", 
"&div;", "&oslash;", "&ugrave;", "&uacute;", "&ucirc;", "&uuml;", 
"&yacute;", "&thorn;", "&yuml;", "&Amacr;", "&amacr;", "&Abreve;", 
"&abreve;", "&Aogon;", "&aogon;", "&Cacute;", "&cacute;", "&Ccirc;", 
"&ccirc;", "&Cdot;", "&cdot;", "&Ccaron;", "&ccaron;", "&Dcaron;", 
"&dcaron;", "&Dstrok;", "&dstrok;", "&Emacr;", "&emacr;", "&Edot;", 
"&edot;", "&Eogon;", "&eogon;", "&Ecaron;", "&ecaron;", "&Gcirc;", 
"&gcirc;", "&Gbreve;", "&gbreve;", "&Gdot;", "&gdot;", "&Gcedil;", 
"&Hcirc;", "&hcirc;", "&Hstrok;", "&hstrok;", "&Itilde;", "&itilde;", 
"&Imacr;", "&imacr;", "&Iogon;", "&iogon;", "&Idot;", "&imath;", 
"&inodot;", "&IJlig;", "&ijlig;", "&Jcirc;", "&jcirc;", "&Kcedil;", 
"&kcedil;", "&kgreen;", "&Lacute;", "&lacute;", "&Lcedil;", "&lcedil;", 
"&Lcaron;", "&lcaron;", "&Lmidot;", "&lmidot;", "&Lstrok;", "&lstrok;", 
"&Nacute;", "&nacute;", "&Ncedil;", "&ncedil;", "&Ncaron;", "&ncaron;", 
"&napos;", "&ENG;", "&eng;", "&Omacr;", "&omacr;", "&Odblac;", 
"&odblac;", "&OElig;", "&oelig;", "&Racute;", "&racute;", "&Rcedil;", 
"&rcedil;", "&Rcaron;", "&rcaron;", "&Sacute;", "&sacute;", "&Scirc;", 
"&scirc;", "&Scedil;", "&scedil;", "&Scaron;", "&scaron;", "&Tcedil;", 
"&tcedil;", "&Tcaron;", "&tcaron;", "&Tstrok;", "&tstrok;", "&Utilde;", 
"&utilde;", "&Umacr;", "&umacr;", "&Ubreve;", "&ubreve;", "&Uring;", 
"&uring;", "&Udblac;", "&udblac;", "&Uogon;", "&uogon;", "&Wcirc;", 
"&wcirc;", "&Ycirc;", "&ycirc;", "&Yuml;", "&Zacute;", "&zacute;", 
"&Zdot;", "&zdot;", "&Zcaron;", "&zcaron;", "&fnof;", "&imped;", 
"&gacute;", "&jmath;", "&circ;", "&caron;", "&Hacek;", "&breve;", 
"&Breve;", "&dot;", "&DiacriticalDot;", "&ring;", "&ogon;", "&tilde;", 
"&DiacriticalTilde;", "&dblac;", "&DiacriticalDoubleAcute;", 
"&DownBreve;", "&Aacgr;", "&Eacgr;", "&EEacgr;", "&Iacgr;", "&Oacgr;", 
"&Uacgr;", "&OHacgr;", "&idiagr;", "&Alpha;", "&Agr;", "&Bgr;", 
"&Beta;", "&Gamma;", "&Ggr;", "&Delta;", "&Dgr;", "&Egr;", "&Epsilon;", 
"&Zgr;", "&Zeta;", "&EEgr;", "&Eta;", "&THgr;", "&Theta;", "&Igr;", 
"&Iota;", "&Kgr;", "&Kappa;", "&Lambda;", "&Lgr;", "&Mgr;", "&Mu;", 
"&Ngr;", "&Nu;", "&Xgr;", "&Xi;", "&Ogr;", "&Omicron;", "&Pgr;", 
"&Pi;", "&Rgr;", "&Rho;", "&Sgr;", "&Sigma;", "&Tgr;", "&Tau;", 
"&Ugr;", "&Upsilon;", "&PHgr;", "&Phi;", "&KHgr;", "&Chi;", "&PSgr;", 
"&Psi;", "&OHgr;", "&Omega;", "&ohm;", "&Idigr;", "&Udigr;", 
"&aacgr;", "&eacgr;", "&eeacgr;", "&iacgr;", "&udiagr;", "&agr;", 
"&alpha;", "&beta;", "&bgr;", "&gamma;", "&ggr;", "&delta;", 
"&dgr;", "&egr;", "&epsi;", "&epsilon;", "&zeta;", "&zgr;", "&eegr;", 
"&eta;", "&theta;", "&thgr;", "&igr;", "&iota;", "&kappa;", "&kgr;", 
"&lambda;", "&lgr;", "&mgr;", "&mu;", "&ngr;", "&nu;", "&xgr;", 
"&xi;", "&omicron;", "&ogr;", "&pgr;", "&pi;", "&rgr;", "&rho;", 
"&sfgr;", "&sigmav;", "&varsigma;", "&sigmaf;", "&sgr;", "&sigma;", 
"&tau;", "&tgr;", "&ugr;", "&upsi;", "&upsilon;", "&phi;", "&phgr;", 
"&chi;", "&khgr;", "&psgr;", "&psi;", "&ohgr;", "&omega;", "&idigr;", 
"&udigr;", "&oacgr;", "&uacgr;", "&ohacgr;", "&thetav;", "&vartheta;", 
"&thetasym;", "&Upsi;", "&upsih;", "&straightphi;", "&phiv;", 
"&varphi;", "&piv;", "&varpi;", "&Gammad;", "&gammad;", "&digamma;", 
"&kappav;", "&varkappa;", "&rhov;", "&varrho;", "&epsiv;", "&straightepsilon;", 
"&varepsilon;", "&bepsi;", "&backepsilon;", "&IOcy;", "&DJcy;", 
"&GJcy;", "&Jukcy;", "&DScy;", "&Iukcy;", "&YIcy;", "&Jsercy;", 
"&LJcy;", "&NJcy;", "&TSHcy;", "&KJcy;", "&Ubrcy;", "&DZcy;", 
"&Acy;", "&Bcy;", "&Vcy;", "&Gcy;", "&Dcy;", "&IEcy;", "&ZHcy;", 
"&Zcy;", "&Icy;", "&Jcy;", "&Kcy;", "&Lcy;", "&Mcy;", "&Ncy;", 
"&Ocy;", "&Pcy;", "&Rcy;", "&Scy;", "&Tcy;", "&Ucy;", "&Fcy;", 
"&KHcy;", "&TScy;", "&CHcy;", "&SHcy;", "&SHCHcy;", "&HARDcy;", 
"&Ycy;", "&SOFTcy;", "&Ecy;", "&YUcy;", "&YAcy;", "&acy;", "&bcy;", 
"&vcy;", "&gcy;", "&dcy;", "&iecy;", "&zhcy;", "&zcy;", "&icy;", 
"&jcy;", "&kcy;", "&lcy;", "&mcy;", "&ncy;", "&ocy;", "&pcy;", 
"&rcy;", "&scy;", "&tcy;", "&ucy;", "&fcy;", "&khcy;", "&tscy;", 
"&chcy;", "&shcy;", "&shchcy;", "&hardcy;", "&ycy;", "&softcy;", 
"&ecy;", "&yucy;", "&yacy;", "&iocy;", "&djcy;", "&gjcy;", "&jukcy;", 
"&dscy;", "&iukcy;", "&yicy;", "&jsercy;", "&ljcy;", "&njcy;", 
"&tshcy;", "&kjcy;", "&ubrcy;", "&dzcy;", "&ensp;", "&emsp;", 
"&emsp13;", "&emsp14;", "&numsp;", "&puncsp;", "&thinsp;", "&ThinSpace;", 
"&hairsp;", "&VeryThinSpace;", "&ZeroWidthSpace;", "&NegativeVeryThinSpace;", 
"&NegativeThinSpace;", "&NegativeMediumSpace;", "&NegativeThickSpace;", 
"&zwnj;", "&zwj;", "&lrm;", "&rlm;", "&hyphen;", "&dash;", "&ndash;", 
"&mdash;", "&horbar;", "&Verbar;", "&Vert;", "&lsquo;", "&OpenCurlyQuote;", 
"&rsquo;", "&rsquor;", "&CloseCurlyQuote;", "&lsquor;", "&sbquo;", 
"&ldquo;", "&OpenCurlyDoubleQuote;", "&rdquo;", "&rdquor;", "&CloseCurlyDoubleQuote;", 
"&ldquor;", "&bdquo;", "&dagger;", "&Dagger;", "&ddagger;", "&bull;", 
"&bullet;", "&nldr;", "&hellip;", "&mldr;", "&permil;", "&pertenk;", 
"&prime;", "&Prime;", "&tprime;", "&bprime;", "&backprime;", 
"&lsaquo;", "&rsaquo;", "&oline;", "&OverBar;", "&caret;", "&hybull;", 
"&frasl;", "&bsemi;", "&qprime;", "&MediumSpace;", "&ThickSpace;", 
"&ThickSpace;", "&NoBreak;", "&ApplyFunction;", "&af;", "&InvisibleTimes;", 
"&it;", "&InvisibleComma;", "&ic;", "&euro;", "&tdot;", "&TripleDot;", 
"&DotDot;", "&Copf;", "&complexes;", "&incare;", "&gscr;", "&hamilt;", 
"&HilbertSpace;", "&Hscr;", "&Hfr;", "&Poincareplane;", "&quaternions;", 
"&Hopf;", "&planckh;", "&planck;", "&hbar;", "&plankv;", "&hslash;", 
"&Iscr;", "&imagline;", "&image;", "&Im;", "&imagpart;", "&Ifr;", 
"&Lscr;", "&lagran;", "&Laplacetrf;", "&ell;", "&Nopf;", "&naturals;", 
"&numero;", "&copysr;", "&weierp;", "&wp;", "&Popf;", "&primes;", 
"&rationals;", "&Qopf;", "&Rscr;", "&realine;", "&real;", "&Re;", 
"&realpart;", "&Rfr;", "&reals;", "&Ropf;", "&rx;", "&trade;", 
"&TRADE;", "&integers;", "&Zopf;", "&mho;", "&Zfr;", "&zeetrf;", 
"&iiota;", "&bernou;", "&Bernoullis;", "&Bscr;", "&Cfr;", "&Cayleys;", 
"&escr;", "&Escr;", "&expectation;", "&Fscr;", "&Fouriertrf;", 
"&phmmat;", "&Mellintrf;", "&Mscr;", "&order;", "&orderof;", 
"&oscr;", "&alefsym;", "&aleph;", "&beth;", "&gimel;", "&daleth;", 
"&CapitalDifferentialD;", "&DD;", "&DifferentialD;", "&dd;", 
"&ExponentialE;", "&exponentiale;", "&ee;", "&ImaginaryI;", "&ii;", 
"&frac13;", "&frac23;", "&frac15;", "&frac25;", "&frac35;", "&frac45;", 
"&frac16;", "&frac56;", "&frac18;", "&frac38;", "&frac58;", "&frac78;", 
"&larr;", "&leftarrow;", "&LeftArrow;", "&slarr;", "&ShortLeftArrow;", 
"&uarr;", "&uparrow;", "&UpArrow;", "&ShortUpArrow;", "&rarr;", 
"&rightarrow;", "&RightArrow;", "&srarr;", "&ShortRightArrow;", 
"&darr;", "&downarrow;", "&DownArrow;", "&ShortDownArrow;", "&harr;", 
"&leftrightarrow;", "&LeftRightArrow;", "&varr;", "&updownarrow;", 
"&UpDownArrow;", "&nwarr;", "&UpperLeftArrow;", "&nwarrow;", 
"&nearr;", "&UpperRightArrow;", "&nearrow;", "&searr;", "&searrow;", 
"&LowerRightArrow;", "&swarr;", "&swarrow;", "&LowerLeftArrow;", 
"&nlarr;", "&nleftarrow;", "&nrarr;", "&nrightarrow;", "&rarrw;", 
"&rightsquigarrow;", "&nrarrw;", "&nrarrw;", "&Larr;", "&twoheadleftarrow;", 
"&Uarr;", "&Rarr;", "&twoheadrightarrow;", "&Darr;", "&larrtl;", 
"&leftarrowtail;", "&rarrtl;", "&rightarrowtail;", "&LeftTeeArrow;", 
"&mapstoleft;", "&UpTeeArrow;", "&mapstoup;", "&map;", "&RightTeeArrow;", 
"&mapsto;", "&DownTeeArrow;", "&mapstodown;", "&larrhk;", "&hookleftarrow;", 
"&rarrhk;", "&hookrightarrow;", "&larrlp;", "&looparrowleft;", 
"&rarrlp;", "&looparrowright;", "&harrw;", "&leftrightsquigarrow;", 
"&nharr;", "&nleftrightarrow;", "&lsh;", "&Lsh;", "&rsh;", "&Rsh;", 
"&ldsh;", "&rdsh;", "&crarr;", "&cularr;", "&curvearrowleft;", 
"&curarr;", "&curvearrowright;", "&olarr;", "&circlearrowleft;", 
"&orarr;", "&circlearrowright;", "&lharu;", "&LeftVector;", "&leftharpoonup;", 
"&lhard;", "&leftharpoondown;", "&DownLeftVector;", "&uharr;", 
"&upharpoonright;", "&RightUpVector;", "&uharl;", "&upharpoonleft;", 
"&LeftUpVector;", "&rharu;", "&RightVector;", "&rightharpoonup;", 
"&rhard;", "&rightharpoondown;", "&DownRightVector;", "&dharr;", 
"&RightDownVector;", "&downharpoonright;", "&dharl;", "&LeftDownVector;", 
"&downharpoonleft;", "&rlarr;", "&rightleftarrows;", "&RightArrowLeftArrow;", 
"&udarr;", "&UpArrowDownArrow;", "&lrarr;", "&leftrightarrows;", 
"&LeftArrowRightArrow;", "&llarr;", "&leftleftarrows;", "&uuarr;", 
"&upuparrows;", "&rrarr;", "&rightrightarrows;", "&ddarr;", "&downdownarrows;", 
"&lrhar;", "&ReverseEquilibrium;", "&leftrightharpoons;", "&rlhar;", 
"&rightleftharpoons;", "&Equilibrium;", "&nlArr;", "&nLeftarrow;", 
"&nhArr;", "&nLeftrightarrow;", "&nrArr;", "&nRightarrow;", "&lArr;", 
"&Leftarrow;", "&DoubleLeftArrow;", "&uArr;", "&Uparrow;", "&DoubleUpArrow;", 
"&rArr;", "&Rightarrow;", "&Implies;", "&DoubleRightArrow;", 
"&dArr;", "&Downarrow;", "&DoubleDownArrow;", "&hArr;", "&Leftrightarrow;", 
"&DoubleLeftRightArrow;", "&iff;", "&vArr;", "&Updownarrow;", 
"&DoubleUpDownArrow;", "&nwArr;", "&neArr;", "&seArr;", "&swArr;", 
"&lAarr;", "&Lleftarrow;", "&rAarr;", "&Rrightarrow;", "&zigrarr;", 
"&larrb;", "&LeftArrowBar;", "&rarrb;", "&RightArrowBar;", "&duarr;", 
"&DownArrowUpArrow;", "&loarr;", "&roarr;", "&hoarr;", "&forall;", 
"&ForAll;", "&comp;", "&complement;", "&part;", "&PartialD;", 
"&npart;", "&npart;", "&exist;", "&Exists;", "&nexist;", "&NotExists;", 
"&nexists;", "&empty;", "&emptyset;", "&emptyv;", "&varnothing;", 
"&nabla;", "&Del;", "&isin;", "&isinv;", "&Element;", "&in;", 
"&notin;", "&NotElement;", "&notinva;", "&niv;", "&ReverseElement;", 
"&ni;", "&SuchThat;", "&notni;", "&notniva;", "&NotReverseElement;", 
"&prod;", "&Product;", "&coprod;", "&Coproduct;", "&sum;", "&Sum;", 
"&minus;", "&mnplus;", "&mp;", "&MinusPlus;", "&plusdo;", "&dotplus;", 
"&setmn;", "&setminus;", "&Backslash;", "&ssetmn;", "&smallsetminus;", 
"&lowast;", "&compfn;", "&SmallCircle;", "&radic;", "&Sqrt;", 
"&prop;", "&propto;", "&Proportional;", "&vprop;", "&varpropto;", 
"&infin;", "&angrt;", "&ang;", "&angle;", "&nang;", "&nang;", 
"&angmsd;", "&measuredangle;", "&angsph;", "&mid;", "&VerticalBar;", 
"&smid;", "&shortmid;", "&nmid;", "&NotVerticalBar;", "&nsmid;", 
"&nshortmid;", "&par;", "&parallel;", "&DoubleVerticalBar;", 
"&spar;", "&shortparallel;", "&npar;", "&nparallel;", "&NotDoubleVerticalBar;", 
"&nspar;", "&nshortparallel;", "&and;", "&wedge;", "&or;", "&vee;", 
"&cap;", "&caps;", "&caps;", "&cup;", "&cups;", "&cups;", "&int;", 
"&Integral;", "&Int;", "&tint;", "&iiint;", "&conint;", "&oint;", 
"&ContourIntegral;", "&Conint;", "&DoubleContourIntegral;", "&Cconint;", 
"&cwint;", "&cwconint;", "&ClockwiseContourIntegral;", "&awconint;", 
"&CounterClockwiseContourIntegral;", "&there4;", "&therefore;", 
"&Therefore;", "&becaus;", "&because;", "&Because;", "&ratio;", 
"&Colon;", "&Proportion;", "&minusd;", "&dotminus;", "&mDDot;", 
"&homtht;", "&sim;", "&Tilde;", "&thksim;", "&thicksim;", "&nvsim;", 
"&nvsim;", "&bsim;", "&backsim;", "&race;", "&race;", "&ac;", 
"&mstpos;", "&acE;", "&acE;", "&acd;", "&wreath;", "&VerticalTilde;", 
"&wr;", "&nsim;", "&NotTilde;", "&esim;", "&EqualTilde;", "&eqsim;", 
"&NotEqualTilde;", "&nesim;", "&NotEqualTilde;", "&nesim;", "&sime;", 
"&TildeEqual;", "&simeq;", "&nsime;", "&nsimeq;", "&NotTildeEqual;", 
"&cong;", "&TildeFullEqual;", "&simne;", "&ncong;", "&NotTildeFullEqual;", 
"&asymp;", "&ap;", "&TildeTilde;", "&approx;", "&thkap;", "&thickapprox;", 
"&nap;", "&NotTildeTilde;", "&napprox;", "&ape;", "&approxeq;", 
"&apid;", "&napid;", "&napid;", "&bcong;", "&backcong;", "&asympeq;", 
"&CupCap;", "&nvap;", "&nvap;", "&bump;", "&HumpDownHump;", "&Bumpeq;", 
"&NotHumpDownHump;", "&nbump;", "&NotHumpDownHump;", "&nbump;", 
"&bumpe;", "&HumpEqual;", "&bumpeq;", "&nbumpe;", "&NotHumpEqual;", 
"&nbumpe;", "&NotHumpEqual;", "&esdot;", "&DotEqual;", "&doteq;", 
"&nedot;", "&nedot;", "&eDot;", "&doteqdot;", "&efDot;", "&fallingdotseq;", 
"&erDot;", "&risingdotseq;", "&colone;", "&coloneq;", "&Assign;", 
"&ecolon;", "&eqcolon;", "&ecir;", "&eqcirc;", "&cire;", "&circeq;", 
"&wedgeq;", "&veeeq;", "&trie;", "&triangleq;", "&equest;", "&questeq;", 
"&ne;", "&NotEqual;", "&equiv;", "&Congruent;", "&bnequiv;", 
"&bnequiv;", "&nequiv;", "&NotCongruent;", "&le;", "&leq;", "&nvle;", 
"&nvle;", "&ge;", "&GreaterEqual;", "&geq;", "&nvge;", "&nvge;", 
"&lE;", "&LessFullEqual;", "&leqq;", "&nlE;", "&nleqq;", "&nlE;", 
"&nleqq;", "&gE;", "&GreaterFullEqual;", "&geqq;", "&ngE;", "&ngeqq;", 
"&NotGreaterFullEqual;", "&ngE;", "&ngeqq;", "&NotGreaterFullEqual;", 
"&lnE;", "&lneqq;", "&lvnE;", "&lvertneqq;", "&lvnE;", "&lvertneqq;", 
"&gnE;", "&gneqq;", "&gvnE;", "&gvertneqq;", "&gvnE;", "&gvertneqq;", 
"&Lt;", "&NestedLessLess;", "&ll;", "&nLtv;", "&NotLessLess;", 
"&nLtv;", "&NotLessLess;", "&nLt;", "&nLt;", "&Gt;", "&NestedGreaterGreater;", 
"&gg;", "&nGtv;", "&NotGreaterGreater;", "&nGtv;", "&NotGreaterGreater;", 
"&nGt;", "&nGt;", "&twixt;", "&between;", "&NotCupCap;", "&nlt;", 
"&NotLess;", "&nless;", "&ngt;", "&NotGreater;", "&ngtr;", "&nle;", 
"&NotLessEqual;", "&nleq;", "&nge;", "&NotGreaterEqual;", "&ngeq;", 
"&lsim;", "&LessTilde;", "&lesssim;", "&gsim;", "&gtrsim;", "&GreaterTilde;", 
"&nlsim;", "&NotLessTilde;", "&ngsim;", "&NotGreaterTilde;", 
"&lg;", "&lessgtr;", "&LessGreater;", "&gl;", "&gtrless;", "&GreaterLess;", 
"&ntlg;", "&NotLessGreater;", "&ntgl;", "&NotGreaterLess;", "&pr;", 
"&Precedes;", "&prec;", "&sc;", "&Succeeds;", "&succ;", "&prcue;", 
"&PrecedesSlantEqual;", "&preccurlyeq;", "&sccue;", "&SucceedsSlantEqual;", 
"&succcurlyeq;", "&prsim;", "&precsim;", "&PrecedesTilde;", "&scsim;", 
"&succsim;", "&SucceedsTilde;", "&NotSucceedsTilde;", "&NotSucceedsTilde;", 
"&npr;", "&nprec;", "&NotPrecedes;", "&nsc;", "&nsucc;", "&NotSucceeds;", 
"&sub;", "&subset;", "&vnsub;", "&nsubset;", "&NotSubset;", "&vnsub;", 
"&nsubset;", "&NotSubset;", "&sup;", "&supset;", "&Superset;", 
"&vnsup;", "&nsupset;", "&NotSuperset;", "&vnsup;", "&nsupset;", 
"&NotSuperset;", "&nsub;", "&nsup;", "&sube;", "&SubsetEqual;", 
"&subseteq;", "&supe;", "&supseteq;", "&SupersetEqual;", "&nsube;", 
"&nsubseteq;", "&NotSubsetEqual;", "&nsupe;", "&nsupseteq;", 
"&NotSupersetEqual;", "&subne;", "&subsetneq;", "&vsubne;", "&varsubsetneq;", 
"&vsubne;", "&varsubsetneq;", "&supne;", "&supsetneq;", "&vsupne;", 
"&varsupsetneq;", "&vsupne;", "&varsupsetneq;", "&cupdot;", "&uplus;", 
"&UnionPlus;", "&sqsub;", "&SquareSubset;", "&sqsubset;", "&NotSquareSubset;", 
"&NotSquareSubset;", "&sqsup;", "&SquareSuperset;", "&sqsupset;", 
"&NotSquareSuperset;", "&NotSquareSuperset;", "&sqsube;", "&SquareSubsetEqual;", 
"&sqsubseteq;", "&sqsupe;", "&SquareSupersetEqual;", "&sqsupseteq;", 
"&sqcap;", "&SquareIntersection;", "&sqcaps;", "&sqcaps;", "&sqcup;", 
"&SquareUnion;", "&sqcups;", "&sqcups;", "&oplus;", "&CirclePlus;", 
"&ominus;", "&CircleMinus;", "&otimes;", "&CircleTimes;", "&osol;", 
"&odot;", "&CircleDot;", "&ocir;", "&circledcirc;", "&oast;", 
"&circledast;", "&odash;", "&circleddash;", "&plusb;", "&boxplus;", 
"&minusb;", "&boxminus;", "&timesb;", "&boxtimes;", "&sdotb;", 
"&dotsquare;", "&vdash;", "&RightTee;", "&dashv;", "&LeftTee;", 
"&top;", "&DownTee;", "&bottom;", "&bot;", "&perp;", "&UpTee;", 
"&models;", "&vDash;", "&DoubleRightTee;", "&Vdash;", "&Vvdash;", 
"&VDash;", "&nvdash;", "&nvDash;", "&nVdash;", "&nVDash;", "&prurel;", 
"&vltri;", "&vartriangleleft;", "&LeftTriangle;", "&vrtri;", 
"&vartriangleright;", "&RightTriangle;", "&ltrie;", "&trianglelefteq;", 
"&LeftTriangleEqual;", "&nvltrie;", "&nvltrie;", "&rtrie;", "&trianglerighteq;", 
"&RightTriangleEqual;", "&nvrtrie;", "&nvrtrie;", "&origof;", 
"&imof;", "&mumap;", "&multimap;", "&hercon;", "&intcal;", "&intercal;", 
"&veebar;", "&barvee;", "&angrtvb;", "&lrtri;", "&xwedge;", "&Wedge;", 
"&bigwedge;", "&xvee;", "&Vee;", "&bigvee;", "&xcap;", "&Intersection;", 
"&bigcap;", "&xcup;", "&Union;", "&bigcup;", "&diam;", "&diamond;", 
"&Diamond;", "&sdot;", "&sstarf;", "&Star;", "&divonx;", "&divideontimes;", 
"&bowtie;", "&ltimes;", "&rtimes;", "&lthree;", "&leftthreetimes;", 
"&rthree;", "&rightthreetimes;", "&bsime;", "&backsimeq;", "&cuvee;", 
"&curlyvee;", "&cuwed;", "&curlywedge;", "&Sub;", "&Subset;", 
"&Sup;", "&Supset;", "&Cap;", "&Cup;", "&fork;", "&pitchfork;", 
"&epar;", "&ltdot;", "&lessdot;", "&gtdot;", "&gtrdot;", "&Ll;", 
"&nLl;", "&nLl;", "&Gg;", "&ggg;", "&nGg;", "&nGg;", "&leg;", 
"&LessEqualGreater;", "&lesseqgtr;", "&lesg;", "&lesg;", "&gel;", 
"&gtreqless;", "&GreaterEqualLess;", "&gesl;", "&gesl;", "&cuepr;", 
"&curlyeqprec;", "&cuesc;", "&curlyeqsucc;", "&nprcue;", "&NotPrecedesSlantEqual;", 
"&nsccue;", "&NotSucceedsSlantEqual;", "&nsqsube;", "&NotSquareSubsetEqual;", 
"&nsqsupe;", "&NotSquareSupersetEqual;", "&lnsim;", "&gnsim;", 
"&prnsim;", "&precnsim;", "&scnsim;", "&succnsim;", "&nltri;", 
"&ntriangleleft;", "&NotLeftTriangle;", "&nrtri;", "&ntriangleright;", 
"&NotRightTriangle;", "&nltrie;", "&ntrianglelefteq;", "&NotLeftTriangleEqual;", 
"&nrtrie;", "&ntrianglerighteq;", "&NotRightTriangleEqual;", 
"&vellip;", "&ctdot;", "&utdot;", "&dtdot;", "&disin;", "&isinsv;", 
"&isins;", "&isindot;", "&notindot;", "&notindot;", "&notinvc;", 
"&notinvb;", "&isinE;", "&notinE;", "&notinE;", "&nisd;", "&xnis;", 
"&nis;", "&notnivc;", "&notnivb;", "&barwed;", "&barwedge;", 
"&Barwed;", "&doublebarwedge;", "&lceil;", "&LeftCeiling;", "&rceil;", 
"&RightCeiling;", "&lfloor;", "&LeftFloor;", "&rfloor;", "&RightFloor;", 
"&drcrop;", "&dlcrop;", "&urcrop;", "&ulcrop;", "&bnot;", "&profline;", 
"&profsurf;", "&telrec;", "&target;", "&ulcorn;", "&ulcorner;", 
"&urcorn;", "&urcorner;", "&dlcorn;", "&llcorner;", "&drcorn;", 
"&lrcorner;", "&frown;", "&sfrown;", "&smile;", "&ssmile;", "&cylcty;", 
"&profalar;", "&topbot;", "&ovbar;", "&solbar;", "&angzarr;", 
"&lmoust;", "&lmoustache;", "&rmoust;", "&rmoustache;", "&tbrk;", 
"&OverBracket;", "&bbrk;", "&UnderBracket;", "&bbrktbrk;", "&OverParenthesis;", 
"&UnderParenthesis;", "&OverBrace;", "&UnderBrace;", "&trpezium;", 
"&elinters;", "&blank;", "&oS;", "&circledS;", "&boxh;", "&HorizontalLine;", 
"&boxv;", "&boxdr;", "&boxdl;", "&boxur;", "&boxul;", "&boxvr;", 
"&boxvl;", "&boxhd;", "&boxhu;", "&boxvh;", "&boxH;", "&boxV;", 
"&boxdR;", "&boxDr;", "&boxDR;", "&boxdL;", "&boxDl;", "&boxDL;", 
"&boxuR;", "&boxUr;", "&boxUR;", "&boxuL;", "&boxUl;", "&boxUL;", 
"&boxvR;", "&boxVr;", "&boxVR;", "&boxvL;", "&boxVl;", "&boxVL;", 
"&boxHd;", "&boxhD;", "&boxHD;", "&boxHu;", "&boxhU;", "&boxHU;", 
"&boxvH;", "&boxVh;", "&boxVH;", "&uhblk;", "&lhblk;", "&block;", 
"&blk14;", "&blk12;", "&blk34;", "&squ;", "&square;", "&Square;", 
"&squf;", "&squarf;", "&blacksquare;", "&FilledVerySmallSquare;", 
"&EmptyVerySmallSquare;", "&rect;", "&marker;", "&fltns;", "&xutri;", 
"&bigtriangleup;", "&utrif;", "&blacktriangle;", "&utri;", "&triangle;", 
"&rtrif;", "&blacktriangleright;", "&rtri;", "&triangleright;", 
"&xdtri;", "&bigtriangledown;", "&dtrif;", "&blacktriangledown;", 
"&dtri;", "&triangledown;", "&ltrif;", "&blacktriangleleft;", 
"&ltri;", "&triangleleft;", "&loz;", "&lozenge;", "&cir;", "&tridot;", 
"&xcirc;", "&bigcirc;", "&ultri;", "&urtri;", "&lltri;", "&EmptySmallSquare;", 
"&FilledSmallSquare;", "&starf;", "&bigstar;", "&star;", "&phone;", 
"&female;", "&male;", "&spades;", "&spadesuit;", "&clubs;", "&clubsuit;", 
"&hearts;", "&heartsuit;", "&diams;", "&diamondsuit;", "&sung;", 
"&flat;", "&natur;", "&natural;", "&sharp;", "&check;", "&checkmark;", 
"&cross;", "&malt;", "&maltese;", "&sext;", "&VerticalSeparator;", 
"&lbbrk;", "&rbbrk;", "&bsolhsub;", "&suphsol;", "&lobrk;", "&LeftDoubleBracket;", 
"&robrk;", "&RightDoubleBracket;", "&lang;", "&LeftAngleBracket;", 
"&langle;", "&rang;", "&RightAngleBracket;", "&rangle;", "&Lang;", 
"&Rang;", "&loang;", "&roang;", "&xlarr;", "&longleftarrow;", 
"&LongLeftArrow;", "&xrarr;", "&longrightarrow;", "&LongRightArrow;", 
"&xharr;", "&longleftrightarrow;", "&LongLeftRightArrow;", "&xlArr;", 
"&Longleftarrow;", "&DoubleLongLeftArrow;", "&xrArr;", "&Longrightarrow;", 
"&DoubleLongRightArrow;", "&xhArr;", "&Longleftrightarrow;", 
"&DoubleLongLeftRightArrow;", "&xmap;", "&longmapsto;", "&dzigrarr;", 
"&nvlArr;", "&nvrArr;", "&nvHarr;", "&Map;", "&lbarr;", "&rbarr;", 
"&bkarow;", "&lBarr;", "&rBarr;", "&dbkarow;", "&RBarr;", "&drbkarow;", 
"&DDotrahd;", "&UpArrowBar;", "&DownArrowBar;", "&Rarrtl;", "&latail;", 
"&ratail;", "&lAtail;", "&rAtail;", "&larrfs;", "&rarrfs;", "&larrbfs;", 
"&rarrbfs;", "&nwarhk;", "&nearhk;", "&searhk;", "&hksearow;", 
"&swarhk;", "&hkswarow;", "&nwnear;", "&nesear;", "&toea;", "&seswar;", 
"&tosa;", "&swnwar;", "&rarrc;", "&nrarrc;", "&nrarrc;", "&cudarrr;", 
"&ldca;", "&rdca;", "&cudarrl;", "&larrpl;", "&curarrm;", "&cularrp;", 
"&rarrpl;", "&harrcir;", "&Uarrocir;", "&lurdshar;", "&ldrushar;", 
"&LeftRightVector;", "&RightUpDownVector;", "&DownLeftRightVector;", 
"&LeftUpDownVector;", "&LeftVectorBar;", "&RightVectorBar;", 
"&RightUpVectorBar;", "&RightDownVectorBar;", "&DownLeftVectorBar;", 
"&DownRightVectorBar;", "&LeftUpVectorBar;", "&LeftDownVectorBar;", 
"&LeftTeeVector;", "&RightTeeVector;", "&RightUpTeeVector;", 
"&RightDownTeeVector;", "&DownLeftTeeVector;", "&DownRightTeeVector;", 
"&LeftUpTeeVector;", "&LeftDownTeeVector;", "&lHar;", "&uHar;", 
"&rHar;", "&dHar;", "&luruhar;", "&ldrdhar;", "&ruluhar;", "&rdldhar;", 
"&lharul;", "&llhard;", "&rharul;", "&lrhard;", "&udhar;", "&UpEquilibrium;", 
"&duhar;", "&ReverseUpEquilibrium;", "&RoundImplies;", "&erarr;", 
"&simrarr;", "&larrsim;", "&rarrsim;", "&rarrap;", "&ltlarr;", 
"&gtrarr;", "&subrarr;", "&suplarr;", "&lfisht;", "&rfisht;", 
"&ufisht;", "&dfisht;", "&lopar;", "&ropar;", "&lbrke;", "&rbrke;", 
"&lbrkslu;", "&rbrksld;", "&lbrksld;", "&rbrkslu;", "&langd;", 
"&rangd;", "&lparlt;", "&rpargt;", "&gtlPar;", "&ltrPar;", "&vzigzag;", 
"&vangrt;", "&angrtvbd;", "&ange;", "&range;", "&dwangle;", "&uwangle;", 
"&angmsdaa;", "&angmsdab;", "&angmsdac;", "&angmsdad;", "&angmsdae;", 
"&angmsdaf;", "&angmsdag;", "&angmsdah;", "&bemptyv;", "&demptyv;", 
"&cemptyv;", "&raemptyv;", "&laemptyv;", "&ohbar;", "&omid;", 
"&opar;", "&operp;", "&olcross;", "&odsold;", "&olcir;", "&ofcir;", 
"&olt;", "&ogt;", "&cirscir;", "&cirE;", "&solb;", "&bsolb;", 
"&boxbox;", "&trisb;", "&rtriltri;", "&LeftTriangleBar;", "&NotLeftTriangleBar;", 
"&NotLeftTriangleBar;", "&RightTriangleBar;", "&NotRightTriangleBar;", 
"&NotRightTriangleBar;", "&iinfin;", "&infintie;", "&nvinfin;", 
"&eparsl;", "&smeparsl;", "&eqvparsl;", "&lozf;", "&blacklozenge;", 
"&RuleDelayed;", "&dsol;", "&xodot;", "&bigodot;", "&xoplus;", 
"&bigoplus;", "&xotime;", "&bigotimes;", "&xuplus;", "&biguplus;", 
"&xsqcup;", "&bigsqcup;", "&qint;", "&iiiint;", "&fpartint;", 
"&cirfnint;", "&awint;", "&rppolint;", "&scpolint;", "&npolint;", 
"&pointint;", "&quatint;", "&intlarhk;", "&pluscir;", "&plusacir;", 
"&simplus;", "&plusdu;", "&plussim;", "&plustwo;", "&mcomma;", 
"&minusdu;", "&loplus;", "&roplus;", "&Cross;", "&timesd;", "&timesbar;", 
"&smashp;", "&lotimes;", "&rotimes;", "&otimesas;", "&Otimes;", 
"&odiv;", "&triplus;", "&triminus;", "&tritime;", "&iprod;", 
"&intprod;", "&amalg;", "&capdot;", "&ncup;", "&ncap;", "&capand;", 
"&cupor;", "&cupcap;", "&capcup;", "&cupbrcap;", "&capbrcup;", 
"&cupcup;", "&capcap;", "&ccups;", "&ccaps;", "&ccupssm;", "&And;", 
"&Or;", "&andand;", "&oror;", "&orslope;", "&andslope;", "&andv;", 
"&orv;", "&andd;", "&ord;", "&wedbar;", "&sdote;", "&simdot;", 
"&congdot;", "&ncongdot;", "&ncongdot;", "&easter;", "&apacir;", 
"&apE;", "&napE;", "&napE;", "&eplus;", "&pluse;", "&Esim;", 
"&Colone;", "&Equal;", "&eDDot;", "&ddotseq;", "&equivDD;", "&ltcir;", 
"&gtcir;", "&ltquest;", "&gtquest;", "&les;", "&LessSlantEqual;", 
"&leqslant;", "&nles;", "&NotLessSlantEqual;", "&nleqslant;", 
"&nles;", "&NotLessSlantEqual;", "&nleqslant;", "&ges;", "&GreaterSlantEqual;", 
"&geqslant;", "&nges;", "&NotGreaterSlantEqual;", "&ngeqslant;", 
"&nges;", "&NotGreaterSlantEqual;", "&ngeqslant;", "&lesdot;", 
"&gesdot;", "&lesdoto;", "&gesdoto;", "&lesdotor;", "&gesdotol;", 
"&lap;", "&lessapprox;", "&gap;", "&gtrapprox;", "&lne;", "&lneq;", 
"&gne;", "&gneq;", "&lnap;", "&lnapprox;", "&gnap;", "&gnapprox;", 
"&lEg;", "&lesseqqgtr;", "&gEl;", "&gtreqqless;", "&lsime;", 
"&gsime;", "&lsimg;", "&gsiml;", "&lgE;", "&glE;", "&lesges;", 
"&gesles;", "&els;", "&eqslantless;", "&egs;", "&eqslantgtr;", 
"&elsdot;", "&egsdot;", "&el;", "&eg;", "&siml;", "&simg;", "&simlE;", 
"&simgE;", "&LessLess;", "&NotNestedLessLess;", "&NotNestedLessLess;", 
"&GreaterGreater;", "&NotNestedGreaterGreater;", "&NotNestedGreaterGreater;", 
"&glj;", "&gla;", "&ltcc;", "&gtcc;", "&lescc;", "&gescc;", "&smt;", 
"&lat;", "&smte;", "&smtes;", "&smtes;", "&late;", "&lates;", 
"&lates;", "&bumpE;", "&pre;", "&preceq;", "&PrecedesEqual;", 
"&npre;", "&npreceq;", "&NotPrecedesEqual;", "&npre;", "&npreceq;", 
"&NotPrecedesEqual;", "&sce;", "&succeq;", "&SucceedsEqual;", 
"&nsce;", "&nsucceq;", "&NotSucceedsEqual;", "&nsce;", "&nsucceq;", 
"&NotSucceedsEqual;", "&prE;", "&scE;", "&prnE;", "&precneqq;", 
"&scnE;", "&succneqq;", "&prap;", "&precapprox;", "&scap;", "&succapprox;", 
"&prnap;", "&precnapprox;", "&scnap;", "&succnapprox;", "&Pr;", 
"&Sc;", "&subdot;", "&supdot;", "&subplus;", "&supplus;", "&submult;", 
"&supmult;", "&subedot;", "&supedot;", "&subE;", "&subseteqq;", 
"&nsubE;", "&nsubseteqq;", "&nsubE;", "&nsubseteqq;", "&supE;", 
"&supseteqq;", "&nsupE;", "&nsupseteqq;", "&nsupE;", "&nsupseteqq;", 
"&subsim;", "&supsim;", "&subnE;", "&subsetneqq;", "&vsubnE;", 
"&varsubsetneqq;", "&vsubnE;", "&varsubsetneqq;", "&supnE;", 
"&supsetneqq;", "&vsupnE;", "&varsupsetneqq;", "&vsupnE;", "&varsupsetneqq;", 
"&csub;", "&csup;", "&csube;", "&csupe;", "&subsup;", "&supsub;", 
"&subsub;", "&supsup;", "&suphsub;", "&supdsub;", "&forkv;", 
"&topfork;", "&mlcp;", "&Dashv;", "&DoubleLeftTee;", "&Vdashl;", 
"&Barv;", "&vBar;", "&vBarv;", "&Vbar;", "&Not;", "&bNot;", "&rnmid;", 
"&cirmid;", "&midcir;", "&topcir;", "&nhpar;", "&parsim;", "&parsl;", 
"&nparsl;", "&nparsl;", "&fflig;", "&filig;", "&fllig;", "&ffilig;", 
"&ffllig;", "&Ascr;", "&Cscr;", "&Dscr;", "&Gscr;", "&Jscr;", 
"&Kscr;", "&Nscr;", "&Oscr;", "&Pscr;", "&Qscr;", "&Sscr;", "&Tscr;", 
"&Uscr;", "&Vscr;", "&Wscr;", "&Xscr;", "&Yscr;", "&Zscr;", "&ascr;", 
"&bscr;", "&cscr;", "&dscr;", "&fscr;", "&hscr;", "&iscr;", "&jscr;", 
"&kscr;", "&lscr;", "&mscr;", "&nscr;", "&pscr;", "&qscr;", "&rscr;", 
"&sscr;", "&tscr;", "&uscr;", "&vscr;", "&wscr;", "&xscr;", "&yscr;", 
"&zscr;", "&Afr;", "&Bfr;", "&Dfr;", "&Efr;", "&Ffr;", "&Gfr;", 
"&Jfr;", "&Kfr;", "&Lfr;", "&Mfr;", "&Nfr;", "&Ofr;", "&Pfr;", 
"&Qfr;", "&Sfr;", "&Tfr;", "&Ufr;", "&Vfr;", "&Wfr;", "&Xfr;", 
"&Yfr;", "&afr;", "&bfr;", "&cfr;", "&dfr;", "&efr;", "&ffr;", 
"&gfr;", "&hfr;", "&ifr;", "&jfr;", "&kfr;", "&lfr;", "&mfr;", 
"&nfr;", "&ofr;", "&pfr;", "&qfr;", "&rfr;", "&sfr;", "&tfr;", 
"&ufr;", "&vfr;", "&wfr;", "&xfr;", "&yfr;", "&zfr;", "&Aopf;", 
"&Bopf;", "&Dopf;", "&Eopf;", "&Fopf;", "&Gopf;", "&Iopf;", "&Jopf;", 
"&Kopf;", "&Lopf;", "&Mopf;", "&Oopf;", "&Sopf;", "&Topf;", "&Uopf;", 
"&Vopf;", "&Wopf;", "&Xopf;", "&Yopf;", "&aopf;", "&bopf;", "&copf;", 
"&dopf;", "&eopf;", "&fopf;", "&gopf;", "&hopf;", "&iopf;", "&jopf;", 
"&kopf;", "&lopf;", "&mopf;", "&nopf;", "&oopf;", "&popf;", "&qopf;", 
"&ropf;", "&sopf;", "&topf;", "&uopf;", "&vopf;", "&wopf;", "&xopf;", 
"&yopf;", "&zopf;", "&b.Gamma;", "&b.Delta;", "&b.Theta;", "&b.Lambda;", 
"&b.Xi;", "&b.Pi;", "&b.Sigma;", "&b.Upsi;", "&b.Phi;", "&b.Psi;", 
"&b.Omega;", "&b.alpha;", "&b.beta;", "&b.gamma;", "&b.delta;", 
"&b.epsi;", "&b.zeta;", "&b.eta;", "&b.thetas;", "&b.iota;", 
"&b.kappa;", "&b.lambda;", "&b.mu;", "&b.nu;", "&b.xi;", "&b.pi;", 
"&b.rho;", "&b.sigmav;", "&b.sigma;", "&b.tau;", "&b.upsi;", 
"&b.phi;", "&b.chi;", "&b.psi;", "&b.omega;", "&b.epsiv;", "&b.thetav;", 
"&b.kappav;", "&b.phiv;", "&b.rhov;", "&b.piv;", "&b.Gammad;", 
"&b.gammad;"), hex = c("&#x0009;", "&#x000A;", "&#x0021;", "&#x0022;", 
"&#x0022;", "&#x0023;", "&#x0024;", "&#x0025;", "&#x0026;", "&#x0026;", 
"&#x0027;", "&#x0028;", "&#x0029;", "&#x002A;", "&#x002A;", "&#x002B;", 
"&#x002C;", "&#x002E;", "&#x002F;", "&#x003A;", "&#x003B;", "&#x003C;", 
"&#x003C;", "&#x003C;", "&#x20D2;", "&#x003D;", "&#x003D;", "&#x20E5;", 
"&#x003E;", "&#x003E;", "&#x003E;", "&#x20D2;", "&#x003F;", "&#x0040;", 
"&#x005B;", "&#x005B;", "&#x005C;", "&#x005D;", "&#x005D;", "&#x005E;", 
"&#x005F;", "&#x005F;", "&#x0060;", "&#x0060;", "&#x0066;", "&#x006A;", 
"&#x007B;", "&#x007B;", "&#x007C;", "&#x007C;", "&#x007C;", "&#x007D;", 
"&#x007D;", "&#x00A0;", "&#x00A0;", "&#x00A1;", "&#x00A2;", "&#x00A3;", 
"&#x00A4;", "&#x00A5;", "&#x00A6;", "&#x00A7;", "&#x00A8;", "&#x00A8;", 
"&#x00A8;", "&#x00A8;", "&#x00A9;", "&#x00A9;", "&#x00AA;", "&#x00AB;", 
"&#x00AC;", "&#x00AD;", "&#x00AE;", "&#x00AE;", "&#x00AE;", "&#x00AF;", 
"&#x00AF;", "&#x00B0;", "&#x00B1;", "&#x00B1;", "&#x00B1;", "&#x00B2;", 
"&#x00B3;", "&#x00B4;", "&#x00B4;", "&#x00B5;", "&#x00B6;", "&#x00B7;", 
"&#x00B7;", "&#x00B7;", "&#x00B8;", "&#x00B8;", "&#x00B9;", "&#x00BA;", 
"&#x00BB;", "&#x00BC;", "&#x00BD;", "&#x00BD;", "&#x00BE;", "&#x00BF;", 
"&#x00C0;", "&#x00C1;", "&#x00C2;", "&#x00C3;", "&#x00C4;", "&#x00C5;", 
"&#x00C5;", "&#x00C6;", "&#x00C7;", "&#x00C8;", "&#x00C9;", "&#x00CA;", 
"&#x00CB;", "&#x00CC;", "&#x00CD;", "&#x00CE;", "&#x00CF;", "&#x00D0;", 
"&#x00D1;", "&#x00D2;", "&#x00D3;", "&#x00D4;", "&#x00D5;", "&#x00D6;", 
"&#x00D7;", "&#x00D8;", "&#x00D9;", "&#x00DA;", "&#x00DB;", "&#x00DC;", 
"&#x00DD;", "&#x00DE;", "&#x00DF;", "&#x00E0;", "&#x00E1;", "&#x00E2;", 
"&#x00E3;", "&#x00E4;", "&#x00E5;", "&#x00E6;", "&#x00E7;", "&#x00E8;", 
"&#x00E9;", "&#x00EA;", "&#x00EB;", "&#x00EC;", "&#x00ED;", "&#x00EE;", 
"&#x00EF;", "&#x00F0;", "&#x00F1;", "&#x00F2;", "&#x00F3;", "&#x00F4;", 
"&#x00F5;", "&#x00F6;", "&#x00F7;", "&#x00F7;", "&#x00F8;", "&#x00F9;", 
"&#x00FA;", "&#x00FB;", "&#x00FC;", "&#x00FD;", "&#x00FE;", "&#x00FF;", 
"&#x0100;", "&#x0101;", "&#x0102;", "&#x0103;", "&#x0104;", "&#x0105;", 
"&#x0106;", "&#x0107;", "&#x0108;", "&#x0109;", "&#x010A;", "&#x010B;", 
"&#x010C;", "&#x010D;", "&#x010E;", "&#x010F;", "&#x0110;", "&#x0111;", 
"&#x0112;", "&#x0113;", "&#x0116;", "&#x0117;", "&#x0118;", "&#x0119;", 
"&#x011A;", "&#x011B;", "&#x011C;", "&#x011D;", "&#x011E;", "&#x011F;", 
"&#x0120;", "&#x0121;", "&#x0122;", "&#x0124;", "&#x0125;", "&#x0126;", 
"&#x0127;", "&#x0128;", "&#x0129;", "&#x012A;", "&#x012B;", "&#x012E;", 
"&#x012F;", "&#x0130;", "&#x0131;", "&#x0131;", "&#x0132;", "&#x0133;", 
"&#x0134;", "&#x0135;", "&#x0136;", "&#x0137;", "&#x0138;", "&#x0139;", 
"&#x013A;", "&#x013B;", "&#x013C;", "&#x013D;", "&#x013E;", "&#x013F;", 
"&#x0140;", "&#x0141;", "&#x0142;", "&#x0143;", "&#x0144;", "&#x0145;", 
"&#x0146;", "&#x0147;", "&#x0148;", "&#x0149;", "&#x014A;", "&#x014B;", 
"&#x014C;", "&#x014D;", "&#x0150;", "&#x0151;", "&#x0152;", "&#x0153;", 
"&#x0154;", "&#x0155;", "&#x0156;", "&#x0157;", "&#x0158;", "&#x0159;", 
"&#x015A;", "&#x015B;", "&#x015C;", "&#x015D;", "&#x015E;", "&#x015F;", 
"&#x0160;", "&#x0161;", "&#x0162;", "&#x0163;", "&#x0164;", "&#x0165;", 
"&#x0166;", "&#x0167;", "&#x0168;", "&#x0169;", "&#x016A;", "&#x016B;", 
"&#x016C;", "&#x016D;", "&#x016E;", "&#x016F;", "&#x0170;", "&#x0171;", 
"&#x0172;", "&#x0173;", "&#x0174;", "&#x0175;", "&#x0176;", "&#x0177;", 
"&#x0178;", "&#x0179;", "&#x017A;", "&#x017B;", "&#x017C;", "&#x017D;", 
"&#x017E;", "&#x0192;", "&#x01B5;", "&#x01F5;", "&#x0237;", "&#x02C6;", 
"&#x02C7;", "&#x02C7;", "&#x02D8;", "&#x02D8;", "&#x02D9;", "&#x02D9;", 
"&#x02DA;", "&#x02DB;", "&#x02DC;", "&#x02DC;", "&#x02DD;", "&#x02DD;", 
"&#x0311;", "&#x0386;", "&#x0388;", "&#x0389;", "&#x038A;", "&#x038C;", 
"&#x038E;", "&#x038F;", "&#x0390;", "&#x0391;", "&#x0391;", "&#x0392;", 
"&#x0392;", "&#x0393;", "&#x0393;", "&#x0394;", "&#x0394;", "&#x0395;", 
"&#x0395;", "&#x0396;", "&#x0396;", "&#x0397;", "&#x0397;", "&#x0398;", 
"&#x0398;", "&#x0399;", "&#x0399;", "&#x039A;", "&#x039A;", "&#x039B;", 
"&#x039B;", "&#x039C;", "&#x039C;", "&#x039D;", "&#x039D;", "&#x039E;", 
"&#x039E;", "&#x039F;", "&#x039F;", "&#x03A0;", "&#x03A0;", "&#x03A1;", 
"&#x03A1;", "&#x03A3;", "&#x03A3;", "&#x03A4;", "&#x03A4;", "&#x03A5;", 
"&#x03A5;", "&#x03A6;", "&#x03A6;", "&#x03A7;", "&#x03A7;", "&#x03A8;", 
"&#x03A8;", "&#x03A9;", "&#x03A9;", "&#x03A9;", "&#x03AA;", "&#x03AB;", 
"&#x03AC;", "&#x03AD;", "&#x03AE;", "&#x03AF;", "&#x03B0;", "&#x03B1;", 
"&#x03B1;", "&#x03B2;", "&#x03B2;", "&#x03B3;", "&#x03B3;", "&#x03B4;", 
"&#x03B4;", "&#x03B5;", "&#x03B5;", "&#x03B5;", "&#x03B6;", "&#x03B6;", 
"&#x03B7;", "&#x03B7;", "&#x03B8;", "&#x03B8;", "&#x03B9;", "&#x03B9;", 
"&#x03BA;", "&#x03BA;", "&#x03BB;", "&#x03BB;", "&#x03BC;", "&#x03BC;", 
"&#x03BD;", "&#x03BD;", "&#x03BE;", "&#x03BE;", "&#x03BF;", "&#x03BF;", 
"&#x03C0;", "&#x03C0;", "&#x03C1;", "&#x03C1;", "&#x03C2;", "&#x03C2;", 
"&#x03C2;", "&#x03C2;", "&#x03C3;", "&#x03C3;", "&#x03C4;", "&#x03C4;", 
"&#x03C5;", "&#x03C5;", "&#x03C5;", "&#x03C6;", "&#x03C6;", "&#x03C7;", 
"&#x03C7;", "&#x03C8;", "&#x03C8;", "&#x03C9;", "&#x03C9;", "&#x03CA;", 
"&#x03CB;", "&#x03CC;", "&#x03CD;", "&#x03CE;", "&#x03D1;", "&#x03D1;", 
"&#x03D1;", "&#x03D2;", "&#x03D2;", "&#x03D5;", "&#x03D5;", "&#x03D5;", 
"&#x03D6;", "&#x03D6;", "&#x03DC;", "&#x03DD;", "&#x03DD;", "&#x03F0;", 
"&#x03F0;", "&#x03F1;", "&#x03F1;", "&#x03F5;", "&#x03F5;", "&#x03F5;", 
"&#x03F6;", "&#x03F6;", "&#x0401;", "&#x0402;", "&#x0403;", "&#x0404;", 
"&#x0405;", "&#x0406;", "&#x0407;", "&#x0408;", "&#x0409;", "&#x040A;", 
"&#x040B;", "&#x040C;", "&#x040E;", "&#x040F;", "&#x0410;", "&#x0411;", 
"&#x0412;", "&#x0413;", "&#x0414;", "&#x0415;", "&#x0416;", "&#x0417;", 
"&#x0418;", "&#x0419;", "&#x041A;", "&#x041B;", "&#x041C;", "&#x041D;", 
"&#x041E;", "&#x041F;", "&#x0420;", "&#x0421;", "&#x0422;", "&#x0423;", 
"&#x0424;", "&#x0425;", "&#x0426;", "&#x0427;", "&#x0428;", "&#x0429;", 
"&#x042A;", "&#x042B;", "&#x042C;", "&#x042D;", "&#x042E;", "&#x042F;", 
"&#x0430;", "&#x0431;", "&#x0432;", "&#x0433;", "&#x0434;", "&#x0435;", 
"&#x0436;", "&#x0437;", "&#x0438;", "&#x0439;", "&#x043A;", "&#x043B;", 
"&#x043C;", "&#x043D;", "&#x043E;", "&#x043F;", "&#x0440;", "&#x0441;", 
"&#x0442;", "&#x0443;", "&#x0444;", "&#x0445;", "&#x0446;", "&#x0447;", 
"&#x0448;", "&#x0449;", "&#x044A;", "&#x044B;", "&#x044C;", "&#x044D;", 
"&#x044E;", "&#x044F;", "&#x0451;", "&#x0452;", "&#x0453;", "&#x0454;", 
"&#x0455;", "&#x0456;", "&#x0457;", "&#x0458;", "&#x0459;", "&#x045A;", 
"&#x045B;", "&#x045C;", "&#x045E;", "&#x045F;", "&#x2002;", "&#x2003;", 
"&#x2004;", "&#x2005;", "&#x2007;", "&#x2008;", "&#x2009;", "&#x2009;", 
"&#x200A;", "&#x200A;", "&#x200B;", "&#x200B;", "&#x200B;", "&#x200B;", 
"&#x200B;", "&#x200C;", "&#x200D;", "&#x200E;", "&#x200F;", "&#x2010;", 
"&#x2010;", "&#x2013;", "&#x2014;", "&#x2015;", "&#x2016;", "&#x2016;", 
"&#x2018;", "&#x2018;", "&#x2019;", "&#x2019;", "&#x2019;", "&#x201A;", 
"&#x201A;", "&#x201C;", "&#x201C;", "&#x201D;", "&#x201D;", "&#x201D;", 
"&#x201E;", "&#x201E;", "&#x2020;", "&#x2021;", "&#x2021;", "&#x2022;", 
"&#x2022;", "&#x2025;", "&#x2026;", "&#x2026;", "&#x2030;", "&#x2031;", 
"&#x2032;", "&#x2033;", "&#x2034;", "&#x2035;", "&#x2035;", "&#x2039;", 
"&#x203A;", "&#x203E;", "&#x203E;", "&#x2041;", "&#x2043;", "&#x2044;", 
"&#x204F;", "&#x2057;", "&#x205F;", "&#x205F;", "&#x200A;", "&#x2060;", 
"&#x2061;", "&#x2061;", "&#x2062;", "&#x2062;", "&#x2063;", "&#x2063;", 
"&#x20AC;", "&#x20DB;", "&#x20DB;", "&#x20DC;", "&#x2102;", "&#x2102;", 
"&#x2105;", "&#x210A;", "&#x210B;", "&#x210B;", "&#x210B;", "&#x210C;", 
"&#x210C;", "&#x210D;", "&#x210D;", "&#x210E;", "&#x210F;", "&#x210F;", 
"&#x210F;", "&#x210F;", "&#x2110;", "&#x2110;", "&#x2111;", "&#x2111;", 
"&#x2111;", "&#x2111;", "&#x2112;", "&#x2112;", "&#x2112;", "&#x2113;", 
"&#x2115;", "&#x2115;", "&#x2116;", "&#x2117;", "&#x2118;", "&#x2118;", 
"&#x2119;", "&#x2119;", "&#x211A;", "&#x211A;", "&#x211B;", "&#x211B;", 
"&#x211C;", "&#x211C;", "&#x211C;", "&#x211C;", "&#x211D;", "&#x211D;", 
"&#x211E;", "&#x2122;", "&#x2122;", "&#x2124;", "&#x2124;", "&#x2127;", 
"&#x2128;", "&#x2128;", "&#x2129;", "&#x212C;", "&#x212C;", "&#x212C;", 
"&#x212D;", "&#x212D;", "&#x212F;", "&#x2130;", "&#x2130;", "&#x2131;", 
"&#x2131;", "&#x2133;", "&#x2133;", "&#x2133;", "&#x2134;", "&#x2134;", 
"&#x2134;", "&#x2135;", "&#x2135;", "&#x2136;", "&#x2137;", "&#x2138;", 
"&#x2145;", "&#x2145;", "&#x2146;", "&#x2146;", "&#x2147;", "&#x2147;", 
"&#x2147;", "&#x2148;", "&#x2148;", "&#x2153;", "&#x2154;", "&#x2155;", 
"&#x2156;", "&#x2157;", "&#x2158;", "&#x2159;", "&#x215A;", "&#x215B;", 
"&#x215C;", "&#x215D;", "&#x215E;", "&#x2190;", "&#x2190;", "&#x2190;", 
"&#x2190;", "&#x2190;", "&#x2191;", "&#x2191;", "&#x2191;", "&#x2191;", 
"&#x2192;", "&#x2192;", "&#x2192;", "&#x2192;", "&#x2192;", "&#x2193;", 
"&#x2193;", "&#x2193;", "&#x2193;", "&#x2194;", "&#x2194;", "&#x2194;", 
"&#x2195;", "&#x2195;", "&#x2195;", "&#x2196;", "&#x2196;", "&#x2196;", 
"&#x2197;", "&#x2197;", "&#x2197;", "&#x2198;", "&#x2198;", "&#x2198;", 
"&#x2199;", "&#x2199;", "&#x2199;", "&#x219A;", "&#x219A;", "&#x219B;", 
"&#x219B;", "&#x219D;", "&#x219D;", "&#x219D;", "&#x0338;", "&#x219E;", 
"&#x219E;", "&#x219F;", "&#x21A0;", "&#x21A0;", "&#x21A1;", "&#x21A2;", 
"&#x21A2;", "&#x21A3;", "&#x21A3;", "&#x21A4;", "&#x21A4;", "&#x21A5;", 
"&#x21A5;", "&#x21A6;", "&#x21A6;", "&#x21A6;", "&#x21A7;", "&#x21A7;", 
"&#x21A9;", "&#x21A9;", "&#x21AA;", "&#x21AA;", "&#x21AB;", "&#x21AB;", 
"&#x21AC;", "&#x21AC;", "&#x21AD;", "&#x21AD;", "&#x21AE;", "&#x21AE;", 
"&#x21B0;", "&#x21B0;", "&#x21B1;", "&#x21B1;", "&#x21B2;", "&#x21B3;", 
"&#x21B5;", "&#x21B6;", "&#x21B6;", "&#x21B7;", "&#x21B7;", "&#x21BA;", 
"&#x21BA;", "&#x21BB;", "&#x21BB;", "&#x21BC;", "&#x21BC;", "&#x21BC;", 
"&#x21BD;", "&#x21BD;", "&#x21BD;", "&#x21BE;", "&#x21BE;", "&#x21BE;", 
"&#x21BF;", "&#x21BF;", "&#x21BF;", "&#x21C0;", "&#x21C0;", "&#x21C0;", 
"&#x21C1;", "&#x21C1;", "&#x21C1;", "&#x21C2;", "&#x21C2;", "&#x21C2;", 
"&#x21C3;", "&#x21C3;", "&#x21C3;", "&#x21C4;", "&#x21C4;", "&#x21C4;", 
"&#x21C5;", "&#x21C5;", "&#x21C6;", "&#x21C6;", "&#x21C6;", "&#x21C7;", 
"&#x21C7;", "&#x21C8;", "&#x21C8;", "&#x21C9;", "&#x21C9;", "&#x21CA;", 
"&#x21CA;", "&#x21CB;", "&#x21CB;", "&#x21CB;", "&#x21CC;", "&#x21CC;", 
"&#x21CC;", "&#x21CD;", "&#x21CD;", "&#x21CE;", "&#x21CE;", "&#x21CF;", 
"&#x21CF;", "&#x21D0;", "&#x21D0;", "&#x21D0;", "&#x21D1;", "&#x21D1;", 
"&#x21D1;", "&#x21D2;", "&#x21D2;", "&#x21D2;", "&#x21D2;", "&#x21D3;", 
"&#x21D3;", "&#x21D3;", "&#x21D4;", "&#x21D4;", "&#x21D4;", "&#x21D4;", 
"&#x21D5;", "&#x21D5;", "&#x21D5;", "&#x21D6;", "&#x21D7;", "&#x21D8;", 
"&#x21D9;", "&#x21DA;", "&#x21DA;", "&#x21DB;", "&#x21DB;", "&#x21DD;", 
"&#x21E4;", "&#x21E4;", "&#x21E5;", "&#x21E5;", "&#x21F5;", "&#x21F5;", 
"&#x21FD;", "&#x21FE;", "&#x21FF;", "&#x2200;", "&#x2200;", "&#x2201;", 
"&#x2201;", "&#x2202;", "&#x2202;", "&#x2202;", "&#x0338;", "&#x2203;", 
"&#x2203;", "&#x2204;", "&#x2204;", "&#x2204;", "&#x2205;", "&#x2205;", 
"&#x2205;", "&#x2205;", "&#x2207;", "&#x2207;", "&#x2208;", "&#x2208;", 
"&#x2208;", "&#x2208;", "&#x2209;", "&#x2209;", "&#x2209;", "&#x220B;", 
"&#x220B;", "&#x220B;", "&#x220B;", "&#x220C;", "&#x220C;", "&#x220C;", 
"&#x220F;", "&#x220F;", "&#x2210;", "&#x2210;", "&#x2211;", "&#x2211;", 
"&#x2212;", "&#x2213;", "&#x2213;", "&#x2213;", "&#x2214;", "&#x2214;", 
"&#x2216;", "&#x2216;", "&#x2216;", "&#x2216;", "&#x2216;", "&#x2217;", 
"&#x2218;", "&#x2218;", "&#x221A;", "&#x221A;", "&#x221D;", "&#x221D;", 
"&#x221D;", "&#x221D;", "&#x221D;", "&#x221E;", "&#x221F;", "&#x2220;", 
"&#x2220;", "&#x2220;", "&#x20D2;", "&#x2221;", "&#x2221;", "&#x2222;", 
"&#x2223;", "&#x2223;", "&#x2223;", "&#x2223;", "&#x2224;", "&#x2224;", 
"&#x2224;", "&#x2224;", "&#x2225;", "&#x2225;", "&#x2225;", "&#x2225;", 
"&#x2225;", "&#x2226;", "&#x2226;", "&#x2226;", "&#x2226;", "&#x2226;", 
"&#x2227;", "&#x2227;", "&#x2228;", "&#x2228;", "&#x2229;", "&#x2229;", 
"&#xFE00;", "&#x222A;", "&#x222A;", "&#xFE00;", "&#x222B;", "&#x222B;", 
"&#x222C;", "&#x222D;", "&#x222D;", "&#x222E;", "&#x222E;", "&#x222E;", 
"&#x222F;", "&#x222F;", "&#x2230;", "&#x2231;", "&#x2232;", "&#x2232;", 
"&#x2233;", "&#x2233;", "&#x2234;", "&#x2234;", "&#x2234;", "&#x2235;", 
"&#x2235;", "&#x2235;", "&#x2236;", "&#x2237;", "&#x2237;", "&#x2238;", 
"&#x2238;", "&#x223A;", "&#x223B;", "&#x223C;", "&#x223C;", "&#x223C;", 
"&#x223C;", "&#x223C;", "&#x20D2;", "&#x223D;", "&#x223D;", "&#x223D;", 
"&#x0331;", "&#x223E;", "&#x223E;", "&#x223E;", "&#x0333;", "&#x223F;", 
"&#x2240;", "&#x2240;", "&#x2240;", "&#x2241;", "&#x2241;", "&#x2242;", 
"&#x2242;", "&#x2242;", "&#x2242;", "&#x2242;", "&#x0338;", "&#x0338;", 
"&#x2243;", "&#x2243;", "&#x2243;", "&#x2244;", "&#x2244;", "&#x2244;", 
"&#x2245;", "&#x2245;", "&#x2246;", "&#x2247;", "&#x2247;", "&#x2248;", 
"&#x2248;", "&#x2248;", "&#x2248;", "&#x2248;", "&#x2248;", "&#x2249;", 
"&#x2249;", "&#x2249;", "&#x224A;", "&#x224A;", "&#x224B;", "&#x224B;", 
"&#x0338;", "&#x224C;", "&#x224C;", "&#x224D;", "&#x224D;", "&#x224D;", 
"&#x20D2;", "&#x224E;", "&#x224E;", "&#x224E;", "&#x224E;", "&#x224E;", 
"&#x0338;", "&#x0338;", "&#x224F;", "&#x224F;", "&#x224F;", "&#x224F;", 
"&#x224F;", "&#x0338;", "&#x0338;", "&#x2250;", "&#x2250;", "&#x2250;", 
"&#x2250;", "&#x0338;", "&#x2251;", "&#x2251;", "&#x2252;", "&#x2252;", 
"&#x2253;", "&#x2253;", "&#x2254;", "&#x2254;", "&#x2254;", "&#x2255;", 
"&#x2255;", "&#x2256;", "&#x2256;", "&#x2257;", "&#x2257;", "&#x2259;", 
"&#x225A;", "&#x225C;", "&#x225C;", "&#x225F;", "&#x225F;", "&#x2260;", 
"&#x2260;", "&#x2261;", "&#x2261;", "&#x2261;", "&#x20E5;", "&#x2262;", 
"&#x2262;", "&#x2264;", "&#x2264;", "&#x2264;", "&#x20D2;", "&#x2265;", 
"&#x2265;", "&#x2265;", "&#x2265;", "&#x20D2;", "&#x2266;", "&#x2266;", 
"&#x2266;", "&#x2266;", "&#x2266;", "&#x0338;", "&#x0338;", "&#x2267;", 
"&#x2267;", "&#x2267;", "&#x2267;", "&#x2267;", "&#x2267;", "&#x0338;", 
"&#x0338;", "&#x0338;", "&#x2268;", "&#x2268;", "&#x2268;", "&#x2268;", 
"&#xFE00;", "&#xFE00;", "&#x2269;", "&#x2269;", "&#x2269;", "&#x2269;", 
"&#xFE00;", "&#xFE00;", "&#x226A;", "&#x226A;", "&#x226A;", "&#x226A;", 
"&#x226A;", "&#x0338;", "&#x0338;", "&#x226A;", "&#x20D2;", "&#x226B;", 
"&#x226B;", "&#x226B;", "&#x226B;", "&#x226B;", "&#x0338;", "&#x0338;", 
"&#x226B;", "&#x20D2;", "&#x226C;", "&#x226C;", "&#x226D;", "&#x226E;", 
"&#x226E;", "&#x226E;", "&#x226F;", "&#x226F;", "&#x226F;", "&#x2270;", 
"&#x2270;", "&#x2270;", "&#x2271;", "&#x2271;", "&#x2271;", "&#x2272;", 
"&#x2272;", "&#x2272;", "&#x2273;", "&#x2273;", "&#x2273;", "&#x2274;", 
"&#x2274;", "&#x2275;", "&#x2275;", "&#x2276;", "&#x2276;", "&#x2276;", 
"&#x2277;", "&#x2277;", "&#x2277;", "&#x2278;", "&#x2278;", "&#x2279;", 
"&#x2279;", "&#x227A;", "&#x227A;", "&#x227A;", "&#x227B;", "&#x227B;", 
"&#x227B;", "&#x227C;", "&#x227C;", "&#x227C;", "&#x227D;", "&#x227D;", 
"&#x227D;", "&#x227E;", "&#x227E;", "&#x227E;", "&#x227F;", "&#x227F;", 
"&#x227F;", "&#x227F;", "&#x0338;", "&#x2280;", "&#x2280;", "&#x2280;", 
"&#x2281;", "&#x2281;", "&#x2281;", "&#x2282;", "&#x2282;", "&#x2282;", 
"&#x2282;", "&#x2282;", "&#x20D2;", "&#x20D2;", "&#x20D2;", "&#x2283;", 
"&#x2283;", "&#x2283;", "&#x2283;", "&#x2283;", "&#x2283;", "&#x20D2;", 
"&#x20D2;", "&#x20D2;", "&#x2284;", "&#x2285;", "&#x2286;", "&#x2286;", 
"&#x2286;", "&#x2287;", "&#x2287;", "&#x2287;", "&#x2288;", "&#x2288;", 
"&#x2288;", "&#x2289;", "&#x2289;", "&#x2289;", "&#x228A;", "&#x228A;", 
"&#x228A;", "&#x228A;", "&#xFE00;", "&#xFE00;", "&#x228B;", "&#x228B;", 
"&#x228B;", "&#x228B;", "&#xFE00;", "&#xFE00;", "&#x228D;", "&#x228E;", 
"&#x228E;", "&#x228F;", "&#x228F;", "&#x228F;", "&#x228F;", "&#x0338;", 
"&#x2290;", "&#x2290;", "&#x2290;", "&#x2290;", "&#x0338;", "&#x2291;", 
"&#x2291;", "&#x2291;", "&#x2292;", "&#x2292;", "&#x2292;", "&#x2293;", 
"&#x2293;", "&#x2293;", "&#xFE00;", "&#x2294;", "&#x2294;", "&#x2294;", 
"&#xFE00;", "&#x2295;", "&#x2295;", "&#x2296;", "&#x2296;", "&#x2297;", 
"&#x2297;", "&#x2298;", "&#x2299;", "&#x2299;", "&#x229A;", "&#x229A;", 
"&#x229B;", "&#x229B;", "&#x229D;", "&#x229D;", "&#x229E;", "&#x229E;", 
"&#x229F;", "&#x229F;", "&#x22A0;", "&#x22A0;", "&#x22A1;", "&#x22A1;", 
"&#x22A2;", "&#x22A2;", "&#x22A3;", "&#x22A3;", "&#x22A4;", "&#x22A4;", 
"&#x22A5;", "&#x22A5;", "&#x22A5;", "&#x22A5;", "&#x22A7;", "&#x22A8;", 
"&#x22A8;", "&#x22A9;", "&#x22AA;", "&#x22AB;", "&#x22AC;", "&#x22AD;", 
"&#x22AE;", "&#x22AF;", "&#x22B0;", "&#x22B2;", "&#x22B2;", "&#x22B2;", 
"&#x22B3;", "&#x22B3;", "&#x22B3;", "&#x22B4;", "&#x22B4;", "&#x22B4;", 
"&#x22B4;", "&#x20D2;", "&#x22B5;", "&#x22B5;", "&#x22B5;", "&#x22B5;", 
"&#x20D2;", "&#x22B6;", "&#x22B7;", "&#x22B8;", "&#x22B8;", "&#x22B9;", 
"&#x22BA;", "&#x22BA;", "&#x22BB;", "&#x22BD;", "&#x22BE;", "&#x22BF;", 
"&#x22C0;", "&#x22C0;", "&#x22C0;", "&#x22C1;", "&#x22C1;", "&#x22C1;", 
"&#x22C2;", "&#x22C2;", "&#x22C2;", "&#x22C3;", "&#x22C3;", "&#x22C3;", 
"&#x22C4;", "&#x22C4;", "&#x22C4;", "&#x22C5;", "&#x22C6;", "&#x22C6;", 
"&#x22C7;", "&#x22C7;", "&#x22C8;", "&#x22C9;", "&#x22CA;", "&#x22CB;", 
"&#x22CB;", "&#x22CC;", "&#x22CC;", "&#x22CD;", "&#x22CD;", "&#x22CE;", 
"&#x22CE;", "&#x22CF;", "&#x22CF;", "&#x22D0;", "&#x22D0;", "&#x22D1;", 
"&#x22D1;", "&#x22D2;", "&#x22D3;", "&#x22D4;", "&#x22D4;", "&#x22D5;", 
"&#x22D6;", "&#x22D6;", "&#x22D7;", "&#x22D7;", "&#x22D8;", "&#x22D8;", 
"&#x0338;", "&#x22D9;", "&#x22D9;", "&#x22D9;", "&#x0338;", "&#x22DA;", 
"&#x22DA;", "&#x22DA;", "&#x22DA;", "&#xFE00;", "&#x22DB;", "&#x22DB;", 
"&#x22DB;", "&#x22DB;", "&#xFE00;", "&#x22DE;", "&#x22DE;", "&#x22DF;", 
"&#x22DF;", "&#x22E0;", "&#x22E0;", "&#x22E1;", "&#x22E1;", "&#x22E2;", 
"&#x22E2;", "&#x22E3;", "&#x22E3;", "&#x22E6;", "&#x22E7;", "&#x22E8;", 
"&#x22E8;", "&#x22E9;", "&#x22E9;", "&#x22EA;", "&#x22EA;", "&#x22EA;", 
"&#x22EB;", "&#x22EB;", "&#x22EB;", "&#x22EC;", "&#x22EC;", "&#x22EC;", 
"&#x22ED;", "&#x22ED;", "&#x22ED;", "&#x22EE;", "&#x22EF;", "&#x22F0;", 
"&#x22F1;", "&#x22F2;", "&#x22F3;", "&#x22F4;", "&#x22F5;", "&#x22F5;", 
"&#x0338;", "&#x22F6;", "&#x22F7;", "&#x22F9;", "&#x22F9;", "&#x0338;", 
"&#x22FA;", "&#x22FB;", "&#x22FC;", "&#x22FD;", "&#x22FE;", "&#x2305;", 
"&#x2305;", "&#x2306;", "&#x2306;", "&#x2308;", "&#x2308;", "&#x2309;", 
"&#x2309;", "&#x230A;", "&#x230A;", "&#x230B;", "&#x230B;", "&#x230C;", 
"&#x230D;", "&#x230E;", "&#x230F;", "&#x2310;", "&#x2312;", "&#x2313;", 
"&#x2315;", "&#x2316;", "&#x231C;", "&#x231C;", "&#x231D;", "&#x231D;", 
"&#x231E;", "&#x231E;", "&#x231F;", "&#x231F;", "&#x2322;", "&#x2322;", 
"&#x2323;", "&#x2323;", "&#x232D;", "&#x232E;", "&#x2336;", "&#x233D;", 
"&#x233F;", "&#x237C;", "&#x23B0;", "&#x23B0;", "&#x23B1;", "&#x23B1;", 
"&#x23B4;", "&#x23B4;", "&#x23B5;", "&#x23B5;", "&#x23B6;", "&#x23DC;", 
"&#x23DD;", "&#x23DE;", "&#x23DF;", "&#x23E2;", "&#x23E7;", "&#x2423;", 
"&#x24C8;", "&#x24C8;", "&#x2500;", "&#x2500;", "&#x2502;", "&#x250C;", 
"&#x2510;", "&#x2514;", "&#x2518;", "&#x251C;", "&#x2524;", "&#x252C;", 
"&#x2534;", "&#x253C;", "&#x2550;", "&#x2551;", "&#x2552;", "&#x2553;", 
"&#x2554;", "&#x2555;", "&#x2556;", "&#x2557;", "&#x2558;", "&#x2559;", 
"&#x255A;", "&#x255B;", "&#x255C;", "&#x255D;", "&#x255E;", "&#x255F;", 
"&#x2560;", "&#x2561;", "&#x2562;", "&#x2563;", "&#x2564;", "&#x2565;", 
"&#x2566;", "&#x2567;", "&#x2568;", "&#x2569;", "&#x256A;", "&#x256B;", 
"&#x256C;", "&#x2580;", "&#x2584;", "&#x2588;", "&#x2591;", "&#x2592;", 
"&#x2593;", "&#x25A1;", "&#x25A1;", "&#x25A1;", "&#x25AA;", "&#x25AA;", 
"&#x25AA;", "&#x25AA;", "&#x25AB;", "&#x25AD;", "&#x25AE;", "&#x25B1;", 
"&#x25B3;", "&#x25B3;", "&#x25B4;", "&#x25B4;", "&#x25B5;", "&#x25B5;", 
"&#x25B8;", "&#x25B8;", "&#x25B9;", "&#x25B9;", "&#x25BD;", "&#x25BD;", 
"&#x25BE;", "&#x25BE;", "&#x25BF;", "&#x25BF;", "&#x25C2;", "&#x25C2;", 
"&#x25C3;", "&#x25C3;", "&#x25CA;", "&#x25CA;", "&#x25CB;", "&#x25EC;", 
"&#x25EF;", "&#x25EF;", "&#x25F8;", "&#x25F9;", "&#x25FA;", "&#x25FB;", 
"&#x25FC;", "&#x2605;", "&#x2605;", "&#x2606;", "&#x260E;", "&#x2640;", 
"&#x2642;", "&#x2660;", "&#x2660;", "&#x2663;", "&#x2663;", "&#x2665;", 
"&#x2665;", "&#x2666;", "&#x2666;", "&#x266A;", "&#x266D;", "&#x266E;", 
"&#x266E;", "&#x266F;", "&#x2713;", "&#x2713;", "&#x2717;", "&#x2720;", 
"&#x2720;", "&#x2736;", "&#x2758;", "&#x2772;", "&#x2773;", "&#x27C8;", 
"&#x27C9;", "&#x27E6;", "&#x27E6;", "&#x27E7;", "&#x27E7;", "&#x27E8;", 
"&#x27E8;", "&#x27E8;", "&#x27E9;", "&#x27E9;", "&#x27E9;", "&#x27EA;", 
"&#x27EB;", "&#x27EC;", "&#x27ED;", "&#x27F5;", "&#x27F5;", "&#x27F5;", 
"&#x27F6;", "&#x27F6;", "&#x27F6;", "&#x27F7;", "&#x27F7;", "&#x27F7;", 
"&#x27F8;", "&#x27F8;", "&#x27F8;", "&#x27F9;", "&#x27F9;", "&#x27F9;", 
"&#x27FA;", "&#x27FA;", "&#x27FA;", "&#x27FC;", "&#x27FC;", "&#x27FF;", 
"&#x2902;", "&#x2903;", "&#x2904;", "&#x2905;", "&#x290C;", "&#x290D;", 
"&#x290D;", "&#x290E;", "&#x290F;", "&#x290F;", "&#x2910;", "&#x2910;", 
"&#x2911;", "&#x2912;", "&#x2913;", "&#x2916;", "&#x2919;", "&#x291A;", 
"&#x291B;", "&#x291C;", "&#x291D;", "&#x291E;", "&#x291F;", "&#x2920;", 
"&#x2923;", "&#x2924;", "&#x2925;", "&#x2925;", "&#x2926;", "&#x2926;", 
"&#x2927;", "&#x2928;", "&#x2928;", "&#x2929;", "&#x2929;", "&#x292A;", 
"&#x2933;", "&#x2933;", "&#x0338;", "&#x2935;", "&#x2936;", "&#x2937;", 
"&#x2938;", "&#x2939;", "&#x293C;", "&#x293D;", "&#x2945;", "&#x2948;", 
"&#x2949;", "&#x294A;", "&#x294B;", "&#x294E;", "&#x294F;", "&#x2950;", 
"&#x2951;", "&#x2952;", "&#x2953;", "&#x2954;", "&#x2955;", "&#x2956;", 
"&#x2957;", "&#x2958;", "&#x2959;", "&#x295A;", "&#x295B;", "&#x295C;", 
"&#x295D;", "&#x295E;", "&#x295F;", "&#x2960;", "&#x2961;", "&#x2962;", 
"&#x2963;", "&#x2964;", "&#x2965;", "&#x2966;", "&#x2967;", "&#x2968;", 
"&#x2969;", "&#x296A;", "&#x296B;", "&#x296C;", "&#x296D;", "&#x296E;", 
"&#x296E;", "&#x296F;", "&#x296F;", "&#x2970;", "&#x2971;", "&#x2972;", 
"&#x2973;", "&#x2974;", "&#x2975;", "&#x2976;", "&#x2978;", "&#x2979;", 
"&#x297B;", "&#x297C;", "&#x297D;", "&#x297E;", "&#x297F;", "&#x2985;", 
"&#x2986;", "&#x298B;", "&#x298C;", "&#x298D;", "&#x298E;", "&#x298F;", 
"&#x2990;", "&#x2991;", "&#x2992;", "&#x2993;", "&#x2994;", "&#x2995;", 
"&#x2996;", "&#x299A;", "&#x299C;", "&#x299D;", "&#x29A4;", "&#x29A5;", 
"&#x29A6;", "&#x29A7;", "&#x29A8;", "&#x29A9;", "&#x29AA;", "&#x29AB;", 
"&#x29AC;", "&#x29AD;", "&#x29AE;", "&#x29AF;", "&#x29B0;", "&#x29B1;", 
"&#x29B2;", "&#x29B3;", "&#x29B4;", "&#x29B5;", "&#x29B6;", "&#x29B7;", 
"&#x29B9;", "&#x29BB;", "&#x29BC;", "&#x29BE;", "&#x29BF;", "&#x29C0;", 
"&#x29C1;", "&#x29C2;", "&#x29C3;", "&#x29C4;", "&#x29C5;", "&#x29C9;", 
"&#x29CD;", "&#x29CE;", "&#x29CF;", "&#x29CF;", "&#x0338;", "&#x29D0;", 
"&#x29D0;", "&#x0338;", "&#x29DC;", "&#x29DD;", "&#x29DE;", "&#x29E3;", 
"&#x29E4;", "&#x29E5;", "&#x29EB;", "&#x29EB;", "&#x29F4;", "&#x29F6;", 
"&#x2A00;", "&#x2A00;", "&#x2A01;", "&#x2A01;", "&#x2A02;", "&#x2A02;", 
"&#x2A04;", "&#x2A04;", "&#x2A06;", "&#x2A06;", "&#x2A0C;", "&#x2A0C;", 
"&#x2A0D;", "&#x2A10;", "&#x2A11;", "&#x2A12;", "&#x2A13;", "&#x2A14;", 
"&#x2A15;", "&#x2A16;", "&#x2A17;", "&#x2A22;", "&#x2A23;", "&#x2A24;", 
"&#x2A25;", "&#x2A26;", "&#x2A27;", "&#x2A29;", "&#x2A2A;", "&#x2A2D;", 
"&#x2A2E;", "&#x2A2F;", "&#x2A30;", "&#x2A31;", "&#x2A33;", "&#x2A34;", 
"&#x2A35;", "&#x2A36;", "&#x2A37;", "&#x2A38;", "&#x2A39;", "&#x2A3A;", 
"&#x2A3B;", "&#x2A3C;", "&#x2A3C;", "&#x2A3F;", "&#x2A40;", "&#x2A42;", 
"&#x2A43;", "&#x2A44;", "&#x2A45;", "&#x2A46;", "&#x2A47;", "&#x2A48;", 
"&#x2A49;", "&#x2A4A;", "&#x2A4B;", "&#x2A4C;", "&#x2A4D;", "&#x2A50;", 
"&#x2A53;", "&#x2A54;", "&#x2A55;", "&#x2A56;", "&#x2A57;", "&#x2A58;", 
"&#x2A5A;", "&#x2A5B;", "&#x2A5C;", "&#x2A5D;", "&#x2A5F;", "&#x2A66;", 
"&#x2A6A;", "&#x2A6D;", "&#x2A6D;", "&#x0338;", "&#x2A6E;", "&#x2A6F;", 
"&#x2A70;", "&#x2A70;", "&#x0338;", "&#x2A71;", "&#x2A72;", "&#x2A73;", 
"&#x2A74;", "&#x2A75;", "&#x2A77;", "&#x2A77;", "&#x2A78;", "&#x2A79;", 
"&#x2A7A;", "&#x2A7B;", "&#x2A7C;", "&#x2A7D;", "&#x2A7D;", "&#x2A7D;", 
"&#x2A7D;", "&#x2A7D;", "&#x2A7D;", "&#x0338;", "&#x0338;", "&#x0338;", 
"&#x2A7E;", "&#x2A7E;", "&#x2A7E;", "&#x2A7E;", "&#x2A7E;", "&#x2A7E;", 
"&#x0338;", "&#x0338;", "&#x0338;", "&#x2A7F;", "&#x2A80;", "&#x2A81;", 
"&#x2A82;", "&#x2A83;", "&#x2A84;", "&#x2A85;", "&#x2A85;", "&#x2A86;", 
"&#x2A86;", "&#x2A87;", "&#x2A87;", "&#x2A88;", "&#x2A88;", "&#x2A89;", 
"&#x2A89;", "&#x2A8A;", "&#x2A8A;", "&#x2A8B;", "&#x2A8B;", "&#x2A8C;", 
"&#x2A8C;", "&#x2A8D;", "&#x2A8E;", "&#x2A8F;", "&#x2A90;", "&#x2A91;", 
"&#x2A92;", "&#x2A93;", "&#x2A94;", "&#x2A95;", "&#x2A95;", "&#x2A96;", 
"&#x2A96;", "&#x2A97;", "&#x2A98;", "&#x2A99;", "&#x2A9A;", "&#x2A9D;", 
"&#x2A9E;", "&#x2A9F;", "&#x2AA0;", "&#x2AA1;", "&#x2AA1;", "&#x0338;", 
"&#x2AA2;", "&#x2AA2;", "&#x0338;", "&#x2AA4;", "&#x2AA5;", "&#x2AA6;", 
"&#x2AA7;", "&#x2AA8;", "&#x2AA9;", "&#x2AAA;", "&#x2AAB;", "&#x2AAC;", 
"&#x2AAC;", "&#xFE00;", "&#x2AAD;", "&#x2AAD;", "&#xFE00;", "&#x2AAE;", 
"&#x2AAF;", "&#x2AAF;", "&#x2AAF;", "&#x2AAF;", "&#x2AAF;", "&#x2AAF;", 
"&#x0338;", "&#x0338;", "&#x0338;", "&#x2AB0;", "&#x2AB0;", "&#x2AB0;", 
"&#x2AB0;", "&#x2AB0;", "&#x2AB0;", "&#x0338;", "&#x0338;", "&#x0338;", 
"&#x2AB3;", "&#x2AB4;", "&#x2AB5;", "&#x2AB5;", "&#x2AB6;", "&#x2AB6;", 
"&#x2AB7;", "&#x2AB7;", "&#x2AB8;", "&#x2AB8;", "&#x2AB9;", "&#x2AB9;", 
"&#x2ABA;", "&#x2ABA;", "&#x2ABB;", "&#x2ABC;", "&#x2ABD;", "&#x2ABE;", 
"&#x2ABF;", "&#x2AC0;", "&#x2AC1;", "&#x2AC2;", "&#x2AC3;", "&#x2AC4;", 
"&#x2AC5;", "&#x2AC5;", "&#x2AC5;", "&#x2AC5;", "&#x0338;", "&#x0338;", 
"&#x2AC6;", "&#x2AC6;", "&#x2AC6;", "&#x2AC6;", "&#x0338;", "&#x0338;", 
"&#x2AC7;", "&#x2AC8;", "&#x2ACB;", "&#x2ACB;", "&#x2ACB;", "&#x2ACB;", 
"&#xFE00;", "&#xFE00;", "&#x2ACC;", "&#x2ACC;", "&#x2ACC;", "&#x2ACC;", 
"&#xFE00;", "&#xFE00;", "&#x2ACF;", "&#x2AD0;", "&#x2AD1;", "&#x2AD2;", 
"&#x2AD3;", "&#x2AD4;", "&#x2AD5;", "&#x2AD6;", "&#x2AD7;", "&#x2AD8;", 
"&#x2AD9;", "&#x2ADA;", "&#x2ADB;", "&#x2AE4;", "&#x2AE4;", "&#x2AE6;", 
"&#x2AE7;", "&#x2AE8;", "&#x2AE9;", "&#x2AEB;", "&#x2AEC;", "&#x2AED;", 
"&#x2AEE;", "&#x2AEF;", "&#x2AF0;", "&#x2AF1;", "&#x2AF2;", "&#x2AF3;", 
"&#x2AFD;", "&#x2AFD;", "&#x20E5;", "&#xFB00;", "&#xFB01;", "&#xFB02;", 
"&#xFB03;", "&#xFB04;", "&#x1D49C;", "&#x1D49E;", "&#x1D49F;", 
"&#x1D4A2;", "&#x1D4A5;", "&#x1D4A6;", "&#x1D4A9;", "&#x1D4AA;", 
"&#x1D4AB;", "&#x1D4AC;", "&#x1D4AE;", "&#x1D4AF;", "&#x1D4B0;", 
"&#x1D4B1;", "&#x1D4B2;", "&#x1D4B3;", "&#x1D4B4;", "&#x1D4B5;", 
"&#x1D4B6;", "&#x1D4B7;", "&#x1D4B8;", "&#x1D4B9;", "&#x1D4BB;", 
"&#x1D4BD;", "&#x1D4BE;", "&#x1D4BF;", "&#x1D4C0;", "&#x1D4C1;", 
"&#x1D4C2;", "&#x1D4C3;", "&#x1D4C5;", "&#x1D4C6;", "&#x1D4C7;", 
"&#x1D4C8;", "&#x1D4C9;", "&#x1D4CA;", "&#x1D4CB;", "&#x1D4CC;", 
"&#x1D4CD;", "&#x1D4CE;", "&#x1D4CF;", "&#x1D504;", "&#x1D505;", 
"&#x1D507;", "&#x1D508;", "&#x1D509;", "&#x1D50A;", "&#x1D50D;", 
"&#x1D50E;", "&#x1D50F;", "&#x1D510;", "&#x1D511;", "&#x1D512;", 
"&#x1D513;", "&#x1D514;", "&#x1D516;", "&#x1D517;", "&#x1D518;", 
"&#x1D519;", "&#x1D51A;", "&#x1D51B;", "&#x1D51C;", "&#x1D51E;", 
"&#x1D51F;", "&#x1D520;", "&#x1D521;", "&#x1D522;", "&#x1D523;", 
"&#x1D524;", "&#x1D525;", "&#x1D526;", "&#x1D527;", "&#x1D528;", 
"&#x1D529;", "&#x1D52A;", "&#x1D52B;", "&#x1D52C;", "&#x1D52D;", 
"&#x1D52E;", "&#x1D52F;", "&#x1D530;", "&#x1D531;", "&#x1D532;", 
"&#x1D533;", "&#x1D534;", "&#x1D535;", "&#x1D536;", "&#x1D537;", 
"&#x1D538;", "&#x1D539;", "&#x1D53B;", "&#x1D53C;", "&#x1D53D;", 
"&#x1D53E;", "&#x1D540;", "&#x1D541;", "&#x1D542;", "&#x1D543;", 
"&#x1D544;", "&#x1D546;", "&#x1D54A;", "&#x1D54B;", "&#x1D54C;", 
"&#x1D54D;", "&#x1D54E;", "&#x1D54F;", "&#x1D550;", "&#x1D552;", 
"&#x1D553;", "&#x1D554;", "&#x1D555;", "&#x1D556;", "&#x1D557;", 
"&#x1D558;", "&#x1D559;", "&#x1D55A;", "&#x1D55B;", "&#x1D55C;", 
"&#x1D55D;", "&#x1D55E;", "&#x1D55F;", "&#x1D560;", "&#x1D561;", 
"&#x1D562;", "&#x1D563;", "&#x1D564;", "&#x1D565;", "&#x1D566;", 
"&#x1D567;", "&#x1D568;", "&#x1D569;", "&#x1D56A;", "&#x1D56B;", 
"&#x1D6AA;", "&#x1D6AB;", "&#x1D6AF;", "&#x1D6B2;", "&#x1D6B5;", 
"&#x1D6B7;", "&#x1D6BA;", "&#x1D6BC;", "&#x1D6BD;", "&#x1D6BF;", 
"&#x1D6C0;", "&#x1D6C2;", "&#x1D6C3;", "&#x1D6C4;", "&#x1D6C5;", 
"&#x1D6C6;", "&#x1D6C7;", "&#x1D6C8;", "&#x1D6C9;", "&#x1D6CA;", 
"&#x1D6CB;", "&#x1D6CC;", "&#x1D6CD;", "&#x1D6CE;", "&#x1D6CF;", 
"&#x1D6D1;", "&#x1D6D2;", "&#x1D6D3;", "&#x1D6D4;", "&#x1D6D5;", 
"&#x1D6D6;", "&#x1D6D7;", "&#x1D6D8;", "&#x1D6D9;", "&#x1D6DA;", 
"&#x1D6DC;", "&#x1D6DD;", "&#x1D6DE;", "&#x1D6DF;", "&#x1D6E0;", 
"&#x1D6E1;", "&#x1D7CA;", "&#x1D7CB;"), dec = c("&#9;", "&#10;", 
"&#33;", "&#34;", "&#34;", "&#35;", "&#36;", "&#37;", "&#38;", 
"&#38;", "&#39;", "&#40;", "&#41;", "&#42;", "&#42;", "&#43;", 
"&#44;", "&#46;", "&#47;", "&#58;", "&#59;", "&#60;", "&#60;", 
"&#60;", "&#8402;", "&#61;", "&#61;", "&#8421;", "&#62;", "&#62;", 
"&#62;", "&#8402;", "&#63;", "&#64;", "&#91;", "&#91;", "&#92;", 
"&#93;", "&#93;", "&#94;", "&#95;", "&#95;", "&#96;", "&#96;", 
"&#102;", "&#106;", "&#123;", "&#123;", "&#124;", "&#124;", "&#124;", 
"&#125;", "&#125;", "&#160;", "&#160;", "&#161;", "&#162;", "&#163;", 
"&#164;", "&#165;", "&#166;", "&#167;", "&#168;", "&#168;", "&#168;", 
"&#168;", "&#169;", "&#169;", "&#170;", "&#171;", "&#172;", "&#173;", 
"&#174;", "&#174;", "&#174;", "&#175;", "&#175;", "&#176;", "&#177;", 
"&#177;", "&#177;", "&#178;", "&#179;", "&#180;", "&#180;", "&#181;", 
"&#182;", "&#183;", "&#183;", "&#183;", "&#184;", "&#184;", "&#185;", 
"&#186;", "&#187;", "&#188;", "&#189;", "&#189;", "&#190;", "&#191;", 
"&#192;", "&#193;", "&#194;", "&#195;", "&#196;", "&#197;", "&#197;", 
"&#198;", "&#199;", "&#200;", "&#201;", "&#202;", "&#203;", "&#204;", 
"&#205;", "&#206;", "&#207;", "&#208;", "&#209;", "&#210;", "&#211;", 
"&#212;", "&#213;", "&#214;", "&#215;", "&#216;", "&#217;", "&#218;", 
"&#219;", "&#220;", "&#221;", "&#222;", "&#223;", "&#224;", "&#225;", 
"&#226;", "&#227;", "&#228;", "&#229;", "&#230;", "&#231;", "&#232;", 
"&#233;", "&#234;", "&#235;", "&#236;", "&#237;", "&#238;", "&#239;", 
"&#240;", "&#241;", "&#242;", "&#243;", "&#244;", "&#245;", "&#246;", 
"&#247;", "&#247;", "&#248;", "&#249;", "&#250;", "&#251;", "&#252;", 
"&#253;", "&#254;", "&#255;", "&#256;", "&#257;", "&#258;", "&#259;", 
"&#260;", "&#261;", "&#262;", "&#263;", "&#264;", "&#265;", "&#266;", 
"&#267;", "&#268;", "&#269;", "&#270;", "&#271;", "&#272;", "&#273;", 
"&#274;", "&#275;", "&#278;", "&#279;", "&#280;", "&#281;", "&#282;", 
"&#283;", "&#284;", "&#285;", "&#286;", "&#287;", "&#288;", "&#289;", 
"&#290;", "&#292;", "&#293;", "&#294;", "&#295;", "&#296;", "&#297;", 
"&#298;", "&#299;", "&#302;", "&#303;", "&#304;", "&#305;", "&#305;", 
"&#306;", "&#307;", "&#308;", "&#309;", "&#310;", "&#311;", "&#312;", 
"&#313;", "&#314;", "&#315;", "&#316;", "&#317;", "&#318;", "&#319;", 
"&#320;", "&#321;", "&#322;", "&#323;", "&#324;", "&#325;", "&#326;", 
"&#327;", "&#328;", "&#329;", "&#330;", "&#331;", "&#332;", "&#333;", 
"&#336;", "&#337;", "&#338;", "&#339;", "&#340;", "&#341;", "&#342;", 
"&#343;", "&#344;", "&#345;", "&#346;", "&#347;", "&#348;", "&#349;", 
"&#350;", "&#351;", "&#352;", "&#353;", "&#354;", "&#355;", "&#356;", 
"&#357;", "&#358;", "&#359;", "&#360;", "&#361;", "&#362;", "&#363;", 
"&#364;", "&#365;", "&#366;", "&#367;", "&#368;", "&#369;", "&#370;", 
"&#371;", "&#372;", "&#373;", "&#374;", "&#375;", "&#376;", "&#377;", 
"&#378;", "&#379;", "&#380;", "&#381;", "&#382;", "&#402;", "&#437;", 
"&#501;", "&#567;", "&#710;", "&#711;", "&#711;", "&#728;", "&#728;", 
"&#729;", "&#729;", "&#730;", "&#731;", "&#732;", "&#732;", "&#733;", 
"&#733;", "&#785;", "&#902;", "&#904;", "&#905;", "&#906;", "&#908;", 
"&#910;", "&#911;", "&#912;", "&#913;", "&#913;", "&#914;", "&#914;", 
"&#915;", "&#915;", "&#916;", "&#916;", "&#917;", "&#917;", "&#918;", 
"&#918;", "&#919;", "&#919;", "&#920;", "&#920;", "&#921;", "&#921;", 
"&#922;", "&#922;", "&#923;", "&#923;", "&#924;", "&#924;", "&#925;", 
"&#925;", "&#926;", "&#926;", "&#927;", "&#927;", "&#928;", "&#928;", 
"&#929;", "&#929;", "&#931;", "&#931;", "&#932;", "&#932;", "&#933;", 
"&#933;", "&#934;", "&#934;", "&#935;", "&#935;", "&#936;", "&#936;", 
"&#937;", "&#937;", "&#937;", "&#938;", "&#939;", "&#940;", "&#941;", 
"&#942;", "&#943;", "&#944;", "&#945;", "&#945;", "&#946;", "&#946;", 
"&#947;", "&#947;", "&#948;", "&#948;", "&#949;", "&#949;", "&#949;", 
"&#950;", "&#950;", "&#951;", "&#951;", "&#952;", "&#952;", "&#953;", 
"&#953;", "&#954;", "&#954;", "&#955;", "&#955;", "&#956;", "&#956;", 
"&#957;", "&#957;", "&#958;", "&#958;", "&#959;", "&#959;", "&#960;", 
"&#960;", "&#961;", "&#961;", "&#962;", "&#962;", "&#962;", "&#962;", 
"&#963;", "&#963;", "&#964;", "&#964;", "&#965;", "&#965;", "&#965;", 
"&#966;", "&#966;", "&#967;", "&#967;", "&#968;", "&#968;", "&#969;", 
"&#969;", "&#970;", "&#971;", "&#972;", "&#973;", "&#974;", "&#977;", 
"&#977;", "&#977;", "&#978;", "&#978;", "&#981;", "&#981;", "&#981;", 
"&#982;", "&#982;", "&#988;", "&#989;", "&#989;", "&#1008;", 
"&#1008;", "&#1009;", "&#1009;", "&#1013;", "&#1013;", "&#1013;", 
"&#1014;", "&#1014;", "&#1025;", "&#1026;", "&#1027;", "&#1028;", 
"&#1029;", "&#1030;", "&#1031;", "&#1032;", "&#1033;", "&#1034;", 
"&#1035;", "&#1036;", "&#1038;", "&#1039;", "&#1040;", "&#1041;", 
"&#1042;", "&#1043;", "&#1044;", "&#1045;", "&#1046;", "&#1047;", 
"&#1048;", "&#1049;", "&#1050;", "&#1051;", "&#1052;", "&#1053;", 
"&#1054;", "&#1055;", "&#1056;", "&#1057;", "&#1058;", "&#1059;", 
"&#1060;", "&#1061;", "&#1062;", "&#1063;", "&#1064;", "&#1065;", 
"&#1066;", "&#1067;", "&#1068;", "&#1069;", "&#1070;", "&#1071;", 
"&#1072;", "&#1073;", "&#1074;", "&#1075;", "&#1076;", "&#1077;", 
"&#1078;", "&#1079;", "&#1080;", "&#1081;", "&#1082;", "&#1083;", 
"&#1084;", "&#1085;", "&#1086;", "&#1087;", "&#1088;", "&#1089;", 
"&#1090;", "&#1091;", "&#1092;", "&#1093;", "&#1094;", "&#1095;", 
"&#1096;", "&#1097;", "&#1098;", "&#1099;", "&#1100;", "&#1101;", 
"&#1102;", "&#1103;", "&#1105;", "&#1106;", "&#1107;", "&#1108;", 
"&#1109;", "&#1110;", "&#1111;", "&#1112;", "&#1113;", "&#1114;", 
"&#1115;", "&#1116;", "&#1118;", "&#1119;", "&#8194;", "&#8195;", 
"&#8196;", "&#8197;", "&#8199;", "&#8200;", "&#8201;", "&#8201;", 
"&#8202;", "&#8202;", "&#8203;", "&#8203;", "&#8203;", "&#8203;", 
"&#8203;", "&#8204;", "&#8205;", "&#8206;", "&#8207;", "&#8208;", 
"&#8208;", "&#8211;", "&#8212;", "&#8213;", "&#8214;", "&#8214;", 
"&#8216;", "&#8216;", "&#8217;", "&#8217;", "&#8217;", "&#8218;", 
"&#8218;", "&#8220;", "&#8220;", "&#8221;", "&#8221;", "&#8221;", 
"&#8222;", "&#8222;", "&#8224;", "&#8225;", "&#8225;", "&#8226;", 
"&#8226;", "&#8229;", "&#8230;", "&#8230;", "&#8240;", "&#8241;", 
"&#8242;", "&#8243;", "&#8244;", "&#8245;", "&#8245;", "&#8249;", 
"&#8250;", "&#8254;", "&#8254;", "&#8257;", "&#8259;", "&#8260;", 
"&#8271;", "&#8279;", "&#8287;", "&#8287;", "&#8202;", "&#8288;", 
"&#8289;", "&#8289;", "&#8290;", "&#8290;", "&#8291;", "&#8291;", 
"&#8364;", "&#8411;", "&#8411;", "&#8412;", "&#8450;", "&#8450;", 
"&#8453;", "&#8458;", "&#8459;", "&#8459;", "&#8459;", "&#8460;", 
"&#8460;", "&#8461;", "&#8461;", "&#8462;", "&#8463;", "&#8463;", 
"&#8463;", "&#8463;", "&#8464;", "&#8464;", "&#8465;", "&#8465;", 
"&#8465;", "&#8465;", "&#8466;", "&#8466;", "&#8466;", "&#8467;", 
"&#8469;", "&#8469;", "&#8470;", "&#8471;", "&#8472;", "&#8472;", 
"&#8473;", "&#8473;", "&#8474;", "&#8474;", "&#8475;", "&#8475;", 
"&#8476;", "&#8476;", "&#8476;", "&#8476;", "&#8477;", "&#8477;", 
"&#8478;", "&#8482;", "&#8482;", "&#8484;", "&#8484;", "&#8487;", 
"&#8488;", "&#8488;", "&#8489;", "&#8492;", "&#8492;", "&#8492;", 
"&#8493;", "&#8493;", "&#8495;", "&#8496;", "&#8496;", "&#8497;", 
"&#8497;", "&#8499;", "&#8499;", "&#8499;", "&#8500;", "&#8500;", 
"&#8500;", "&#8501;", "&#8501;", "&#8502;", "&#8503;", "&#8504;", 
"&#8517;", "&#8517;", "&#8518;", "&#8518;", "&#8519;", "&#8519;", 
"&#8519;", "&#8520;", "&#8520;", "&#8531;", "&#8532;", "&#8533;", 
"&#8534;", "&#8535;", "&#8536;", "&#8537;", "&#8538;", "&#8539;", 
"&#8540;", "&#8541;", "&#8542;", "&#8592;", "&#8592;", "&#8592;", 
"&#8592;", "&#8592;", "&#8593;", "&#8593;", "&#8593;", "&#8593;", 
"&#8594;", "&#8594;", "&#8594;", "&#8594;", "&#8594;", "&#8595;", 
"&#8595;", "&#8595;", "&#8595;", "&#8596;", "&#8596;", "&#8596;", 
"&#8597;", "&#8597;", "&#8597;", "&#8598;", "&#8598;", "&#8598;", 
"&#8599;", "&#8599;", "&#8599;", "&#8600;", "&#8600;", "&#8600;", 
"&#8601;", "&#8601;", "&#8601;", "&#8602;", "&#8602;", "&#8603;", 
"&#8603;", "&#8605;", "&#8605;", "&#8605;", "&#824;", "&#8606;", 
"&#8606;", "&#8607;", "&#8608;", "&#8608;", "&#8609;", "&#8610;", 
"&#8610;", "&#8611;", "&#8611;", "&#8612;", "&#8612;", "&#8613;", 
"&#8613;", "&#8614;", "&#8614;", "&#8614;", "&#8615;", "&#8615;", 
"&#8617;", "&#8617;", "&#8618;", "&#8618;", "&#8619;", "&#8619;", 
"&#8620;", "&#8620;", "&#8621;", "&#8621;", "&#8622;", "&#8622;", 
"&#8624;", "&#8624;", "&#8625;", "&#8625;", "&#8626;", "&#8627;", 
"&#8629;", "&#8630;", "&#8630;", "&#8631;", "&#8631;", "&#8634;", 
"&#8634;", "&#8635;", "&#8635;", "&#8636;", "&#8636;", "&#8636;", 
"&#8637;", "&#8637;", "&#8637;", "&#8638;", "&#8638;", "&#8638;", 
"&#8639;", "&#8639;", "&#8639;", "&#8640;", "&#8640;", "&#8640;", 
"&#8641;", "&#8641;", "&#8641;", "&#8642;", "&#8642;", "&#8642;", 
"&#8643;", "&#8643;", "&#8643;", "&#8644;", "&#8644;", "&#8644;", 
"&#8645;", "&#8645;", "&#8646;", "&#8646;", "&#8646;", "&#8647;", 
"&#8647;", "&#8648;", "&#8648;", "&#8649;", "&#8649;", "&#8650;", 
"&#8650;", "&#8651;", "&#8651;", "&#8651;", "&#8652;", "&#8652;", 
"&#8652;", "&#8653;", "&#8653;", "&#8654;", "&#8654;", "&#8655;", 
"&#8655;", "&#8656;", "&#8656;", "&#8656;", "&#8657;", "&#8657;", 
"&#8657;", "&#8658;", "&#8658;", "&#8658;", "&#8658;", "&#8659;", 
"&#8659;", "&#8659;", "&#8660;", "&#8660;", "&#8660;", "&#8660;", 
"&#8661;", "&#8661;", "&#8661;", "&#8662;", "&#8663;", "&#8664;", 
"&#8665;", "&#8666;", "&#8666;", "&#8667;", "&#8667;", "&#8669;", 
"&#8676;", "&#8676;", "&#8677;", "&#8677;", "&#8693;", "&#8693;", 
"&#8701;", "&#8702;", "&#8703;", "&#8704;", "&#8704;", "&#8705;", 
"&#8705;", "&#8706;", "&#8706;", "&#8706;", "&#824;", "&#8707;", 
"&#8707;", "&#8708;", "&#8708;", "&#8708;", "&#8709;", "&#8709;", 
"&#8709;", "&#8709;", "&#8711;", "&#8711;", "&#8712;", "&#8712;", 
"&#8712;", "&#8712;", "&#8713;", "&#8713;", "&#8713;", "&#8715;", 
"&#8715;", "&#8715;", "&#8715;", "&#8716;", "&#8716;", "&#8716;", 
"&#8719;", "&#8719;", "&#8720;", "&#8720;", "&#8721;", "&#8721;", 
"&#8722;", "&#8723;", "&#8723;", "&#8723;", "&#8724;", "&#8724;", 
"&#8726;", "&#8726;", "&#8726;", "&#8726;", "&#8726;", "&#8727;", 
"&#8728;", "&#8728;", "&#8730;", "&#8730;", "&#8733;", "&#8733;", 
"&#8733;", "&#8733;", "&#8733;", "&#8734;", "&#8735;", "&#8736;", 
"&#8736;", "&#8736;", "&#8402;", "&#8737;", "&#8737;", "&#8738;", 
"&#8739;", "&#8739;", "&#8739;", "&#8739;", "&#8740;", "&#8740;", 
"&#8740;", "&#8740;", "&#8741;", "&#8741;", "&#8741;", "&#8741;", 
"&#8741;", "&#8742;", "&#8742;", "&#8742;", "&#8742;", "&#8742;", 
"&#8743;", "&#8743;", "&#8744;", "&#8744;", "&#8745;", "&#8745;", 
"&#65024;", "&#8746;", "&#8746;", "&#65024;", "&#8747;", "&#8747;", 
"&#8748;", "&#8749;", "&#8749;", "&#8750;", "&#8750;", "&#8750;", 
"&#8751;", "&#8751;", "&#8752;", "&#8753;", "&#8754;", "&#8754;", 
"&#8755;", "&#8755;", "&#8756;", "&#8756;", "&#8756;", "&#8757;", 
"&#8757;", "&#8757;", "&#8758;", "&#8759;", "&#8759;", "&#8760;", 
"&#8760;", "&#8762;", "&#8763;", "&#8764;", "&#8764;", "&#8764;", 
"&#8764;", "&#8764;", "&#8402;", "&#8765;", "&#8765;", "&#8765;", 
"&#817;", "&#8766;", "&#8766;", "&#8766;", "&#819;", "&#8767;", 
"&#8768;", "&#8768;", "&#8768;", "&#8769;", "&#8769;", "&#8770;", 
"&#8770;", "&#8770;", "&#8770;", "&#8770;", "&#824;", "&#824;", 
"&#8771;", "&#8771;", "&#8771;", "&#8772;", "&#8772;", "&#8772;", 
"&#8773;", "&#8773;", "&#8774;", "&#8775;", "&#8775;", "&#8776;", 
"&#8776;", "&#8776;", "&#8776;", "&#8776;", "&#8776;", "&#8777;", 
"&#8777;", "&#8777;", "&#8778;", "&#8778;", "&#8779;", "&#8779;", 
"&#824;", "&#8780;", "&#8780;", "&#8781;", "&#8781;", "&#8781;", 
"&#8402;", "&#8782;", "&#8782;", "&#8782;", "&#8782;", "&#8782;", 
"&#824;", "&#824;", "&#8783;", "&#8783;", "&#8783;", "&#8783;", 
"&#8783;", "&#824;", "&#824;", "&#8784;", "&#8784;", "&#8784;", 
"&#8784;", "&#824;", "&#8785;", "&#8785;", "&#8786;", "&#8786;", 
"&#8787;", "&#8787;", "&#8788;", "&#8788;", "&#8788;", "&#8789;", 
"&#8789;", "&#8790;", "&#8790;", "&#8791;", "&#8791;", "&#8793;", 
"&#8794;", "&#8796;", "&#8796;", "&#8799;", "&#8799;", "&#8800;", 
"&#8800;", "&#8801;", "&#8801;", "&#8801;", "&#8421;", "&#8802;", 
"&#8802;", "&#8804;", "&#8804;", "&#8804;", "&#8402;", "&#8805;", 
"&#8805;", "&#8805;", "&#8805;", "&#8402;", "&#8806;", "&#8806;", 
"&#8806;", "&#8806;", "&#8806;", "&#824;", "&#824;", "&#8807;", 
"&#8807;", "&#8807;", "&#8807;", "&#8807;", "&#8807;", "&#824;", 
"&#824;", "&#824;", "&#8808;", "&#8808;", "&#8808;", "&#8808;", 
"&#65024;", "&#65024;", "&#8809;", "&#8809;", "&#8809;", "&#8809;", 
"&#65024;", "&#65024;", "&#8810;", "&#8810;", "&#8810;", "&#8810;", 
"&#8810;", "&#824;", "&#824;", "&#8810;", "&#8402;", "&#8811;", 
"&#8811;", "&#8811;", "&#8811;", "&#8811;", "&#824;", "&#824;", 
"&#8811;", "&#8402;", "&#8812;", "&#8812;", "&#8813;", "&#8814;", 
"&#8814;", "&#8814;", "&#8815;", "&#8815;", "&#8815;", "&#8816;", 
"&#8816;", "&#8816;", "&#8817;", "&#8817;", "&#8817;", "&#8818;", 
"&#8818;", "&#8818;", "&#8819;", "&#8819;", "&#8819;", "&#8820;", 
"&#8820;", "&#8821;", "&#8821;", "&#8822;", "&#8822;", "&#8822;", 
"&#8823;", "&#8823;", "&#8823;", "&#8824;", "&#8824;", "&#8825;", 
"&#8825;", "&#8826;", "&#8826;", "&#8826;", "&#8827;", "&#8827;", 
"&#8827;", "&#8828;", "&#8828;", "&#8828;", "&#8829;", "&#8829;", 
"&#8829;", "&#8830;", "&#8830;", "&#8830;", "&#8831;", "&#8831;", 
"&#8831;", "&#8831;", "&#824;", "&#8832;", "&#8832;", "&#8832;", 
"&#8833;", "&#8833;", "&#8833;", "&#8834;", "&#8834;", "&#8834;", 
"&#8834;", "&#8834;", "&#8402;", "&#8402;", "&#8402;", "&#8835;", 
"&#8835;", "&#8835;", "&#8835;", "&#8835;", "&#8835;", "&#8402;", 
"&#8402;", "&#8402;", "&#8836;", "&#8837;", "&#8838;", "&#8838;", 
"&#8838;", "&#8839;", "&#8839;", "&#8839;", "&#8840;", "&#8840;", 
"&#8840;", "&#8841;", "&#8841;", "&#8841;", "&#8842;", "&#8842;", 
"&#8842;", "&#8842;", "&#65024;", "&#65024;", "&#8843;", "&#8843;", 
"&#8843;", "&#8843;", "&#65024;", "&#65024;", "&#8845;", "&#8846;", 
"&#8846;", "&#8847;", "&#8847;", "&#8847;", "&#8847;", "&#824;", 
"&#8848;", "&#8848;", "&#8848;", "&#8848;", "&#824;", "&#8849;", 
"&#8849;", "&#8849;", "&#8850;", "&#8850;", "&#8850;", "&#8851;", 
"&#8851;", "&#8851;", "&#65024;", "&#8852;", "&#8852;", "&#8852;", 
"&#65024;", "&#8853;", "&#8853;", "&#8854;", "&#8854;", "&#8855;", 
"&#8855;", "&#8856;", "&#8857;", "&#8857;", "&#8858;", "&#8858;", 
"&#8859;", "&#8859;", "&#8861;", "&#8861;", "&#8862;", "&#8862;", 
"&#8863;", "&#8863;", "&#8864;", "&#8864;", "&#8865;", "&#8865;", 
"&#8866;", "&#8866;", "&#8867;", "&#8867;", "&#8868;", "&#8868;", 
"&#8869;", "&#8869;", "&#8869;", "&#8869;", "&#8871;", "&#8872;", 
"&#8872;", "&#8873;", "&#8874;", "&#8875;", "&#8876;", "&#8877;", 
"&#8878;", "&#8879;", "&#8880;", "&#8882;", "&#8882;", "&#8882;", 
"&#8883;", "&#8883;", "&#8883;", "&#8884;", "&#8884;", "&#8884;", 
"&#8884;", "&#8402;", "&#8885;", "&#8885;", "&#8885;", "&#8885;", 
"&#8402;", "&#8886;", "&#8887;", "&#8888;", "&#8888;", "&#8889;", 
"&#8890;", "&#8890;", "&#8891;", "&#8893;", "&#8894;", "&#8895;", 
"&#8896;", "&#8896;", "&#8896;", "&#8897;", "&#8897;", "&#8897;", 
"&#8898;", "&#8898;", "&#8898;", "&#8899;", "&#8899;", "&#8899;", 
"&#8900;", "&#8900;", "&#8900;", "&#8901;", "&#8902;", "&#8902;", 
"&#8903;", "&#8903;", "&#8904;", "&#8905;", "&#8906;", "&#8907;", 
"&#8907;", "&#8908;", "&#8908;", "&#8909;", "&#8909;", "&#8910;", 
"&#8910;", "&#8911;", "&#8911;", "&#8912;", "&#8912;", "&#8913;", 
"&#8913;", "&#8914;", "&#8915;", "&#8916;", "&#8916;", "&#8917;", 
"&#8918;", "&#8918;", "&#8919;", "&#8919;", "&#8920;", "&#8920;", 
"&#824;", "&#8921;", "&#8921;", "&#8921;", "&#824;", "&#8922;", 
"&#8922;", "&#8922;", "&#8922;", "&#65024;", "&#8923;", "&#8923;", 
"&#8923;", "&#8923;", "&#65024;", "&#8926;", "&#8926;", "&#8927;", 
"&#8927;", "&#8928;", "&#8928;", "&#8929;", "&#8929;", "&#8930;", 
"&#8930;", "&#8931;", "&#8931;", "&#8934;", "&#8935;", "&#8936;", 
"&#8936;", "&#8937;", "&#8937;", "&#8938;", "&#8938;", "&#8938;", 
"&#8939;", "&#8939;", "&#8939;", "&#8940;", "&#8940;", "&#8940;", 
"&#8941;", "&#8941;", "&#8941;", "&#8942;", "&#8943;", "&#8944;", 
"&#8945;", "&#8946;", "&#8947;", "&#8948;", "&#8949;", "&#8949;", 
"&#824;", "&#8950;", "&#8951;", "&#8953;", "&#8953;", "&#824;", 
"&#8954;", "&#8955;", "&#8956;", "&#8957;", "&#8958;", "&#8965;", 
"&#8965;", "&#8966;", "&#8966;", "&#8968;", "&#8968;", "&#8969;", 
"&#8969;", "&#8970;", "&#8970;", "&#8971;", "&#8971;", "&#8972;", 
"&#8973;", "&#8974;", "&#8975;", "&#8976;", "&#8978;", "&#8979;", 
"&#8981;", "&#8982;", "&#8988;", "&#8988;", "&#8989;", "&#8989;", 
"&#8990;", "&#8990;", "&#8991;", "&#8991;", "&#8994;", "&#8994;", 
"&#8995;", "&#8995;", "&#9005;", "&#9006;", "&#9014;", "&#9021;", 
"&#9023;", "&#9084;", "&#9136;", "&#9136;", "&#9137;", "&#9137;", 
"&#9140;", "&#9140;", "&#9141;", "&#9141;", "&#9142;", "&#9180;", 
"&#9181;", "&#9182;", "&#9183;", "&#9186;", "&#9191;", "&#9251;", 
"&#9416;", "&#9416;", "&#9472;", "&#9472;", "&#9474;", "&#9484;", 
"&#9488;", "&#9492;", "&#9496;", "&#9500;", "&#9508;", "&#9516;", 
"&#9524;", "&#9532;", "&#9552;", "&#9553;", "&#9554;", "&#9555;", 
"&#9556;", "&#9557;", "&#9558;", "&#9559;", "&#9560;", "&#9561;", 
"&#9562;", "&#9563;", "&#9564;", "&#9565;", "&#9566;", "&#9567;", 
"&#9568;", "&#9569;", "&#9570;", "&#9571;", "&#9572;", "&#9573;", 
"&#9574;", "&#9575;", "&#9576;", "&#9577;", "&#9578;", "&#9579;", 
"&#9580;", "&#9600;", "&#9604;", "&#9608;", "&#9617;", "&#9618;", 
"&#9619;", "&#9633;", "&#9633;", "&#9633;", "&#9642;", "&#9642;", 
"&#9642;", "&#9642;", "&#9643;", "&#9645;", "&#9646;", "&#9649;", 
"&#9651;", "&#9651;", "&#9652;", "&#9652;", "&#9653;", "&#9653;", 
"&#9656;", "&#9656;", "&#9657;", "&#9657;", "&#9661;", "&#9661;", 
"&#9662;", "&#9662;", "&#9663;", "&#9663;", "&#9666;", "&#9666;", 
"&#9667;", "&#9667;", "&#9674;", "&#9674;", "&#9675;", "&#9708;", 
"&#9711;", "&#9711;", "&#9720;", "&#9721;", "&#9722;", "&#9723;", 
"&#9724;", "&#9733;", "&#9733;", "&#9734;", "&#9742;", "&#9792;", 
"&#9794;", "&#9824;", "&#9824;", "&#9827;", "&#9827;", "&#9829;", 
"&#9829;", "&#9830;", "&#9830;", "&#9834;", "&#9837;", "&#9838;", 
"&#9838;", "&#9839;", "&#10003;", "&#10003;", "&#10007;", "&#10016;", 
"&#10016;", "&#10038;", "&#10072;", "&#10098;", "&#10099;", "&#10184;", 
"&#10185;", "&#10214;", "&#10214;", "&#10215;", "&#10215;", "&#10216;", 
"&#10216;", "&#10216;", "&#10217;", "&#10217;", "&#10217;", "&#10218;", 
"&#10219;", "&#10220;", "&#10221;", "&#10229;", "&#10229;", "&#10229;", 
"&#10230;", "&#10230;", "&#10230;", "&#10231;", "&#10231;", "&#10231;", 
"&#10232;", "&#10232;", "&#10232;", "&#10233;", "&#10233;", "&#10233;", 
"&#10234;", "&#10234;", "&#10234;", "&#10236;", "&#10236;", "&#10239;", 
"&#10498;", "&#10499;", "&#10500;", "&#10501;", "&#10508;", "&#10509;", 
"&#10509;", "&#10510;", "&#10511;", "&#10511;", "&#10512;", "&#10512;", 
"&#10513;", "&#10514;", "&#10515;", "&#10518;", "&#10521;", "&#10522;", 
"&#10523;", "&#10524;", "&#10525;", "&#10526;", "&#10527;", "&#10528;", 
"&#10531;", "&#10532;", "&#10533;", "&#10533;", "&#10534;", "&#10534;", 
"&#10535;", "&#10536;", "&#10536;", "&#10537;", "&#10537;", "&#10538;", 
"&#10547;", "&#10547;", "&#824;", "&#10549;", "&#10550;", "&#10551;", 
"&#10552;", "&#10553;", "&#10556;", "&#10557;", "&#10565;", "&#10568;", 
"&#10569;", "&#10570;", "&#10571;", "&#10574;", "&#10575;", "&#10576;", 
"&#10577;", "&#10578;", "&#10579;", "&#10580;", "&#10581;", "&#10582;", 
"&#10583;", "&#10584;", "&#10585;", "&#10586;", "&#10587;", "&#10588;", 
"&#10589;", "&#10590;", "&#10591;", "&#10592;", "&#10593;", "&#10594;", 
"&#10595;", "&#10596;", "&#10597;", "&#10598;", "&#10599;", "&#10600;", 
"&#10601;", "&#10602;", "&#10603;", "&#10604;", "&#10605;", "&#10606;", 
"&#10606;", "&#10607;", "&#10607;", "&#10608;", "&#10609;", "&#10610;", 
"&#10611;", "&#10612;", "&#10613;", "&#10614;", "&#10616;", "&#10617;", 
"&#10619;", "&#10620;", "&#10621;", "&#10622;", "&#10623;", "&#10629;", 
"&#10630;", "&#10635;", "&#10636;", "&#10637;", "&#10638;", "&#10639;", 
"&#10640;", "&#10641;", "&#10642;", "&#10643;", "&#10644;", "&#10645;", 
"&#10646;", "&#10650;", "&#10652;", "&#10653;", "&#10660;", "&#10661;", 
"&#10662;", "&#10663;", "&#10664;", "&#10665;", "&#10666;", "&#10667;", 
"&#10668;", "&#10669;", "&#10670;", "&#10671;", "&#10672;", "&#10673;", 
"&#10674;", "&#10675;", "&#10676;", "&#10677;", "&#10678;", "&#10679;", 
"&#10681;", "&#10683;", "&#10684;", "&#10686;", "&#10687;", "&#10688;", 
"&#10689;", "&#10690;", "&#10691;", "&#10692;", "&#10693;", "&#10697;", 
"&#10701;", "&#10702;", "&#10703;", "&#10703;", "&#824;", "&#10704;", 
"&#10704;", "&#824;", "&#10716;", "&#10717;", "&#10718;", "&#10723;", 
"&#10724;", "&#10725;", "&#10731;", "&#10731;", "&#10740;", "&#10742;", 
"&#10752;", "&#10752;", "&#10753;", "&#10753;", "&#10754;", "&#10754;", 
"&#10756;", "&#10756;", "&#10758;", "&#10758;", "&#10764;", "&#10764;", 
"&#10765;", "&#10768;", "&#10769;", "&#10770;", "&#10771;", "&#10772;", 
"&#10773;", "&#10774;", "&#10775;", "&#10786;", "&#10787;", "&#10788;", 
"&#10789;", "&#10790;", "&#10791;", "&#10793;", "&#10794;", "&#10797;", 
"&#10798;", "&#10799;", "&#10800;", "&#10801;", "&#10803;", "&#10804;", 
"&#10805;", "&#10806;", "&#10807;", "&#10808;", "&#10809;", "&#10810;", 
"&#10811;", "&#10812;", "&#10812;", "&#10815;", "&#10816;", "&#10818;", 
"&#10819;", "&#10820;", "&#10821;", "&#10822;", "&#10823;", "&#10824;", 
"&#10825;", "&#10826;", "&#10827;", "&#10828;", "&#10829;", "&#10832;", 
"&#10835;", "&#10836;", "&#10837;", "&#10838;", "&#10839;", "&#10840;", 
"&#10842;", "&#10843;", "&#10844;", "&#10845;", "&#10847;", "&#10854;", 
"&#10858;", "&#10861;", "&#10861;", "&#824;", "&#10862;", "&#10863;", 
"&#10864;", "&#10864;", "&#824;", "&#10865;", "&#10866;", "&#10867;", 
"&#10868;", "&#10869;", "&#10871;", "&#10871;", "&#10872;", "&#10873;", 
"&#10874;", "&#10875;", "&#10876;", "&#10877;", "&#10877;", "&#10877;", 
"&#10877;", "&#10877;", "&#10877;", "&#824;", "&#824;", "&#824;", 
"&#10878;", "&#10878;", "&#10878;", "&#10878;", "&#10878;", "&#10878;", 
"&#824;", "&#824;", "&#824;", "&#10879;", "&#10880;", "&#10881;", 
"&#10882;", "&#10883;", "&#10884;", "&#10885;", "&#10885;", "&#10886;", 
"&#10886;", "&#10887;", "&#10887;", "&#10888;", "&#10888;", "&#10889;", 
"&#10889;", "&#10890;", "&#10890;", "&#10891;", "&#10891;", "&#10892;", 
"&#10892;", "&#10893;", "&#10894;", "&#10895;", "&#10896;", "&#10897;", 
"&#10898;", "&#10899;", "&#10900;", "&#10901;", "&#10901;", "&#10902;", 
"&#10902;", "&#10903;", "&#10904;", "&#10905;", "&#10906;", "&#10909;", 
"&#10910;", "&#10911;", "&#10912;", "&#10913;", "&#10913;", "&#824;", 
"&#10914;", "&#10914;", "&#824;", "&#10916;", "&#10917;", "&#10918;", 
"&#10919;", "&#10920;", "&#10921;", "&#10922;", "&#10923;", "&#10924;", 
"&#10924;", "&#65024;", "&#10925;", "&#10925;", "&#65024;", "&#10926;", 
"&#10927;", "&#10927;", "&#10927;", "&#10927;", "&#10927;", "&#10927;", 
"&#824;", "&#824;", "&#824;", "&#10928;", "&#10928;", "&#10928;", 
"&#10928;", "&#10928;", "&#10928;", "&#824;", "&#824;", "&#824;", 
"&#10931;", "&#10932;", "&#10933;", "&#10933;", "&#10934;", "&#10934;", 
"&#10935;", "&#10935;", "&#10936;", "&#10936;", "&#10937;", "&#10937;", 
"&#10938;", "&#10938;", "&#10939;", "&#10940;", "&#10941;", "&#10942;", 
"&#10943;", "&#10944;", "&#10945;", "&#10946;", "&#10947;", "&#10948;", 
"&#10949;", "&#10949;", "&#10949;", "&#10949;", "&#824;", "&#824;", 
"&#10950;", "&#10950;", "&#10950;", "&#10950;", "&#824;", "&#824;", 
"&#10951;", "&#10952;", "&#10955;", "&#10955;", "&#10955;", "&#10955;", 
"&#65024;", "&#65024;", "&#10956;", "&#10956;", "&#10956;", "&#10956;", 
"&#65024;", "&#65024;", "&#10959;", "&#10960;", "&#10961;", "&#10962;", 
"&#10963;", "&#10964;", "&#10965;", "&#10966;", "&#10967;", "&#10968;", 
"&#10969;", "&#10970;", "&#10971;", "&#10980;", "&#10980;", "&#10982;", 
"&#10983;", "&#10984;", "&#10985;", "&#10987;", "&#10988;", "&#10989;", 
"&#10990;", "&#10991;", "&#10992;", "&#10993;", "&#10994;", "&#10995;", 
"&#11005;", "&#11005;", "&#8421;", "&#64256;", "&#64257;", "&#64258;", 
"&#64259;", "&#64260;", "&#119964;", "&#119966;", "&#119967;", 
"&#119970;", "&#119973;", "&#119974;", "&#119977;", "&#119978;", 
"&#119979;", "&#119980;", "&#119982;", "&#119983;", "&#119984;", 
"&#119985;", "&#119986;", "&#119987;", "&#119988;", "&#119989;", 
"&#119990;", "&#119991;", "&#119992;", "&#119993;", "&#119995;", 
"&#119997;", "&#119998;", "&#119999;", "&#120000;", "&#120001;", 
"&#120002;", "&#120003;", "&#120005;", "&#120006;", "&#120007;", 
"&#120008;", "&#120009;", "&#120010;", "&#120011;", "&#120012;", 
"&#120013;", "&#120014;", "&#120015;", "&#120068;", "&#120069;", 
"&#120071;", "&#120072;", "&#120073;", "&#120074;", "&#120077;", 
"&#120078;", "&#120079;", "&#120080;", "&#120081;", "&#120082;", 
"&#120083;", "&#120084;", "&#120086;", "&#120087;", "&#120088;", 
"&#120089;", "&#120090;", "&#120091;", "&#120092;", "&#120094;", 
"&#120095;", "&#120096;", "&#120097;", "&#120098;", "&#120099;", 
"&#120100;", "&#120101;", "&#120102;", "&#120103;", "&#120104;", 
"&#120105;", "&#120106;", "&#120107;", "&#120108;", "&#120109;", 
"&#120110;", "&#120111;", "&#120112;", "&#120113;", "&#120114;", 
"&#120115;", "&#120116;", "&#120117;", "&#120118;", "&#120119;", 
"&#120120;", "&#120121;", "&#120123;", "&#120124;", "&#120125;", 
"&#120126;", "&#120128;", "&#120129;", "&#120130;", "&#120131;", 
"&#120132;", "&#120134;", "&#120138;", "&#120139;", "&#120140;", 
"&#120141;", "&#120142;", "&#120143;", "&#120144;", "&#120146;", 
"&#120147;", "&#120148;", "&#120149;", "&#120150;", "&#120151;", 
"&#120152;", "&#120153;", "&#120154;", "&#120155;", "&#120156;", 
"&#120157;", "&#120158;", "&#120159;", "&#120160;", "&#120161;", 
"&#120162;", "&#120163;", "&#120164;", "&#120165;", "&#120166;", 
"&#120167;", "&#120168;", "&#120169;", "&#120170;", "&#120171;", 
"&#120490;", "&#120491;", "&#120495;", "&#120498;", "&#120501;", 
"&#120503;", "&#120506;", "&#120508;", "&#120509;", "&#120511;", 
"&#120512;", "&#120514;", "&#120515;", "&#120516;", "&#120517;", 
"&#120518;", "&#120519;", "&#120520;", "&#120521;", "&#120522;", 
"&#120523;", "&#120524;", "&#120525;", "&#120526;", "&#120527;", 
"&#120529;", "&#120530;", "&#120531;", "&#120532;", "&#120533;", 
"&#120534;", "&#120535;", "&#120536;", "&#120537;", "&#120538;", 
"&#120540;", "&#120541;", "&#120542;", "&#120543;", "&#120544;", 
"&#120545;", "&#120778;", "&#120779;")), .Names = c("title", 
"unicode", "named", "hex", "dec"), row.names = c(NA, -2330L), class = "data.frame")
