% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ggnet2_network_plot.R
\name{ggnet2_network_plot}
\alias{ggnet2_network_plot}
\title{Visualize the posterior similarity matrix (i.e., posterior probability matrix)}
\usage{
ggnet2_network_plot(
  .matrix_graph,
  .subject_names = vector(),
  .subject_class_names = vector(),
  .class_colors,
  .class_shapes,
  .random_seed = 7,
  .node_size = 6,
  .add_node_labels = TRUE
)
}
\arguments{
\item{.matrix_graph}{Matrix: a matrix M where each element Mij corresponds to the posterior
probability that the (i)th subject and the (j)th subject are in the same cluster.}

\item{.subject_names}{Vector of characters: an optional vector of subject names that will appear in the graph plot.}

\item{.subject_class_names}{Vector of characters: an optional vector of class names corresponding to each subject (i.e. vertex in the graph)
which influences each vertex's color and shape. For example, the subject class names can be the true label
(for the purpose of research) or it can be any other label that analyst chooses.}

\item{.class_colors}{Named vector of characters: an optional named vector of colors that
correspond to each unique value in \code{.subject_class_names}. The vector names are
required to be the unique .subject_class_names whereas the vector values are required to be the colors.}

\item{.class_shapes}{Named vector of integers: an optional named vector of shapes that correspond
to each unique value in the \code{.subject_class_names}. The vector names are required to be
the unique \code{.subject_class_names} whereas the vector values are required to be positive integers
(i.e., pch values like 15, 16, 17, and so on).}

\item{.random_seed}{Numeric: the plot uses the random layout, so set a seed for reproducibility.}

\item{.node_size}{Positive integer: the size of each node (i.e., vertex) in the graph plot.}

\item{.add_node_labels}{Boolean (i.e., TRUE or FALSE): should individual node labels be added to each node (i.e., vertex) in the graph plot?}
}
\value{
ggnet2 network plot: a network plot with respect to the undirected network given by .matrix_graph. This is used to visualize the posterior similarity matrix.
}
\description{
A function that produces a ggnet2 network plot to visualize the posterior similarity matrix (i.e., the matrix of posterior probabilities).
}
\examples{
\donttest{

# Import the tip library
library(tip)

# Choose an arbitrary random seed to generate the data
set.seed(4*8*15*16*23*42)

# Generate a symmetric posterior probability matrix
# Each element is the probability that the two subjects belong
# to the same cluster
n1 <- 10
posterior_prob_matrix <- matrix(NA, nrow = n1, ncol = n1)
for(i in 1:n1){
  for(j in i:n1){
    if(i != j){
      posterior_prob_matrix[i,j] <- runif(n=1,min=0,max=1)
      posterior_prob_matrix[j,i] <- posterior_prob_matrix[i,j]
    }else{
      posterior_prob_matrix[i,j] <- 1.0
    }
  }
}

# --- BEGIN GRAPH PLOT 1: NO COLORS OR NODE LABELS ---

# Set an arbitrary random seed
random_seed <- 815

# Set add_node_labels to FALSE
add_node_labels <- FALSE

# Set the node size
node_size <- 6

# Construct the graph plot
ggnet2_network_plot(.matrix_graph = posterior_prob_matrix,
                    .subject_names = vector(),
                    .subject_class_names = vector(),
                    .class_colors = vector(),
                    .class_shapes = vector(),
                    .random_seed = random_seed,
                    .node_size = node_size,
                    .add_node_labels = add_node_labels)

# --- END GRAPH PLOT 1: NO COLORS OR NODE LABELS ---

# --- BEGIN GRAPH PLOT 2: NO COLORS, BUT ADD NODE LABELS ---

# Set an arbitrary random seed
random_seed <- 815

# Add node labels to the plot
add_node_labels <- TRUE

# Set graph nodes (i.e. subject identifier) a larger size
node_size <- 6

# Add subject names to the plot
subject_names <- paste("S", 1:n1, sep = "")

# Construct the graph plot
ggnet2_network_plot(.matrix_graph = posterior_prob_matrix,
                    .subject_names = subject_names,
                    .subject_class_names = vector(),
                    .class_colors = vector(),
                    .class_shapes = vector(),
                    .random_seed = random_seed,
                    .node_size = 6,
                    .add_node_labels = TRUE)

# --- END GRAPH PLOT 2: NO COLORS, BUT ADD NODE LABELS ---


# --- BEGIN GRAPH PLOT 3: ADD COLORS AND NODE LABELS ---

# Set an arbitrary random seed
random_seed <- 815

# Add node labels to the plot
add_node_labels <- TRUE

# Set graph nodes (i.e. subject identifier) a larger size
node_size <- 10

# Add subject names to the plot
subject_names <- paste("S", 1:n1, sep = "")

# Create a vector of class labels
subject_class_names <- c("Class2","Class2","Class1","Class2","Class1",
                              "Class1","Class2","Class1","Class1","Class2")

# Assign a color to each class; this can be a character value or a hex value
class_colors <- c("Class1" = "skyblue", "Class2" = "#FF9999")

# Assign a pch integer value to each class
class_shapes <- c("Class1" = 16, "Class2" = 17)

# Generate the plot
ggnet2_network_plot(.matrix_graph = posterior_prob_matrix,
                    .subject_names = subject_names,
                    .subject_class_names = subject_class_names,
                    .class_colors = class_colors,
                    .class_shapes = class_shapes,
                    .random_seed = random_seed,
                    .node_size = node_size,
                    .add_node_labels = add_node_labels)

# --- END GRAPH PLOT 3: ADD COLORS AND NODE LABELS ---
}
}
