% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rulelist.R
\name{calculate.rulelist}
\alias{calculate.rulelist}
\title{\code{calculate} metrics for a \link{rulelist}}
\usage{
\method{calculate}{rulelist}(x, metrics_to_exclude = NULL, ...)
}
\arguments{
\item{x}{A \link{rulelist}}

\item{metrics_to_exclude}{(character vector) Names of metrics to exclude}

\item{...}{Named list of custom metrics. See 'details'.}
}
\value{
A dataframe of metrics with a \code{rule_nbr} column.
}
\description{
Computes some metrics (based on \code{estimation_type}) in cumulative
window function style over the rulelist (in the same order) ignoring the
keys.
}
\details{
\subsection{Default Metrics}{

These metrics are calculated by default:
\itemize{
\item \code{cumulative_coverage}: For nth rule in the rulelist, number of distinct \code{row_nbr}s (of \code{new_data}) covered by nth and all preceding rules (in order). In weighted case, we sum the weights corresponding to the distinct \code{row_nbr}s.
\item \code{cumulative_overlap}: Up til nth rule in the rulelist, number of distinct \code{row_nbr}s (of \code{new_data}) already covered by some preceding rule (in order). In weighted case, we sum the weights corresponding to the distinct \code{row_nbr}s.
}

For classification:
\itemize{
\item \code{cumulative_accuracy}: For nth rule in the rulelist, fraction of \code{row_nbr}s such that \code{RHS} matches the \code{y_name} column (of \code{new_data}) by nth and all preceding rules (in order). In weighted case, weighted accuracy is computed.
}

For regression:
\itemize{
\item \code{cumulative_RMSE}: For nth rule in the rulelist, weighted RMSE of all predictions (\code{RHS}) predicted by nth rule and all preceding rules.
}
}

\subsection{Custom metrics}{

Custom metrics to be computed should be passed a named list of function(s) in
\code{...}. The custom metric function should take these arguments in same order:
\code{rulelist}, \code{new_data}, \code{y_name}, \code{weight}. The custom metric function should
return a numeric vector of same length as the number of rows of rulelist.
}
}
\examples{
library("magrittr")
model_c5  = C50::C5.0(Attrition ~., data = modeldata::attrition, rules = TRUE)
tidy_c5   = tidy(model_c5) \%>\%
            set_validation_data(modeldata::attrition, "Attrition") \%>\%
            set_keys(NULL)

# calculate default metrics (classification)
calculate(tidy_c5)

model_rpart = rpart::rpart(MonthlyIncome ~., data = modeldata::attrition)
tidy_rpart  =
  tidy(model_rpart) \%>\%
  set_validation_data(modeldata::attrition, "MonthlyIncome") \%>\%
  set_keys(NULL)

# calculate default metrics (regression)
calculate(tidy_rpart)

# calculate default metrics with a custom metric
#' custom function to get cumulative MAE
library("tidytable")
get_cumulative_MAE = function(rulelist, new_data, y_name, weight){

  priority_df =
    rulelist \%>\%
    select(rule_nbr) \%>\%
    mutate(priority = 1:nrow(rulelist)) \%>\%
    select(rule_nbr, priority)

  pred_df =
    predict(rulelist, new_data) \%>\%
    left_join(priority_df, by = "rule_nbr") \%>\%
    mutate(weight = local(weight)) \%>\%
    select(rule_nbr, row_nbr, weight, priority)

  new_data2 =
    new_data \%>\%
    mutate(row_nbr = 1:n()) \%>\%
    select(all_of(c("row_nbr", y_name)))

  rmse_till_rule = function(rn){

    if (is.character(rulelist$RHS)) {
      inter_df =
        pred_df \%>\%
        tidytable::filter(priority <= rn) \%>\%
        left_join(mutate(new_data, row_nbr = 1:n()), by = "row_nbr") \%>\%
        left_join(select(rulelist, rule_nbr, RHS), by = "rule_nbr") \%>\%
        nest(.by = c("RHS", "rule_nbr", "row_nbr", "priority", "weight")) \%>\%
        mutate(RHS = purrr::map2_dbl(RHS,
                                     data,
                                     ~ eval(parse(text = .x), envir = .y)
                                     )
               ) \%>\%
        unnest(data)
    } else {

      inter_df =
        pred_df \%>\%
        tidytable::filter(priority <= rn) \%>\%
        left_join(new_data2, by = "row_nbr") \%>\%
        left_join(select(rulelist, rule_nbr, RHS), by = "rule_nbr")
    }

    inter_df \%>\%
      summarise(rmse = MetricsWeighted::mae(RHS,
                                             .data[[y_name]],
                                             weight,
                                             na.rm = TRUE
                                             )
                ) \%>\%
      `[[`("rmse")
  }

  res = purrr::map_dbl(1:nrow(rulelist), rmse_till_rule)
  return(res)
}

calculate(tidy_rpart,
          metrics_to_exclude = NULL,
          list("cumulative_mae" = get_cumulative_MAE)
          )

}
\seealso{
\link{rulelist}, \link{tidy}, \link[=augment.rulelist]{augment},
\link[=predict.rulelist]{predict}, \link[=calculate.rulelist]{calculate},
\link[=prune.rulelist]{prune}, \link[=reorder.rulelist]{reorder}
}
