% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mcmc-functions.R
\name{mcmc_sample_chain}
\alias{mcmc_sample_chain}
\title{Implements Markov chain Monte Carlo via repeated \code{TransitionKernel} steps.}
\usage{
mcmc_sample_chain(
  kernel = NULL,
  num_results,
  current_state,
  previous_kernel_results = NULL,
  num_burnin_steps = 0,
  num_steps_between_results = 0,
  trace_fn = NULL,
  return_final_kernel_results = FALSE,
  parallel_iterations = 10,
  seed = NULL,
  name = NULL
)
}
\arguments{
\item{kernel}{An instance of \code{tfp$mcmc$TransitionKernel} which implements one step
of the Markov chain.}

\item{num_results}{Integer number of Markov chain draws.}

\item{current_state}{\code{Tensor} or \code{list} of \code{Tensor}s representing the
current state(s) of the Markov chain(s).}

\item{previous_kernel_results}{A \code{Tensor} or a nested collection of \code{Tensor}s
representing internal calculations made within the previous call to this
function (or as returned by \code{bootstrap_results}).}

\item{num_burnin_steps}{Integer number of chain steps to take before starting to
collect results. Default value: 0 (i.e., no burn-in).}

\item{num_steps_between_results}{Integer number of chain steps between collecting
a result. Only one out of every \code{num_steps_between_samples + 1} steps is
included in the returned results.  The number of returned chain states is
still equal to \code{num_results}.  Default value: 0 (i.e., no thinning).}

\item{trace_fn}{A function that takes in the current chain state and the previous
kernel results and return a \code{Tensor} or a nested collection of \code{Tensor}s
that is then traced along with the chain state.}

\item{return_final_kernel_results}{If \code{TRUE}, then the final kernel results are
returned alongside the chain state and the trace specified by the \code{trace_fn}.}

\item{parallel_iterations}{The number of iterations allowed to run in parallel. It
must be a positive integer. See \code{tf$while_loop} for more details.}

\item{seed}{Optional, a seed for reproducible sampling.}

\item{name}{string prefixed to Ops created by this function. Default value: \code{NULL},
(i.e., "mcmc_sample_chain").}
}
\value{
list of:
\itemize{
\item checkpointable_states_and_trace: if \code{return_final_kernel_results} is
\code{TRUE}. The return value is an instance of \code{CheckpointableStatesAndTrace}.
\item all_states: if \code{return_final_kernel_results} is \code{FALSE} and \code{trace_fn} is
\code{NULL}. The return value is a \code{Tensor} or Python list of \code{Tensor}s
representing the state(s) of the Markov chain(s) at each result step. Has
same shape as input \code{current_state} but with a prepended
\code{num_results}-size dimension.
\item states_and_trace: if \code{return_final_kernel_results} is \code{FALSE} and
\code{trace_fn} is not \code{NULL}. The return value is an instance of
\code{StatesAndTrace}.
}
}
\description{
This function samples from an Markov chain at \code{current_state} and whose
stationary distribution is governed by the supplied \code{TransitionKernel}
instance (\code{kernel}).
}
\details{
This function can sample from multiple chains, in parallel. (Whether or not
there are multiple chains is dictated by the \code{kernel}.)

The \code{current_state} can be represented as a single \code{Tensor} or a \code{list} of
\code{Tensors} which collectively represent the current state.
Since MCMC states are correlated, it is sometimes desirable to produce
additional intermediate states, and then discard them, ending up with a set of
states with decreased autocorrelation.  See Owen (2017). Such "thinning"
is made possible by setting \code{num_steps_between_results > 0}. The chain then
takes \code{num_steps_between_results} extra steps between the steps that make it
into the results. The extra steps are never materialized (in calls to
\code{sess$run}), and thus do not increase memory requirements.

Warning: when setting a \code{seed} in the \code{kernel}, ensure that \code{sample_chain}'s
\code{parallel_iterations=1}, otherwise results will not be reproducible.
In addition to returning the chain state, this function supports tracing of
auxiliary variables used by the kernel. The traced values are selected by
specifying \code{trace_fn}. By default, all kernel results are traced but in the
future the default will be changed to no results being traced, so plan
accordingly. See below for some examples of this feature.
}
\section{References}{

\itemize{
\item \href{https://arxiv.org/abs/1510.07727}{Art B. Owen. Statistically efficient thinning of a Markov chain sampler. \emph{Technical Report}, 2017.}
}
}

\examples{
\dontrun{
  dims <- 10
  true_stddev <- sqrt(seq(1, 3, length.out = dims))
  likelihood <- tfd_multivariate_normal_diag(scale_diag = true_stddev)

  kernel <- mcmc_hamiltonian_monte_carlo(
    target_log_prob_fn = likelihood$log_prob,
    step_size = 0.5,
    num_leapfrog_steps = 2
  )

  states <- kernel \%>\% mcmc_sample_chain(
    num_results = 1000,
    num_burnin_steps = 500,
    current_state = rep(0, dims),
    trace_fn = NULL
  )

  sample_mean <- tf$reduce_mean(states, axis = 0L)
  sample_stddev <- tf$sqrt(
    tf$reduce_mean(tf$math$squared_difference(states, sample_mean), axis = 0L))
}
}
\seealso{
Other mcmc_functions: 
\code{\link{mcmc_effective_sample_size}()},
\code{\link{mcmc_potential_scale_reduction}()},
\code{\link{mcmc_sample_annealed_importance_chain}()},
\code{\link{mcmc_sample_halton_sequence}()}
}
\concept{mcmc_functions}
