% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/algorithms_1sided.R
\name{opt_1sided}
\alias{opt_1sided}
\alias{rna}
\alias{sga}
\alias{sgaplus}
\alias{coma}
\title{Algorithms for Optimum Sample Allocation Under One-Sided Bounds}
\usage{
rna(
  total_cost,
  A,
  bounds = NULL,
  unit_costs = 1,
  check_violations = .Primitive(">="),
  details = FALSE
)

sga(total_cost, A, M)

sgaplus(total_cost, A, M)

coma(total_cost, A, M)
}
\arguments{
\item{total_cost}{(\code{number})\cr total cost \eqn{c} of the survey. A strictly
positive scalar.}

\item{A}{(\code{numeric})\cr population constants \eqn{A_1,\ldots,A_H}. Strictly
positive numbers.}

\item{bounds}{(\code{numeric} or \code{NULL}) \cr optional lower bounds
\eqn{m_1,\ldots,m_H}, or upper bounds \eqn{M_1,\ldots,M_H}, or \code{NULL} to
indicate that there is no inequality constraints in the optimization
problem considered.
If not \code{NULL}, the \code{bounds} is to be treated either as:
\itemize{
\item lower bounds, if \code{check_violations = .Primitive("<=")}. In this case, it
is required that \code{total_cost >= sum(unit_costs * bounds)}, \cr
or
\item upper bounds, if \code{check_violations = .Primitive(">=")}. In this case, it
is required that \code{total_cost <= sum(unit_costs * bounds)}.
}}

\item{unit_costs}{(\code{numeric})\cr costs \eqn{c_1,\ldots,c_H}, of surveying one
element in stratum. A strictly positive numbers. Can be also of length 1,
if all unit costs are the same for all strata. In this case, the elements
will be recycled to the length of \code{bounds}.}

\item{check_violations}{(\code{function}) \cr 2-arguments binary operator function
that allows the comparison of values in atomic vectors. It must either be
set to \code{.Primitive("<=")} or \code{.Primitive(">=")}. The first of these choices
causes that \code{bounds} are treated as lower bounds and then \code{rna()} function
performs the \emph{LRNA} algorithm. The latter option causes that \code{bounds}
are treated as upper bounds, and then \code{rna()} function performs the
\emph{RNA} algorithm. This argument is ignored when \code{bounds} is set to \code{NULL}.}

\item{details}{(\code{flag}) \cr should detailed information about strata
assignments (either to take-Neyman or take-bound), values of set function
\eqn{s} and number of iterations be added to the output?}

\item{M}{(\code{numeric} or \code{NULL})\cr upper bounds \eqn{M_1,\ldots,M_H},
optionally imposed on sample sizes in strata. If no upper bounds should be
imposed, then \code{M} must be set to \code{NULL}. Otherwise, it is required that
\code{total_cost <= sum(unit_costs * M)}. Strictly positive numbers.}
}
\value{
Numeric vector with optimal sample allocations in strata. In case
of the \code{rna()} only, it can also be a \code{\link{list}} with optimal sample
allocations and strata assignments (either to take-Neyman or take-bound).
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#stable}{\figure{lifecycle-stable.svg}{options: alt='[Stable]'}}}{\strong{[Stable]}}

Functions that implement selected optimal allocation algorithms that compute
a solution to the optimal allocation problem defined in the language of
mathematical optimization as follows.

Minimize
\deqn{f(x_1,\ldots,x_H) = \sum_{h=1}^H \frac{A^2_h}{x_h}}
subject to
\deqn{\sum_{h=1}^H c_h x_h = c}
and either
\deqn{x_h \leq M_h, \quad h = 1,\ldots,H}
or
\deqn{x_h \geq m_h, \quad h = 1,\ldots,H,}
where
\eqn{c > 0,\, c_h > 0,\, A_h > 0,\, m_h > 0,\, M_h > 0,\, h = 1,\ldots,H},
are given numbers. The minimization is on \eqn{\mathbb R_+^H}.

The inequality constraints are optional and user can choose whether and how
they are to be added to the optimization problem.
If one-sided lower bounds \eqn{m_h,\, h = 1,\ldots,H}, must be imposed, it
is then required that \eqn{c \geq \sum_{h=1}^H c_h m_h}.
If one-sided upper bounds \eqn{M_h,\, h = 1,\ldots,H}, must be imposed, it
is then required that \eqn{0 < c \leq \sum_{h=1}^H c_h M_h}.
Lower bounds can be specified instead of the upper bounds only in case of the
\emph{LRNA} algorithm. All other algorithms allow only for specification of
the upper bounds. For the sake of clarity, we emphasize that in the
optimization problem consider here, the lower and upper bounds cannot be
imposed jointly.

Costs \eqn{c_h,\, h = 1,\ldots,H}, of surveying one element in stratum, can
be specified by the user only in case of the \emph{RNA} and \emph{LRNA}
algorithms. For remaining algorithms, these costs are fixed at 1, i.e.
\eqn{c_h = 1,\, h = 1,\ldots,H}.

The following is the list of all the algorithms available to use along with
the name of the function that implements a given algorithm. See the
description of a specific function to find out more about the corresponding
algorithm.
\itemize{
\item \emph{RNA} - \code{rna()}
\item \emph{LRNA}- \code{rna()}
\item \emph{SGA}- \code{sga()}
\item \emph{SGAPLUS} - \code{sgaplus()}
\item \emph{COMA} - \code{coma()}
}

Functions in this family should not be called directly by the user. Use
\code{\link[=opt]{opt()}} or \code{\link[=optcost]{optcost()}} instead.
}
\section{Functions}{
\itemize{
\item \code{rna()}: \emph{Recursive Neyman Algorithm} (\emph{RNA}) and its
twin version, \emph{Lower Recursive Neyman Algorithm} (\emph{LRNA})
dedicated to the allocation problem with one-sided lower-bounds constraints.
The \emph{RNA} is described in Wesołowski et al. (2021), while \emph{LRNA} is
introduced in Wójciak (2023).

\item \code{sga()}: Stenger-Gabler type algorithm \emph{SGA}, described in
Wesołowski et al. (2021) and in Stenger and Gabler (2005).
This algorithm solves the problem with one-sided upper-bounds constraints.
It also assumes unit costs are constant and equal to 1, i.e.
\eqn{c_h = 1,\, h = 1,\ldots,H}.

\item \code{sgaplus()}: modified Stenger-Gabler type algorithm, described in
Wójciak (2019) as \emph{Sequential Allocation (version 1)} algorithm.
This algorithm solves the problem with one-sided upper-bounds constraints.
It also assumes unit costs are constant and equal to 1, i.e.
\eqn{c_h = 1,\, h = 1,\ldots,H}.

\item \code{coma()}: \emph{Change of Monotonicity Algorithm} (\emph{COMA}),
described in Wesołowski et al. (2021).
This algorithm solves the problem with one-sided upper-bounds constraints.
It also assumes unit costs are constant and equal to 1, i.e.
\eqn{c_h = 1,\, h = 1,\ldots,H}.

}}
\note{
If no inequality constraints are added, the allocation is given by the
Neyman allocation as:
\deqn{x_h = \frac{A_h}{\sqrt{c_h}} \frac{n}{\sum_{i=1}^H A_i \sqrt{c_i}},
  \quad h = 1,\ldots,H.}
For \emph{stratified \eqn{\pi} estimator} of the population total with
\emph{stratified simple random sampling without replacement} design in use,
the parameters of the objective function \eqn{f} are:
\deqn{A_h = N_h S_h, \quad h = 1,\ldots,H,}
where \eqn{N_h} is the size of stratum \eqn{h} and \eqn{S_h} denotes
standard deviation of a given study variable in stratum \eqn{h}.
}
\examples{
A <- c(3000, 4000, 5000, 2000)
m <- c(50, 40, 10, 30) # lower bounds
M <- c(100, 90, 70, 80) # upper bounds

rna(total_cost = 190, A = A, bounds = M)
rna(total_cost = 190, A = A, bounds = m, check_violations = .Primitive("<="))
sga(total_cost = 190, A = A, M = M)
sgaplus(total_cost = 190, A = A, M = M)
coma(total_cost = 190, A = A, M = M)
}
\references{
Wójciak, W. (2023).
Another Solution of Some Optimum Allocation Problem.
\emph{Statistics in Transition new series}, 24(5) (in press).
\url{https://arxiv.org/abs/2204.04035} \cr

Wesołowski, J., Wieczorkowski, R., Wójciak, W. (2021).
Optimality of the Recursive Neyman Allocation.
\emph{Journal of Survey Statistics and Methodology}, 10(5), pp. 1263–1275.
\doi{10.1093/jssam/smab018},
\doi{10.48550/arXiv.2105.14486} \cr

Wójciak, W. (2019). Optimal Allocation in Stratified Sampling Schemes.
\emph{MSc Thesis}, Warsaw University of Technology, Warsaw, Poland.
\url{http://home.elka.pw.edu.pl/~wwojciak/msc_optimal_allocation.pdf} \cr

Stenger, H., Gabler, S. (2005).
Combining random sampling and census strategies -
Justification of inclusion probabilities equal to 1.
\emph{Metrika}, 61(2), pp. 137–156.
\doi{10.1007/s001840400328} \cr

Särndal, C.-E., Swensson, B. and Wretman, J. (1992).
\emph{Model Assisted Survey Sampling}, Springer, New York.
}
\seealso{
\code{\link[=opt]{opt()}}, \code{\link[=optcost]{optcost()}}, \code{\link[=rnabox]{rnabox()}}.
}
