% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/standardize_custom.R
\name{standardize}
\alias{standardize}
\title{Get standardized estimates using the g-formula with a custom model}
\usage{
standardize(
  fitter,
  arguments,
  predict_fun,
  data,
  values,
  B = NULL,
  ci_level = 0.95,
  contrasts = NULL,
  reference = NULL,
  seed = NULL,
  times = NULL,
  transforms = NULL,
  progressbar = TRUE
)
}
\arguments{
\item{fitter}{The function to call to fit the data.}

\item{arguments}{The arguments to be used in the fitter function as a \code{list}.}

\item{predict_fun}{The function used to predict the means/probabilities
for a new data set on the response level. For survival data,
this should be a matrix where each column is the time, and each
row the data.}

\item{data}{The data.}

\item{values}{A named list or data.frame specifying the variables and values
at which marginal means of the outcome will be estimated.}

\item{B}{Number of nonparametric bootstrap resamples. Default is \code{NULL} (no bootstrap).}

\item{ci_level}{Coverage probability of confidence intervals.}

\item{contrasts}{A vector of contrasts in the following format:
If set to \code{"difference"} or \code{"ratio"}, then \eqn{\psi(x)-\psi(x_0)}
or \eqn{\psi(x) / \psi(x_0)} are constructed, where \eqn{x_0} is a reference
level specified by the \code{reference} argument. Has to be \code{NULL}
if no references are specified.}

\item{reference}{A vector of reference levels in the following format:
If \code{contrasts} is not \code{NULL}, the desired reference level(s). This
must be a vector or list the same length as \code{contrasts}, and if not named,
it is assumed that the order is as specified in contrasts.}

\item{seed}{The seed to use with the nonparametric bootstrap.}

\item{times}{For use with survival data. Set to \code{NULL} otherwise.}

\item{transforms}{A vector of transforms in the following format:
If set to \code{"log"}, \code{"logit"}, or \code{"odds"}, the standardized
mean \eqn{\theta(x)} is transformed into \eqn{\psi(x)=\log\{\theta(x)\}},
\eqn{\psi(x)=\log[\theta(x)/\{1-\theta(x)\}]}, or
\eqn{\psi(x)=\theta(x)/\{1-\theta(x)\}}, respectively.
If the vector is \code{NULL}, then \eqn{\psi(x)=\theta(x)}.}

\item{progressbar}{Logical, if TRUE will print bootstrapping progress to the console}
}
\value{
An object of class \code{std_custom}. Obtain numeric results using \link{tidy.std_custom}.
This is a list with the following components:
\describe{
 \item{res_contrast}{An unnamed list with one element for each of the requested contrasts. Each element is itself a list with the elements:
 \describe{
 \item{B}{The number of bootstrap replicates}
 \item{estimates}{Estimated counterfactual means and standard errors for each exposure level}
 \item{fit_outcome}{The estimated regression model for the outcome}
 \item{estimates_boot}{A list of estimates, one for each bootstrap resample}
 \item{exposure_names}{A character vector of the exposure variable names}
 \item{times}{The vector of times at which the calculation is done, if relevant}
 \item{est_table}{Data.frame of the estimates of the contrast with inference}
 \item{transform}{The transform argument used for this contrast}
 \item{contrast}{The requested contrast type}
 \item{reference}{The reference level of the exposure}
 \item{ci_level}{Confidence interval level}
}}
\item{res}{A named list with the elements:
 \describe{
 \item{B}{The number of bootstrap replicates}
 \item{estimates}{Estimated counterfactual means and standard errors for each exposure level}
 \item{fit_outcome}{The estimated regression model for the outcome}
 \item{estimates_boot}{A list of estimates, one for each bootstrap resample}
 \item{exposure_names}{A character vector of the exposure variable names}
 \item{times}{The vector of times at which the calculation is done, if relevant}
}
}}
}
\description{
Get standardized estimates using the g-formula with a custom model
}
\details{
Let \eqn{Y}, \eqn{X}, and \eqn{Z} be the outcome, the exposure, and a
vector of covariates, respectively.
\code{standardize} uses a
model to estimate the standardized
mean \eqn{\theta(x)=E\{E(Y|X=x,Z)\}},
where \eqn{x} is a specific value of \eqn{X},
and the outer expectation is over the marginal distribution of \eqn{Z}.
With survival data, \eqn{Y=I(T > t)},
and a vector of different time points \code{times} (\eqn{t}) can be given,
where \eqn{T} is the uncensored survival time.
}
\examples{

set.seed(6)
n <- 100
Z <- rnorm(n)
X <- rnorm(n, mean = Z)
Y <- rbinom(n, 1, prob = (1 + exp(X + Z))^(-1))
dd <- data.frame(Z, X, Y)
prob_predict.glm <- function(...) predict.glm(..., type = "response")

x <- standardize(
  fitter = "glm",
  arguments = list(
    formula = Y ~ X * Z,
    family = "binomial"
  ),
  predict_fun = prob_predict.glm,
  data = dd,
  values = list(X = seq(-1, 1, 0.1)),
  B = 100,
  reference = 0,
  contrasts = "difference"
)
x

require(survival)
prob_predict.coxph <- function(object, newdata, times) {
  fit.detail <- suppressWarnings(basehaz(object))
  cum.haz <- fit.detail$hazard[sapply(times, function(x) max(which(fit.detail$time <= x)))]
  predX <- predict(object = object, newdata = newdata, type = "risk")
  res <- matrix(NA, ncol = length(times), nrow = length(predX))
  for (ti in seq_len(length(times))) {
    res[, ti] <- exp(-predX * cum.haz[ti])
  }
  res
}
set.seed(68)
n <- 500
Z <- rnorm(n)
X <- rnorm(n, mean = Z)
T <- rexp(n, rate = exp(X + Z + X * Z)) # survival time
C <- rexp(n, rate = exp(X + Z + X * Z)) # censoring time
U <- pmin(T, C) # time at risk
D <- as.numeric(T < C) # event indicator
dd <- data.frame(Z, X, U, D)
x <- standardize(
fitter = "coxph",
  arguments = list(
    formula = Surv(U, D) ~ X + Z + X * Z,
    method = "breslow",
    x = TRUE,
    y = TRUE
  ),
  predict_fun = prob_predict.coxph,
  data = dd,
  times = 1:5,
  values = list(X = c(-1, 0, 1)),
  B = 100,
  reference = 0,
  contrasts = "difference"
)
x
}
\references{
Rothman K.J., Greenland S., Lash T.L. (2008).
\emph{Modern Epidemiology}, 3rd edition.
Lippincott, Williams & Wilkins.

Sjölander A. (2016).
Regression standardization with the R-package stdReg.
\emph{European Journal of Epidemiology} \bold{31}(6), 563-574.

Sjölander A. (2016).
Estimation of causal effect measures with the R-package stdReg.
\emph{European Journal of Epidemiology} \bold{33}(9), 847-858.
}
