% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/classification.R
\name{aucroc}
\alias{aucroc}
\title{Area under the ROC curve}
\usage{
aucroc(
  actual,
  pred,
  na.rm = FALSE,
  positive = NULL,
  sample_size = 10000,
  seed = 0
)
}
\arguments{
\item{actual}{any atomic vector. Actual label values from a dataset. They must be binary; that is, there must be exactly two distinct values (other than missing values, which are allowed). The "true" or "positive" class is determined by coercing \code{actual} to logical \code{TRUE} and \code{FALSE} following the rules of \code{\link[=as.logical]{as.logical()}}. If this is not the intended meaning of "positive", then specify which of the two values should be considered \code{TRUE} with the argument \code{positive}.}

\item{pred}{numeric vector. Predictions corresponding to each respective element in \code{actual}. Any numeric value (not only probabilities) are permissible.}

\item{na.rm}{logical(1). \code{TRUE} if missing values should be removed; \code{FALSE} if they should be retained. If \code{TRUE}, then if any element of either \code{actual} or \code{pred} is missing, its paired element will be also removed.}

\item{positive}{any single atomic value. The value of \code{actual} that is considered \code{TRUE}; any other value of \code{actual} is considered \code{FALSE}. For example, if \code{2} means \code{TRUE} and \code{1} means \code{FALSE}, then set \code{positive = 2}.}

\item{sample_size}{single positive integer. To keep the computation relatively rapid, when \code{actual} and \code{pred} are longer than \code{sample_size} elements, then a random sample of \code{sample_size} of \code{actual} and \code{pred} will be selected and the ROC and AUC will be calculated on this sample. To disable random sampling for long inputs, set \code{sample_size = NA}.}

\item{seed}{numeric(1). Random seed used only if \code{length(actual) > sample_size}.}
}
\value{
List with the following elements:
\itemize{
\item \code{roc_opt}: tibble with optimistic ROC data. "Optimistic" means that when predictions are tied, the TRUE/positive actual values are ordered before the FALSE/negative ones.
\item \code{roc_pess}: tibble with pessimistic ROC data. "Pessimistic" means that when predictions are tied, the FALSE/negative actual values are ordered before the TRUE/positive ones. Note that this difference is not merely in the sort order: when there are ties, the way that true positives, true negatives, etc. are counted is different for optimistic and pessimistic approaches. If there are no tied predictions, then \code{roc_opt} and \code{roc_pess} are identical.
\item \code{auc_opt}: area under the ROC curve for optimistic ROC.
\item \code{auc_pess}: area under the ROC curve for pessimistic ROC.
\item \code{auc}: mean of \code{auc_opt} and \code{auc_pess}. If there are no tied predictions, then \code{auc_opt}, \code{auc_pess}, and \code{auc} are identical.
\item \code{ties}: \code{TRUE} if there are two or more tied predictions; \code{FALSE} if there are no ties.
}
}
\description{
Returns the area under the ROC curve based on comparing the predicted scores to the actual binary values. Tied predictions are handled by calculating the optimistic AUC (positive cases sorted first, resulting in higher AUC) and the pessimistic AUC (positive cases sorted last, resulting in lower AUC) and then returning the average of the two. For the ROC, a "tie" means at least one pair of \code{pred} predictions whose value is identical yet their corresponding values of \code{actual} are different. (If the value of \code{actual} are the same for identical predictions, then these are unproblematic and are not considered "ties".)
}
\examples{
set.seed(0)
# Generate some simulated "actual" data
a <- sample(c(TRUE, FALSE), 50, replace = TRUE)

# Generate some simulated predictions
p <- runif(50) |> round(2)
p[c(7, 8, 22, 35, 40, 41)] <- 0.5

# Calculate AUCROC with its components
ar <- aucroc(a, p)
ar$auc

}
