% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/build_lines.R
\name{build_lines}
\alias{build_lines}
\title{Build Lines}
\usage{
build_lines(
  DT = NULL,
  projection = NULL,
  id = NULL,
  coords = NULL,
  sortBy = NULL,
  splitBy = NULL
)
}
\arguments{
\item{DT}{input data.table}

\item{projection}{numeric or character defining the coordinate reference
system to be passed to \link[sf:st_crs]{sf::st_crs}. For example, either
\code{projection = "EPSG:32736"} or \code{projection = 32736}.}

\item{id}{Character string of ID column name}

\item{coords}{Character vector of X coordinate and Y coordinate column names}

\item{sortBy}{Character string of date time column(s) to sort rows by. Must
be a POSIXct.}

\item{splitBy}{(optional) character string or vector of grouping column
name(s) upon which the grouping will be calculated}
}
\value{
\code{build_lines} returns an sf LINESTRING object with a line
for each individual (and optionally \code{splitBy} combination).

Individuals (or combinations of individuals and \code{splitBy}) with less than two
relocations are dropped since it requires at least two relocations to
build a line.
}
\description{
\code{build_lines} generates a simple feature collection with LINESTRINGs from a
\code{data.table}. The function accepts a \code{data.table} with relocation data,
individual identifiers, a sorting column and a \code{projection}. The relocation
data is transformed into LINESTRINGs for each individual and, optionally,
combination of columns listed in \code{splitBy}. Relocation data should be in two
columns representing the X and Y coordinates.
}
\details{
\subsection{R-spatial evolution}{

Please note, spatsoc has followed updates from R spatial, GDAL and PROJ
for handling projections, see more at
\url{https://r-spatial.org/r/2020/03/17/wkt.html}.

In addition, \code{build_lines} previously used \link[sp:SpatialLines]{sp::SpatialLines} but has been
updated to use \link[sf:st_as_sf]{sf::st_as_sf} and \link[sf:st]{sf::st_linestring} according to the
R-spatial evolution, see more at
\url{https://r-spatial.org/r/2022/04/12/evolution.html}.
}

\subsection{Notes on arguments}{

The \code{projection} argument expects a numeric or character defining the
coordinate reference system.
For example, for UTM zone 36N (EPSG 32736), the projection argument is either
\code{projection = 'EPSG:32736'} or \code{projection = 32736}.
See details in \code{\link[sf:st_crs]{sf::st_crs()}} and \url{https://spatialreference.org}
for a list of EPSG codes.

The \code{sortBy} argument is used to order the input \code{DT} when creating
sf LINESTRINGs. It must a column in the input \code{DT} of type
POSIXct to ensure the rows are sorted by date time.

The \code{splitBy} argument offers further control building LINESTRINGs.
If in your input \code{DT}, you have multiple temporal groups (e.g.: years) for
example, you can provide the name of the column which identifies them and
build LINESTRINGs for each individual in each year.

\code{build_lines} is used by \code{group_lines} for grouping overlapping
lines generated from relocations.
}
}
\examples{
# Load data.table
library(data.table)
\dontshow{data.table::setDTthreads(1)}

# Read example data
DT <- fread(system.file("extdata", "DT.csv", package = "spatsoc"))

# Cast the character column to POSIXct
DT[, datetime := as.POSIXct(datetime, tz = 'UTC')]

# EPSG code for example data
utm <- 32736

# Build lines for each individual
lines <- build_lines(DT, projection = utm, id = 'ID', coords = c('X', 'Y'),
            sortBy = 'datetime')

# Build lines for each individual by year
DT[, yr := year(datetime)]
lines <- build_lines(DT, projection = utm, id = 'ID', coords = c('X', 'Y'),
            sortBy = 'datetime', splitBy = 'yr')
}
\seealso{
\code{\link{group_lines}}

Other Build functions: 
\code{\link{build_polys}()}
}
\concept{Build functions}
