% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sparseDFM.R
\name{sparseDFM}
\alias{sparseDFM}
\title{Estimate a Sparse Dynamic Factor Model}
\usage{
sparseDFM(
  X,
  r,
  q = 0,
  alphas = logspace(-2, 3, 100),
  alg = "EM-sparse",
  err = "IID",
  kalman = "univariate",
  store.parameters = FALSE,
  standardize = TRUE,
  max_iter = 100,
  threshold = 1e-04
)
}
\arguments{
\item{X}{\code{n x p} numeric data matrix or data frame of (stationary) time series.}

\item{r}{Integer. Number of factors.}

\item{q}{Integer. The first q series (columns of X) should not be made sparse. Default q = 0.}

\item{alphas}{Numeric vector or value of LASSO regularisation parameters. Default is alphas = logspace(-2,3,100).}

\item{alg}{Character. Option for estimation algorithm. Default is \code{"EM-sparse"}. Options are:
\tabular{llll}{
\code{"PCA"} \tab\tab principle components analysis (PCA) for static factors seen in Stock and Watson (2002). \cr\cr
\code{"2Stage"} \tab\tab the two-stage framework of PCA plus Kalman filter/smoother seen in Giannone et al. (2008) and Doz et al. (2011). \cr\cr
\code{"EM"} \tab\tab the quasi-maximum likelihood approach using the EM algorithm to handle arbitrary patterns of missing data seen in Banbura and Modugno (2014). \cr\cr
\code{"EM-sparse"} \tab\tab the novel sparse EM approach allowing LASSO regularisation on factor loadings seen in (cite our paper). \cr\cr
}}

\item{err}{Character. Option for idiosyncratic errors. Default is \code{"IID"}. Options are:
\tabular{llll}{
\code{"IID"} \tab\tab errors are IID white noise. \cr\cr
\code{"AR1"} \tab\tab errors follow an AR(1) process. \cr\cr
}}

\item{kalman}{Character. Option for Kalman filter and smoother equations. Default is \code{"univariate"}. Options are:
\tabular{llll}{
\code{"multivariate"} \tab\tab classic Kalman filter and smoother equations seen in Shumway and Stoffer (1982). \cr\cr
\code{"univaraite"} \tab\tab univariate treatment (sequential processing) of the multivariate equations for fast Kalman filter and smoother seen in Koopman and Durbin (2000). \cr\cr
}}

\item{store.parameters}{Logical. Store outputs for every alpha L1 penalty parameter. Default is FALSE.}

\item{standardize}{Logical. Standardize the data before estimating the model. Default is \code{TRUE}.}

\item{max_iter}{Integer. Maximum number of EM iterations. Default is 100.}

\item{threshold}{Numeric. Tolerance on EM iterates. Default is 1e-4.}
}
\value{
A list-of-lists-like S3 object of class 'sparseDFM' with the following elements:
\item{\code{data}}{A list containing information about the data with the following elements:
\tabular{llll}{
\code{X} \tab\tab is the original \eqn{n \times p}{n x p} numeric data matrix of (stationary) time series. \cr\cr
\code{standardize} \tab\tab is a logical value indicating whether the original data was standardized.\cr\cr
\code{X.mean} \tab\tab is a p-dimensional numeric vector of column means of \eqn{X}.  \cr\cr
\code{X.sd} \tab\tab is a p-dimensional numeric vector of column standard deviations of \eqn{X}.  \cr\cr
\code{X.bal} \tab\tab is a \eqn{n \times p}{n x p} numeric data matrix of the original \eqn{X} with missing data interpolated using \code{fillNA()}. \cr\cr
\code{eigen} \tab\tab is the eigen decomposition of \code{X.bal}. \cr\cr
\code{fitted} \tab\tab is the \eqn{n \times p}{n x p} predicted data matrix using the estimated parameters: \eqn{\hat{\Lambda}\hat{F}}{\hat{\Lambda}\hat{F}}. \cr\cr
\code{fitted.unscaled} \tab\tab is the \eqn{n \times p}{n x p} predicted data matrix using the estimated parameters: \eqn{\hat{\Lambda}\hat{F}}{\hat{\Lambda}\hat{F}} that has been unscaled back to original data scale if \code{standardize} is \code{TRUE}. \cr\cr
\code{method} \tab\tab the estimation algorithm used (\code{alg}). \cr\cr
\code{err} \tab\tab the type of idiosyncratic errors assumed. Either \code{IID} or \code{AR1}. \cr\cr
\code{call} \tab\tab call object obtained from \code{match.call()}. \cr\cr
}
}
\item{\code{params}}{A list containing the estimated parameters of the model with the following elements:
\tabular{llll}{
\code{A} \tab\tab the \eqn{r \times r}{r x r} factor transition matrix. \cr\cr
\code{Phi} \tab\tab the p-dimensional vector of AR(1) coefficients for the idiosyncratic errors. \cr\cr
\code{Lambda} \tab\tab the \eqn{p \times r}{p x r} loadings matrix. \cr\cr
\code{Sigma_u} \tab\tab the \eqn{r \times r}{r x r} factor transition error covariance matrix. \cr\cr
\code{Sigma_epsilon} \tab\tab the p-dimensional vector of idiosyncratic error variances. As \eqn{\bm{\Sigma}_{\epsilon}}{\Sigma(\epsilon)} is assumed to be diagonal. \cr\cr
}
}
\item{\code{state}}{A list containing the estimated states and state covariances with the following elements:
\tabular{llll}{
\code{factors} \tab\tab the \eqn{n \times r}{n x r} matrix of factor estimates. \cr\cr
\code{errors} \tab\tab the \eqn{n \times p}{n x p} matrix of AR(1) idiosyncratic error estimates. For err = AR1 only. \cr\cr
\code{factors.cov} \tab\tab the \eqn{r \times r \times n}{r x r x n} covariance matrices of the factor estimates. \cr\cr
\code{errors.cov} \tab\tab the \eqn{p \times p \times n}{p x p x n} covariance matrices of the AR(1) idiosyncratic error estimates. For err = AR1 only. \cr\cr
}
}
\item{\code{em}}{A list containing information about the EM algorithm with the following elements:
\tabular{llll}{
\code{converged} \tab\tab a logical value indicating whether the EM algorithm converged. \cr\cr
\code{alpha_grid} \tab\tab a numerical vector containing the LASSO tuning parameters considered in BIC evaluation before stopping. \cr\cr
\code{alpha_opt} \tab\tab the optimal LASSO tuning parameter used. \cr\cr
\code{bic} \tab\tab a numerical vector containing BIC values for the corresponding LASSO tuning parameter in \code{alpha_grid}. \cr\cr
\code{loglik} \tab\tab the log-likelihood of the innovations from the Kalman filter in the final model. \cr\cr
\code{num_iter} \tab\tab number of iterations taken by the EM algorithm. \cr\cr
\code{tol} \tab\tab tolerance for EM convergence. Matches \code{threshold} in the input. \cr\cr
\code{max_iter} \tab\tab maximum number of iterations allowed for the EM algorithm. Matches \code{max_iter} in the input. \cr\cr
\code{em_time} \tab\tab time taken for EM convergence \cr\cr
}
}
\item{\code{alpha.output}}{Parameter and state outputs for each L1-norm penalty parameter in \code{alphas} if \code{store.parameters = TRUE}.
}
}
\description{
Main function to allow estimation of a DFM or a sparse DFM (with sparse loadings) on stationary data that may have arbitrary patterns of missing data. We allow the user:
\itemize{
\item an option for estimation method - \code{"PCA"}, \code{"2Stage"}, \code{"EM"} or \code{"EM-sparse"}
\item an option for \code{IID} or \code{AR1} idiosyncratic errors
\item an option for Kalman Filter/Smoother estimation using standard \code{multivariate} equations or fast \code{univariate} filtering equations
}
}
\details{
For full details of the model please refer to Mosley et al. (2023).
}
\examples{
# load inflation data set 
data = inflation

# reduce the size for these examples - full data found in vignette 
data = data[1:60,]

# make stationary by taking first differences 
new_data = transformData(data, rep(2,ncol(data)))

# tune for the number of factors to use 
tuneFactors(new_data, type = 2)

# fit a PCA using 3 PC's
fit.pca <- sparseDFM(new_data, r = 3, alg = 'PCA')

# fit a DFM using the two-stage approach 
fit.2stage <- sparseDFM(new_data, r = 3, alg = '2Stage')

# fit a DFM using EM algorithm with 3 factors 
fit.dfm <- sparseDFM(new_data, r = 3, alg = 'EM')

# fit a Sparse DFM with 3 factors
fit.sdfm <- sparseDFM(new_data, r = 3, alg = 'EM-sparse')

# observe the factor loadings of the sparse DFM
plot(fit.sdfm, type = 'loading.heatmap')

# observe the factors 
plot(fit.sdfm, type = 'factor')

# observe the residuals 
plot(fit.sdfm, type = 'residual')

# observe the LASSO parameter selected and BIC values 
plot(fit.sdfm, type = 'lasso.bic')

# predict 3 steps ahead 
predict(fit.sdfm, h = 3)

 
}
\references{
Banbura, M., & Modugno, M. (2014). Maximum likelihood estimation of factor models on datasets with arbitrary pattern of missing data. \emph{Journal of Applied Econometrics, 29}(1), 133-160.

Doz, C., Giannone, D., & Reichlin, L. (2011). A two-step estimator for large approximate dynamic factor models based on Kalman filtering. \emph{Journal of Econometrics, 164}(1), 188-205.

Giannone, D., Reichlin, L., & Small, D. (2008). Nowcasting: The real-time informational content of macroeconomic data. \emph{Journal of monetary economics, 55}(4), 665-676.

Koopman, S. J., & Durbin, J. (2000). Fast filtering and smoothing for multivariate state space models. \emph{Journal of Time Series Analysis, 21}(3), 281-296.

Mosley, L., Chan, TS., & Gibberd, A. (2023). sparseDFM: An R Package to Estimate Dynamic Factor Models with Sparse Loadings.

Shumway, R. H., & Stoffer, D. S. (1982). An approach to time series smoothing and forecasting using the EM algorithm. \emph{Journal of time series analysis, 3}(4), 253-264.

Stock, J. H., & Watson, M. W. (2002). Forecasting using principal components from a large number of predictors. \emph{Journal of the American statistical association, 97}(460), 1167-1179.
}
