% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fit.sf.R
\name{fit.sf}
\alias{fit.sf}
\title{Non-parametric stochastic frontier}
\usage{
fit.sf(X, y, X.constrained = NA, H.inv = NA, H.mult = 1,
  method = "u", scale.constraints = TRUE)
}
\arguments{
\item{X}{Matrix of inputs}

\item{y}{Vector of outputs}

\item{X.constrained}{Matrix of inputs where constraints apply}

\item{H.inv}{Inverse of the smoothing matrix (must be positive definite); defaults to rule of thumb}

\item{H.mult}{Scaling factor for rule of thumb smoothing matrix}

\item{method}{Constraints to apply; "u" for unconstrained, "m" for monotonically increasing, and "mc" for monotonically increasing and concave}

\item{scale.constraints}{Boolean, whether to scale constraints by their average value, can help with convergence}
}
\value{
Returns a list with the following elements
\item{y.fit}{Estimated value of the frontier at X.fit}
\item{gradient.fit}{Estimated gradient of the frontier at X.fit}
\item{mean.efficiency}{Average efficiency for X, y as a whole}
\item{mode.efficiency}{Modal efficiencies for each observation in X, y}
\item{X.eval}{Matrix of inputs used for fitting}
\item{X.constrained}{Matrix of inputs where constraints apply}
\item{X.fit}{Matrix of inputs where curve is fit}
\item{H.inv}{Inverse smoothing matrix used in fitting}
\item{method}{Method used to fit frontier}
\item{scaling.factor}{Factor by which constraints are multiplied before quadratic programming}
}
\description{
Fits stochastic frontier of data with kernel smoothing, imposing monotonicity and/or concavity constraints.
}
\details{
This method fits non-parametric stochastic frontier models. The data-generating process
is assumed to be of the form

\deqn{\ln y_i = \ln f(x_i) + v_i - u_i,}

where \eqn{y_i} is the \eqn{i}th observation of output, \eqn{f} is a continuous
function, \eqn{x_i} is the \eqn{i}th observation of input, \eqn{v_i} is a
normally-distributed error term (\eqn{v_i\sim N(0, \sigma_v^2)}), and \eqn{u_i} is a
normally-distributed error term truncated below at zero (\eqn{u_i\sim N^+(0, \sigma_u)}). 
Aigner et al. developed methods to decompose
\eqn{\varepsilon_i = v_i - u_i} into its basic components.

This procedure first fits the mean of the data using \code{fit.mean},
producing estimates of output \eqn{\hat{y}}. Log-proportional errors are calculated as

\deqn{\varepsilon_i = \ln(y_i / \hat{y}_i).}

Following Aigner et al. (1977), parameters of one- and two-sided error distributions
are estimated via maximum likelihood. First,

\deqn{\hat{\sigma}^2 = \frac1N \sum_{i=1}^N \varepsilon_i^2.}

Then, \eqn{\hat{\lambda}} is estimated by solving

\deqn{\frac1{\hat{\sigma}^2} \sum_{i=1}^N \varepsilon_i\hat{y}_i + \frac{\hat{\lambda}}{\hat{\sigma}} \sum_{i=1}^N \frac{f_i^*}{1 - F_i^*}y_i = 0,}

where \eqn{f_i^*} and \eqn{F_i^*} are standard normal density and distribution function,
respectively, evaluated at \eqn{\varepsilon_i\hat{\lambda}\hat{\sigma}^{-1}}. Parameters of
the one- and two-sided distributions are found by solving the identities

\deqn{\sigma^2 = \sigma_u^2 + \sigma_v^2}
\deqn{\lambda = \frac{\sigma_u}{\sigma_v}.}

Mean efficiency over the sample is given by 

\deqn{\exp\left(-\frac{\sqrt{2}}{\sqrt{\pi}}\right)\sigma_u,}

and modal efficiency for each observation is given by

\deqn{-\varepsilon(\sigma_u^2/\sigma^2).}
}
\examples{
data(USMacro)

USMacro <- USMacro[complete.cases(USMacro),]

#Extract data
X <- as.matrix(USMacro[,c("K", "L")])
y <- USMacro$Y

#Fit frontier
fit.sf <- fit.sf(X, y,
                 X.constrained = X,
                 method = "mc")

print(fit.sf$mean.efficiency)
# [1] 0.9772484

#Plot efficiency over time
library(ggplot2)

plot.df <- data.frame(Year = USMacro$Year,
                      Efficiency = fit.sf$mode.efficiency)

ggplot(plot.df, aes(Year, Efficiency)) +
  geom_line()
  
}
\references{
\insertRef{AignerLovellSchmidt}{snfa}\cr\cr
\insertRef{ParmeterRacine}{snfa}
}
