% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sk_var.R
\name{sk_var}
\alias{sk_var}
\title{Generate a covariance matrix or its factorization}
\usage{
sk_var(
  g,
  pars = NULL,
  scaled = FALSE,
  fac_method = "none",
  X = NULL,
  fac = NULL,
  sep = FALSE
)
}
\arguments{
\item{g}{a sk grid object or a list with entries 'gdim', 'gres', 'gval'}

\item{pars}{list of form returned by \code{sk_pars} (with entries 'y', 'x', 'eps', 'psill')}

\item{scaled}{logical, whether to scale by \code{1/pars$psill}}

\item{fac_method}{character, the factorization to return, one of 'none', 'chol', 'eigen'}

\item{X}{numeric matrix, the \code{X} in \code{t(X) \%*\% V \%*\% X} (default is identity, see details)}

\item{fac}{matrix or list of matrices, the variance factorization (only used with X)}

\item{sep}{logical, indicating to return correlation components instead of full covariance matrix}
}
\value{
either matrix \code{V}, or X^T V^{-1} X, or a factorization ('chol' or 'eigen')
}
\description{
Computes the covariance matrix V (or one of its factorizations) for the non-NA points
in sk grid \code{g}, given the model parameters list \code{pars}
}
\details{
By default the output matrix is V. Alternatively, if \code{X} is supplied, the function
returns the quadratic form X^T V^{-1} X.

When \code{fac_method=='eigen'} the function instead returns the eigen-decomposition of the
output matrix, and when \code{fac_method=='chol'} its lower triangular Cholesky factor is
returned. Supplying this factorization in argument \code{fac} in a subsequent call with \code{X}
can speed up calculations. \code{fac} is ignored when \code{X} is not supplied.

\code{scaled=TRUE} returns the matrix scaled by the reciprocal of the partial sill,
\code{1/pars$psill}, before factorization. This is the form expected by functions
\code{sk_var_mult} and \code{sk_LL} in argument \code{fac}.

Numerical precision issues with poorly conditioned covariance matrices can often be
resolved by using 'eigen' factorization method (instead 'chol') and making sure that
\code{pars$eps > 0}.

If all grid points are observed, then the output V becomes separable. Setting \code{sep=TRUE}
in this case causes the function to return the x and y component correlation matrices (or
their factorizations, as requested in \code{fac_method}) separately, in a list. \code{scaled} has no
effect in this output mode. Note also that \code{sep} has no effect when \code{X} is supplied.

If the function is passed an empty grid \code{g} (all points \code{NA}) it returns results for the
complete case (no \code{NA}s). If it is passed a list that is not a sk grid object, it must
include entries 'gdim', 'gres', 'gval' and/or 'idx_grid' (as they are specified in
\code{sk}), all other entries are ignored in this case.
}
\examples{
# define example grid with NAs and example predictors matrix
gdim = c(12, 13)
n = prod(gdim)
n_obs = floor(n/3)
idx_obs = sort(sample.int(n, n_obs))
g = g_empty = sk(gdim)
g[idx_obs] = stats::rnorm(n_obs)
plot(g)

# example kernel
psill = 0.3
pars = utils::modifyList(sk_pars(g), list(psill=psill))

# plot the covariance matrix for observed data, its cholesky factor and eigen-decomposition
V_obs = sk_var(g, pars)
V_obs_chol = sk_var(g, pars, fac_method='chol')
V_obs_eigen = sk_var(g, pars, fac_method='eigen')
sk_plot(V_obs)
sk_plot(V_obs_chol)
sk_plot(V_obs_eigen$vectors)

# empty and complete cases are treated the same

# get the full covariance matrix with sep=FALSE (default)
V_full = sk_var(g_empty, pars)

# check that the correct sub-matrix is there
max(abs( V_obs - V_full[idx_obs, idx_obs] ))

# get 1d correlation matrices with sep=TRUE...
corr_components = sk_var(g_empty, pars, sep=TRUE)
str(corr_components)
sk_plot(corr_components[['x']])

# ... these are related to the full covariance matrix through psill and eps
corr_mat = kronecker(corr_components[['x']], corr_components[['y']])
V_full_compare = pars$psill * corr_mat + diag(pars$eps, n)
max(abs(V_full - V_full_compare))

# ... their factorizations can be returned as (nested) lists
str(sk_var(g_empty, pars, fac_method='chol', sep=TRUE))
str(sk_var(g_empty, pars, fac_method='eigen', sep=TRUE))

# compare to the full covariance matrix factorizations (default sep=FALSE)
str(sk_var(g_empty, pars, fac_method='chol'))
str(sk_var(g_empty, pars, fac_method='eigen'))

# test quadratic form with X
nX = 3
X_all = cbind(1, matrix(stats::rnorm(nX * n), ncol=nX))
cprod_all = crossprod(X_all, chol2inv(chol(V_full))) \%*\% X_all
abs(max(sk_var(g_empty, pars, X=X_all) - cprod_all ))

# test products with inverse of quadratic form with X
mult_test = stats::rnorm(nX + 1)
cprod_all_inv = chol2inv(chol(cprod_all))
cprod_all_inv_chol = sk_var(g_empty, pars, X=X_all, scaled=TRUE, fac_method='eigen')
sk_var_mult(mult_test, pars, fac=cprod_all_inv_chol) - cprod_all_inv \%*\% mult_test

# repeat with missing data
X_obs = X_all[idx_obs,]
cprod_obs = crossprod(X_obs, chol2inv(chol(V_obs))) \%*\% X_obs

abs(max(sk_var(g, pars, X=X_obs) - cprod_obs ))
cprod_obs_inv = chol2inv(chol(cprod_obs))
cprod_obs_inv_chol = sk_var(g, pars, X=X_obs, scaled=TRUE, fac_method='eigen')
sk_var_mult(mult_test, pars, fac=cprod_obs_inv_chol) - cprod_obs_inv \%*\% mult_test

# `scaled` indicates to divide matrix by psill
print( pars[['eps']]/pars[['psill']] )
diag(sk_var(g, pars, scaled=TRUE)) # diagonal elements equal to 1 + eps/psill
( sk_var(g, pars) - psill * sk_var(g, pars, scaled=TRUE) ) |> abs() |> max()
( sk_var(g, pars, X=X_obs, scaled=TRUE) - ( cprod_obs/psill ) ) |> abs() |> max()

# in Cholesky factor this produces a scaling by square root of psill
max(abs( V_obs_chol - sqrt(psill) * sk_var(g, pars, fac_method='chol', scaled=TRUE) ))

# and in the eigendecomposition, a scaling of the eigenvalues
vals_scaled = sk_var(g, pars, fac_method='eigen', scaled=TRUE)$values
max(abs( sk_var(g, pars, fac_method='eigen')$values - psill*vals_scaled ))

}
\seealso{
sk

Other variance-related functions: 
\code{\link{sk_GLS}()},
\code{\link{sk_LL}()},
\code{\link{sk_cmean}()},
\code{\link{sk_nLL}()},
\code{\link{sk_sim}()}
}
\concept{variance-related functions}
