% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/beta_builder.R
\name{beta_builder}
\alias{beta_builder}
\title{Create a Parameter Vector from Lattice Locations}
\usage{
beta_builder(
  row.index,
  col.index,
  im.res,
  B0 = 0,
  B.values,
  index.type = "manual",
  decay.fn = "gaussian",
  phi = 0.5,
  max.d = Inf,
  h,
  w,
  bayesian = FALSE,
  bayesian.dist = NULL,
  bayesian.scale = NULL,
  output.indices = TRUE
)
}
\arguments{
\item{row.index, col.index}{Vectors of row/columns indices for non-zero 
parameters.
If \code{index.type = "manual"}, each vector should contain specific
coordinates.
If \code{index.type = "rectangle"}, each vector should specify rectangle 
length; e.g. row.index = 1:3 means the rectangle's 'length' is from rows
1 to 3.
If \code{index.type = "ellipse"}, the arguments should be scalar values
specifying the row/column coordinates for the center of the ellipse.
If \code{index.type = "decay"}, the arguments should specify the row/column coordinates,
respectively, of the peak parameter value.}

\item{im.res}{A vector specifying the dimension/resolution of the image. 
The first entry is the number of 'rows' in the lattice/image, and the
second entry is the number of 'columns' in the lattice/image.}

\item{B0}{is the "true" intercept value; default is 0.}

\item{B.values}{is a vector "true" parameter values for non-zero parameters.
The order of assignment is by row. If B.values argument is a single value,
then all non-zero parameters are assigned to that value, unless 
\code{decay.fn} has been specified, in which case \code{B.values} is the 
"peak", and non-zero parameters decay smoothly by distance.}

\item{index.type}{is one of index.type = c("manual", "rectangle", "ellipse", "decay")
\itemize{
    \item \code{index.type = "manual"} uses row.index and col.index 
    arguments to specify manually selected non-zero locations. This 
    setting is good for irregular shaped regions.
    \item \code{index.type = "rectangle"} uses row.index and col.index 
    arguments to specify a rectangular region of non-zero parameters.
    \item \code{index.type = "ellipse"} uses \code{w} and \code{h} 
    arguments to specify elliptical region of non-zero parameters.
    \item \code{index.type = "decay"} allows the user to specify a peak
    location with \code{row.index} and \code{col.index}, as with 
    \code{index.type = "ellipse"}. However, the non-zero parameter values
     decay as a function of distance from the peak.
}}

\item{decay.fn}{An argument to specify the decay function of non-zero 
parameters decay from the peak when \code{index.type = "decay"}. Options 
are "exponential" or "gaussian". The rate of decay is given by 
\eqn{B.values * exp(-phi * d)} and \eqn{B.values * exp(-phi * d ^ 2)} for
"exponential" and "gaussian", respectively. The default is 
\code{decay.fn = "gaussian"}. Note that \eqn{d} is the Euclidean distance
between the peak and a specified location, while \eqn{phi} is the rate of 
decay and is set by the user with \code{phi}.}

\item{phi}{A scalar value greater than 0 that determines the decay rate of
non-zero parameters when \code{index.type = "decay"}. The default is 
\code{phi = 0.5}.}

\item{max.d}{When \code{index.type = "decay"}, \code{max.d} determines the
maximum Euclidean distance from the peak that is allowed to be non-zero; 
parameters for locations further than \code{max.d} from the peak are set 
to zero. If this argument is not set by the user then all parameter values 
are determined by the decay function.}

\item{w, h}{If index.type = "ellipse" then the width and height of the 
ellipse, respectively.}

\item{bayesian}{If \code{TRUE}, then parameters are drawn from distributions
based on initial
\code{B.values} vector. Default is \code{FALSE}.}

\item{bayesian.dist}{When \code{bayesian = TRUE}, specifies the distribution
of the parameters. Options are \code{"gaussian"} and \code{uniform}.}

\item{bayesian.scale}{A list. When \code{bayesian = TRUE} and 
\code{bayesian.dist = "gaussian"}, specifies the sd for the distributions 
of parameters. When \code{bayesian = TRUE} and 
\code{bayesian.dist = "uniform"}, specifies the width for the uniform 
distributions for the parameters. The first entry should be one of 
\code{"unique", "binary"}. If  \code{"unique"}, then the second entry in 
the list should be a vector with length equal to \code{B.values + 1} with 
unique values for the sd's/widths, including B0. B0 can be set to a constant
value by setting the first position of \code{bayesian.scale[[2]]} to 0. 
If \code{"binary"}, then the second entry in the list should be a 3-element
vector whose first entry is the sd/width of B0, second entry the sd/width 
of "non-zero" or "important" parameters, and the third entry is the 
sd/width of the "zero" or  "irrelevant"  parameters.}

\item{output.indices}{If \code{output.indices = TRUE}, then the first 
element of the returned list contains the indices for the non-zero parameter
locations (Default). If \code{output.indices = FALSE}, then only the 
parameter vector is returned.}
}
\value{
A 2-element list containing (1) indices for the locations of "true"
non-zero parameters, and (2) a parameter vector.
}
\description{
Specify the locations in the lattice/image that have non-zero parameters
as well as the values for those parameters, and the function creates
the parameter vector that matches the correct locations in the design 
matrix.
}
\note{
The order of the parameters is by row. That is, if the lattice/image
is 4x4, then parameters 1-4 make up the first row, 5-8 then second, and so
forth.
}
\examples{
## example
Bex1 <- beta_builder(row.index = c(5, 5, 6, 6),
                    col.index = c(5, 6, 5, 6),
                    im.res = c(10, 10),
                    B0 = 0, B.values = rep(1, 4))

## True non-zero parameters are locations 45, 46, 55, 56 in B
## i.e. locations (5, 5), (5, 6), (6, 5), (6, 6)

## Suppose that we index rows by i = 1, ... , I
##                       cols by j = 1, ... , J

## The index for a parameter is given by  J * (i - 1) + j
## In this example, I = 10, J = 10; Thus:

## (5, 5) -> 10 * (5 - 1) + 5 = 45
## (5, 6) -> 10 * (5 - 1) + 6 = 46
## (6, 5) -> 10 * (6 - 1) + 5 = 55
## (6, 6) -> 10 * (6 - 1) + 6 = 45
Bex1
## length 101 (includes B0 w/ 100 variable parameter values)
length(Bex1$B)

## example: index.type = "rectangle"
Bex2 <- beta_builder(row.index = 12:15, col.index = 6:19,
                    im.res = c(20, 20), B0 = 16,
                    B.values = 1:(length(12:15) * length(6:19)),
                    index.type = "rectangle")

Bex2
matrix(Bex2$B[-1], nrow = 20, byrow = TRUE)

## example: index.type = "ellipse"
Bex3 <- beta_builder(row.index = 4, col.index = 5,
                    im.res = c(10, 10),
                    B0 = 16, B.values = 3,
                    index.type = "ellipse",
                    h = 5, w = 4)
Bex3
matrix(Bex3$B[-1], nrow = 10, byrow = TRUE)

## decaying parameter values
Bex4 <- beta_builder(row.index = 10, col.index = 20,
                     im.res = c(30, 30), B0 = 0, B.values = 10,
                     index.type = "decay", max.d = 7,
                     output.indices = FALSE)
inf_2D_image(B = Bex4, im.res = c(30, 30), binarize.B = FALSE)

Bex5 <- beta_builder(row.index = 4, col.index = 5,
                     im.res = c(10, 10),
                     B0 = 16, B.values = 5,
                     index.type = "ellipse",
                     h = 5, w = 4,
                     bayesian = TRUE,
                     bayesian.dist = "gaussian",
                     bayesian.scale = list("binary", c(0, 1, 0.25)))

inf_2D_image(B = Bex5$B, im.res = c(10, 10), binarize.B = FALSE)

}
