source('sanitizeCovr.R')

test_that('1 rule makefile', {
  job <- list(rule(target='target.Rdata', depends=c('dep1', 'dep2'), build='buildCmd', clean='cleanCmd'))
  res <- makefile(job, fileName=NULL)

  res <- sanitizeCovr(res)
  res <- res[-seq_len(which(res == ''))]
  expect_equal(res, c('.PHONY: all',
                      'all: Makefile target.Rdata',
                      '\t',
                      'target.Rdata: dep1 dep2',
                      '\tbuildCmd',
                      '.PHONY: clean',
                      'clean: ',
                      '\tcleanCmd',
                      'Makefile: Makefile.R',
                      '\t$(R) -e \'{\' \\',
                      ifelse(getRversion() > '3.4.4',
                             '\t-e \'    params <- list(.target = \"Makefile\", .script = \"Makefile.R\", .depends = NULL, .task = \"all\")\' \\',
                             '\t-e \'    params <- structure(list(.target = \"Makefile\", .script = \"Makefile.R\", .depends = NULL, .task = \"all\"), .Names = c(\".target\", \".script\", \".depends\", \".task\"))\' \\'),
                      '\t-e \'    source("Makefile.R")\' \\',
                      '\t-e \'}\''))
})

test_that('not all in makefile', {
  job <- list(rule(target='target.Rdata', depends=c('dep1', 'dep2'), build='buildCmd', clean='cleanCmd'))
  res <- makefile(job, fileName=NULL, all=FALSE)
  res <- sanitizeCovr(res)
  res <- res[-seq_len(which(res == ''))]
  expect_equal(res, c('target.Rdata: dep1 dep2',
                      '\tbuildCmd',
                      '.PHONY: clean',
                      'clean: ',
                      '\tcleanCmd',
                      'Makefile: Makefile.R',
                      '\t$(R) -e \'{\' \\',
                      ifelse(getRversion() > '3.4.4',
                             '\t-e \'    params <- list(.target = \"Makefile\", .script = \"Makefile.R\", .depends = NULL, .task = \"all\")\' \\',
                             '\t-e \'    params <- structure(list(.target = \"Makefile\", .script = \"Makefile.R\", .depends = NULL, .task = \"all\"), .Names = c(\".target\", \".script\", \".depends\", \".task\"))\' \\'),
                      '\t-e \'    source("Makefile.R")\' \\',
                      '\t-e \'}\''))
})

test_that('not clean in makefile', {
  job <- list(rule(target='target.Rdata', depends=c('dep1', 'dep2'), build='buildCmd', clean='cleanCmd'))
  res <- makefile(job, fileName=NULL, clean=FALSE)
  res <- sanitizeCovr(res)
  res <- res[-seq_len(which(res == ''))]
  expect_equal(res, c('.PHONY: all',
                      'all: Makefile target.Rdata',
                      '\t',
                      'target.Rdata: dep1 dep2',
                      '\tbuildCmd',
                      'Makefile: Makefile.R',
                      '\t$(R) -e \'{\' \\',
                      ifelse(getRversion() > '3.4.4',
                             '\t-e \'    params <- list(.target = \"Makefile\", .script = \"Makefile.R\", .depends = NULL, .task = \"all\")\' \\',
                             '\t-e \'    params <- structure(list(.target = \"Makefile\", .script = \"Makefile.R\", .depends = NULL, .task = \"all\"), .Names = c(\".target\", \".script\", \".depends\", \".task\"))\' \\'),
                      '\t-e \'    source("Makefile.R")\' \\',
                      '\t-e \'}\''))
})

test_that('not Makefile in makefile', {
  job <- list(rule(target='target.Rdata', depends=c('dep1', 'dep2'), build='buildCmd', clean='cleanCmd'))
  res <- makefile(job, fileName=NULL, makefile=FALSE)
  res <- sanitizeCovr(res)
  res <- res[-seq_len(which(res == ''))]
  expect_equal(res, c('.PHONY: all',
                      'all: target.Rdata',
                      '\t',
                      'target.Rdata: dep1 dep2',
                      '\tbuildCmd',
                      '.PHONY: clean',
                      'clean: ',
                      '\tcleanCmd'))
})

test_that('custom vars in makefile', {
  job <- list(rule(target='target.Rdata', depends=c('dep1', 'dep2'), build='buildCmd', clean='cleanCmd'))
  res <- makefile(job, fileName=NULL, vars=c(CPP='gcc'))
  res <- sanitizeCovr(res)
  res <- res[seq_len(which(res == ''))]
  if (.Platform$OS.type == 'unix') {
    expect_equal(res, c('# Generated by rmake: do not edit by hand',
                        'SHELL=/bin/sh',
                        'R="$(R_HOME)/bin$(R_ARCH)/Rscript"',
                        'RM=rm',
                        'CP=cp',
                        'CPP=gcc',
                        ''))
  } else {
    expect_equal(res, c('# Generated by rmake: do not edit by hand',
                        'SHELL=/bin/sh',
                        'R="$(R_HOME)/bin$(R_ARCH)/Rscript"',
                        'RM=cmd //C del',
                        'CP=cmd //C copy',
                        'CPP=gcc',
                        ''))
  }
})


test_that('redefined vars in makefile', {
  job <- list(rule(target='target.Rdata', depends=c('dep1', 'dep2'), build='buildCmd', clean='cleanCmd'))
  res <- makefile(job, fileName=NULL, vars=c(R='MyR', CPP='gcc'))
  res <- sanitizeCovr(res)
  res <- res[seq_len(which(res == ''))]
  if (.Platform$OS.type == 'unix') {
    expect_equal(res, c('# Generated by rmake: do not edit by hand',
                        'SHELL=/bin/sh',
                        'R=MyR',
                        'RM=rm',
                        'CP=cp',
                        'CPP=gcc',
                        ''))
  } else {
    expect_equal(res, c('# Generated by rmake: do not edit by hand',
                        'SHELL=/bin/sh',
                        'R=MyR',
                        'RM=cmd //C del',
                        'CP=cmd //C copy',
                        'CPP=gcc',
                        ''))
  }
})

test_that('multiple target rule makefile', {
  job <- list(rule(target=c('target.pdf', 'target.docx'),
                     depends=c('dep1', 'dep2'),
                     build='buildCmd',
                     clean='cleanCmd'))
  res <- makefile(job, fileName=NULL)
  res <- sanitizeCovr(res)
  res <- res[-seq_len(which(res == ''))]
  expect_equal(res, c('.PHONY: all',
                      'all: Makefile target.pdf target.docx',
                      '\t',
                      'target%pdf target%docx: dep1 dep2',
                      '\tbuildCmd',
                      '.PHONY: clean',
                      'clean: ',
                      '\tcleanCmd',
                      'Makefile: Makefile.R',
                      '\t$(R) -e \'{\' \\',
                      ifelse(getRversion() > '3.4.4',
                             '\t-e \'    params <- list(.target = \"Makefile\", .script = \"Makefile.R\", .depends = NULL, .task = \"all\")\' \\',
                             '\t-e \'    params <- structure(list(.target = \"Makefile\", .script = \"Makefile.R\", .depends = NULL, .task = \"all\"), .Names = c(\".target\", \".script\", \".depends\", \".task\"))\' \\'),
                      '\t-e \'    source("Makefile.R")\' \\',
                      '\t-e \'}\''))
})

test_that('makefile with tasks', {
  job <- list(rule(target=c('target1.pdf', 'target1.docx'),
                     depends=c('dep1.1', 'dep1.2'),
                     build='buildCmd1',
                     clean='cleanCmd1',
                     task='task1'),
              rule(target=c('target2.pdf', 'target2.docx'),
                     depends=c('dep2.1', 'dep2.2'),
                     build='buildCmd2',
                     clean='cleanCmd2',
                     task='task2'),
              rule(target=c('target3.pdf', 'target3.docx'),
                     depends=c('dep3.1', 'dep3.2'),
                     build='buildCmd3',
                     clean='cleanCmd3'))
  res <- makefile(job, fileName=NULL)
  res <- sanitizeCovr(res)
  res <- res[-seq_len(which(res == ''))]
  expect_equal(res, c('.PHONY: all',
                      'all: Makefile task1 task2 target3.pdf target3.docx',
                      '\t',
                      '.PHONY: task1',
                      'task1: Makefile target1.pdf target1.docx',
                      '\t',
                      '.PHONY: task2',
                      'task2: Makefile target2.pdf target2.docx',
                      '\t',
                      'target1%pdf target1%docx: dep1.1 dep1.2',
                      '\tbuildCmd1',
                      'target2%pdf target2%docx: dep2.1 dep2.2',
                      '\tbuildCmd2',
                      'target3%pdf target3%docx: dep3.1 dep3.2',
                      '\tbuildCmd3',
                      '.PHONY: clean',
                      'clean: ',
                      '\tcleanCmd1',
                      '\tcleanCmd2',
                      '\tcleanCmd3',
                      '.PHONY: clean_task1',
                      'clean_task1: ',
                      '\tcleanCmd1',
                      '.PHONY: clean_task2',
                      'clean_task2: ',
                      '\tcleanCmd2',
                      'Makefile: Makefile.R',
                      '\t$(R) -e \'{\' \\',
                      ifelse(getRversion() > '3.4.4',
                             '\t-e \'    params <- list(.target = \"Makefile\", .script = \"Makefile.R\", .depends = NULL, .task = \"all\")\' \\',
                             '\t-e \'    params <- structure(list(.target = \"Makefile\", .script = \"Makefile.R\", .depends = NULL, .task = \"all\"), .Names = c(\".target\", \".script\", \".depends\", \".task\"))\' \\'),
                      '\t-e \'    source("Makefile.R")\' \\',
                      '\t-e \'}\''))
})


test_that('error on unevaluated variables', {
  job <- 'data-$[VAR].csv' %>>% rRule('script.R') %>>% 'result.csv'
  expect_error(makefile(job))
})


test_that('error on non-unique targets', {
  job <- c('data.csv' %>>% rRule('script.R') %>>% 'result.csv',
           'data2.csv' %>>% rRule('script2.R') %>>% 'result.csv')
  expect_error(makefile(job))
})


test_that('duplicate rules are ignored', {
  job <- c('data.csv' %>>% rRule('script.R') %>>% 'result.csv',
           'data.csv' %>>% rRule('script.R') %>>% 'result.csv')
  expect_silent(out <- makefile(job))
})
