% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/AR_reml.R
\name{fitAR}
\alias{fitAR}
\alias{AR_fun}
\title{AR regressions by REML}
\usage{
fitAR(formula, data = NULL)

AR_fun(par, y, X, logLik.only = TRUE)
}
\arguments{
\item{formula}{a model formula, as used by \code{stats::lm()}}

\item{data}{optional data environment to search for variables in \code{formula}.
As used by \code{lm()}}

\item{par}{AR parameter value}

\item{y}{vector of time series (response)}

\item{X}{model matrix (predictors)}

\item{logLik.only}{logical: should only the partial log-likelihood be computed}
}
\value{
\code{fitAR} returns a list object of class "remoteTS", which contains
the following elements.

\describe{
    \item{call}{the function call}
    \item{coefficients}{a named vector of coefficients}
    \item{SE}{the standard errors of parameter estimates}
    \item{tstat}{the t-statistics for coefficients}
    \item{pval}{the p-values corresponding to t-tests of coefficients}
    \item{MSE}{the model mean squared error}
    \item{logLik}{the log-likelihood of the model fit}
    \item{residuals}{the residuals: response minus fitted values}
    \item{fitted.values}{the fitted mean values}
    \item{rho}{The AR parameter, determined via REML}
    \item{rank}{the numeric rank of the fitted model}
    \item{df.residual}{the residual degrees of freedom}
    \item{terms}{the \code{stats::terms} object used}
}

Output is structured similarly to an "lm" object.

When \code{logLik.only == F}, \code{AR_fun} returns the output described in
\code{?fitAR}. When \code{logLik.only == T}, it returns a quantity that is
linearly and negatively related to the restricted log likelihood
(i.e., partial log-likelihood).
}
\description{
\code{fitAR} is used to fit AR(1) time series regression
analysis using restricted maximum likelihood
}
\details{
This function finds the restricted maximum likelihood (REML) to estimate
parameters for the regression model with AR(1) random error terms

\deqn{y(t) =  X(t) \beta + \varepsilon(t)}{y(t) = X(t)*beta + e(t)}
\deqn{\varepsilon(t) =  \rho \varepsilon(t-1) + \delta(t)}{e(t) = rho*e(t-1) + delta(t)}

where \eqn{y(t)} is the response at time \eqn{t};

\eqn{X(t)} is a model matrix containing covariates;

\eqn{\beta}{beta} is a vector of effects of \eqn{X(t)};
\eqn{\varepsilon(t)}{e(t)} is the autocorrelated random error;

\eqn{\delta \sim N(0, \sigma)}{delta ~ N(0, sigma)} is a temporally independent
Gaussian random variable with mean zero and standard deviation
\eqn{\sigma}{sigma};

and \eqn{\rho}{rho} is the AR(1) autoregression parameter

\code{fitAR} estimates the parameter via mathematical optimization
of the restricted log-likelihood function.

\code{AR_fun} is the work horse behind \code{fitAR} that is called
by \code{optim} to estimate the autoregression parameter \eqn{\rho}{rho}.
}
\examples{

# simulate dummy data
t = 1:30 # times series
Z = rnorm(30) # random independent variable
x = .2*Z + (.05*t) # generate dependent effects
x[2:30] = x[2:30] + .2*x[1:29] # add autocorrelation

# fit the AR model, using Z as a covariate
(AR = fitAR(x ~ Z))

# get specific components
AR$residuals
AR$coefficients
AR$pval

# now using time as a covariate
(AR.time <- fitAR(x ~ t))

# source variable from a dataframe
df = data.frame(y = x, t.scaled = t/30, Z = Z)
fitAR(y ~ t.scaled + Z, data = df)

## Methods
summary(AR)
residuals(AR)
coefficients(AR)

}
\references{
Ives, A. R., K. C. Abbott, and N. L. Ziebarth. 2010. Analysis of ecological

    time series with ARMA(p,q) models. Ecology 91:858-871.
}
\seealso{
\code{\link{fitAR_map}} to easily apply \code{fit_AR} to many pixels;
\code{\link{fitCLS}} and \code{\link{fitCLS_map}} for conditional least squares
time series analyses.

Other remoteTS: 
\code{\link{fitAR_map}()},
\code{\link{fitCLS_map}()},
\code{\link{fitCLS}()}

Other remoteTS: 
\code{\link{fitAR_map}()},
\code{\link{fitCLS_map}()},
\code{\link{fitCLS}()}
}
\concept{remoteTS}
