% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ref.R
\name{ref}
\alias{ref}
\title{Create Reference to an Object}
\usage{
ref(x)
}
\arguments{
\item{x}{object to be referenced. \code{x} can be a symbol, character, or an expression containing a symbol.}
}
\value{
a list of class \code{"ref"} containing a reference to the environment of the object and the name of
the object to be found within the environment, or an expression of class \code{"rfexpr"} containing references
}
\description{
Create a reference to an arbitrary R object. Use \code{\link{deref}} or \code{`!`} to obtain the values
within the referenced object. Use \code{\link{sref}} to create a safer reference that limits modification
in place.
}
\details{
Since R does not have reference semantics outside of environments, \code{ref} records the environment location
of an object rather than its memory address.\code{ref(x)} searches for object with name \code{"x"} within the
search path. If found, a reference to the environment and the name \code{"x"} are recorded. Otherwise, an
error is returned.

\code{ref} can also create a reference to objects within an expression. \code{ref} searches the uncalled names
within the expression and replaces them with a reference to the object and a call to deref. For example,
\code{ref(x[[y]][2])} inserts a reference to variable \code{x} and variable \code{y} from the search path into
the expression then wraps the expression into an object of class \code{"ref_exp"}. These objects are
dereferenced by evaluating the expression. An error is returned only if the corresponding variables cannot
be found along the search path.

\code{\link{deref}} can be used to find the objects at the referenced location. This usually results in a
copy of the objects. If the object is no longer available, \code{NULL} will be returned. Generic functions on
a \code{ref} object, such as arithmetic or \code{`sqrt`}, will automatically dereference the object before
applying the generic function. See \link{Methods} and \link{Extract} for a list of available functions
where explicit dereferencing is not needed. If this behavior is not desired, then \code{\link{sref}} can
be used to force the explicit use of \code{deref}.

See \link{Extract} and \code{\link{modify_by}} for functions that modify the underlying value in place.

An active binding could also be used instead of creating a reference. Active bindings, though, can be more
difficult to pass around and may have additional overhead since they are functions.

\code{ref} can provide unsafe or inconsistent code that is susceptible to side-effects. Apply caution and
restraint with its use and be sure to \code{deref} before exporting any \code{ref} objects.
}
\examples{
# Create a vectors of random numbers
x <- rnorm(10)
y <- runif(10)

# Create a reference to the random numbers
ref_to_x <- ref(x)
ref_to_y <- ref(y)

# Place references in a list
list_of_refs <- list(x = ref_to_x, y = ref_to_y)

# Check sum of refs 'x' and 'y'
# Note that both `+` and `sum` automatically deref
sum1 <- sum(list_of_refs$x + list_of_refs$y)

# Update 'x' and calculate new sum
x <- rnorm(10)
sum2 <- sum(list_of_refs$x + list_of_refs$y)

# check diff in sums to see if 'list_of_refs' updated
sum2 - sum1

# Obtain a reference to an expression
ref_to_part <- ref(x[2:5] + 3)
deref(ref_to_part)

# Another expression reference
refs_to_list <- ref(list(x, y))
deref(refs_to_list)

x <- "hello"
y <- "world"

deref(refs_to_list)

# Alternative, `!` can be used for dereferencing
!refs_to_list

identical(!refs_to_list, deref(refs_to_list))

# Referencing data.frame columns
dat <- data.frame(first = 1:4, second = 5:8)
ref_to_first <- ref(dat$first)
mean1 <- mean(!ref_to_first)

dat$first <- dat$first * 4
mean2 <- mean(!ref_to_first)

mean2 == 4*mean1

# Many operations automatically dereference
ref_to_first * 5
ref_to_x == ref_to_y
cos(ref_to_first)
max(ref_to_first)

}
