% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/neiGWAS.R
\name{neiGWAS}
\alias{neiGWAS}
\title{Genome-wide association mapping of neighbor effects}
\usage{
neiGWAS(
  geno,
  pheno,
  gmap,
  smap,
  scale,
  addcovar = NULL,
  grouping = rep(1, nrow(smap)),
  response = c("quantitative", "binary"),
  model = c("lmm", "lm"),
  n_core = 1L
)
}
\arguments{
\item{geno}{An individual x marker matrix. Bialleles (i.e., A or a) must be converted into -1 or 1 digit.}

\item{pheno}{A numeric vector including phenotypes for individuals}

\item{gmap}{A matrix or data.frame including chromosome numbers in the first column, and SNP positions in the second column.}

\item{smap}{A matrix showing a spatial map for individuals. The first and second column include spatial points along an x-axis and y-axis, respectively.}

\item{scale}{A numeric scalar indicating the maximum spatial distance between a focal individual and neighbors to define neighbor effects.}

\item{addcovar}{An optional matrix including additional non-genetic covariates. It contains no. of individuals x no. of covariates.}

\item{grouping}{A positive integer vector assigning each individual to a group. This argument can be useful when a "smap" contains different experimental replicates. Default setting means that all individuals are belong to a single group.}

\item{response}{An option to select if the phenotype is a \code{"quantitative"} trait subject to linear models, or a \code{"binary"} trait subject to logistic models.}

\item{model}{An option to select linear mixed model \code{"lmm"} or linear model \code{"lm"}. Default setting is to use a mixed model.}

\item{n_core}{No. of cores for a multi-core computation. This does not work for Windows OS. Default is a single-core computation.}
}
\value{
A data.frame including the chromosome number, marker position, and p-values.
\itemize{
\item chr Chromosome number
\item pos Marker position
\item p p-value by a likelihood ratio test between models with or without neighbor effects
}
}
\description{
A function to test neighbor effects for each marker and to calculate p-values at a given reference scale.
}
\details{
This function calls a mixed model via the \code{gaston} package. If \code{"lmm"} with \code{"binary"} is selected, p-values are based on Wald tests.
This is because the logistic mixed model is based on a pseudo-likelihood and thus likelihood ratio tests are not applicable. See Chen et al. (2016) for the theory.
}
\examples{
set.seed(1)
g <- matrix(sample(c(-1,1),100*1000,replace = TRUE),100,1000)
gmap <- cbind(c(rep(1,nrow(g)/2),rep(2,nrow(g)/2)),c(1:ncol(g)))
x <- runif(nrow(g),1,100)
y <- runif(nrow(g),1,100)
smap <- cbind(x,y)
grouping <- c(rep(1,nrow(g)/2),rep(2,nrow(g)/2))
pheno <- nei_simu(geno=g,smap=smap,scale=44,grouping=grouping,n_causal=50,pveB=0.4,pve=0.8)

fake_nei <- list()
fake_nei[[1]] <- g
fake_nei[[2]] <- gmap
fake_nei[[3]] <- smap
fake_nei[[4]] <- data.frame(pheno,grouping)
names(fake_nei) <- c("geno","gmap","smap","pheno")

scale <- 43
gwas_out <- neiGWAS(geno=fake_nei$geno, pheno=fake_nei$pheno[,1],
                    gmap=fake_nei$gmap, smap=fake_nei$smap,
                    scale=scale, addcovar=as.matrix(fake_nei$pheno$grouping),
                    grouping=fake_nei$pheno$grouping)

gaston::manhattan(gwas_out)
gaston::qqplot.pvalues(gwas_out$p)
}
\references{
Chen H, Wang C, Conomos M. et al. (2016) Control for population structure and relatedness for binary traits in genetic association studies via logistic mixed models. The American Journal of Human Genetics 98: 653-666.
}
\author{
Yasuhiro Sato (\email{sato.yasuhiro.36c@kyoto-u.jp})
}
