% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lcp.R
\name{find_lcp}
\alias{find_lcp}
\alias{find_lcp,Quadtree-method}
\alias{find_lcp.Quadtree}
\alias{find_lcp,LcpFinder-method}
\alias{find_lcp.LcpFinder}
\title{Find the LCP between two points on a \code{Quadtree}}
\usage{
\S4method{find_lcp}{Quadtree}(
  x,
  start_point,
  end_point,
  use_orig_points = TRUE,
  xlim = NULL,
  ylim = NULL,
  search_by_centroid = FALSE
)

\S4method{find_lcp}{LcpFinder}(x, end_point, allow_same_cell_path = FALSE)
}
\arguments{
\item{x}{a \code{\link{LcpFinder}} or a \code{\link{Quadtree}}}

\item{start_point}{two-element numeric vector; the x and y coordinates of the
starting point. Not used if \code{x} is a \code{\link{LcpFinder}} since the
start point is determined when the \code{\link{LcpFinder}} is created
(using \code{\link{lcp_finder}()}).}

\item{end_point}{two-element numeric vector; the x and y coordinates of the
destination point}

\item{use_orig_points}{boolean; if \code{TRUE} (the default), the path is
calculated between \code{start_point} and \code{end_point}. If
\code{FALSE}, the path is calculated between the centroids of the cells the
points fall in.}

\item{xlim}{two-element numeric vector (xmin, xmax); passed to
\code{\link{lcp_finder}()}; constrains the nodes included in the network to
those whose x limits fall in the range specified in \code{xlim}. If
\code{NULL} the x limits of \code{x} are used}

\item{ylim}{same as \code{xlim}, but for y}

\item{search_by_centroid}{boolean; passed to \code{\link{lcp_finder}()};
determines which cells are considered to be "in" the box specified by
\code{xlim} and \code{ylim}. If \code{FALSE} (the default) any cell that
overlaps with the box is included. If \code{TRUE}, a cell is only included
if its \strong{centroid} falls inside the box.}

\item{allow_same_cell_path}{boolean; default is FALSE; if TRUE, allows
paths to be found between two points that fall in the same cell. See
'Details' for more.}
}
\value{
Returns a five column matrix representing the LCP. It has the
  following columns:
  \itemize{
     \item{\code{x}: }{x coordinate of this point (centroid of the cell)}
     \item{\code{y}: }{y coordinate of this point (centroid of the cell)}
     \item{\code{cost_tot}: }{the cumulative cost up to this point}
     \item{\code{dist_tot}: }{the cumulative distance up to this point - note
     that this is not straight-line distance, but instead the distance along
     the path}
     \item{\code{cost_cell}: }{the cost of the cell that contains this point}
     \item{\code{id}:}{ the ID of the cell that contains this point}
   }

  If no path is possible between the two points, a zero-row matrix with the
  previously described columns is returned.
}
\description{
Finds the least-cost path (LCP) from the start point (the point
  used to create the \code{\link{LcpFinder}}) to another point, using a
  \code{\link{Quadtree}} as a resistance surface.
}
\details{
See the vignette 'quadtree-lcp' for more details and examples (i.e. run
  \code{vignette("quadtree-lcp", package = "quadtree")})
  
  Using \code{find_lcp(<Quadtree>)} rather than \code{find_lcp(<LcpFinder>)}
  is simply a matter of convenience - when a \code{\link{Quadtree}} is passed
  to \code{find_lcp()}, it automatically creates an \code{\link{LcpFinder}}
  and then uses \code{find_lcp(<LcpFinder>)} to get the path between the two
  points. This is convenient if you only want a single LCP. However, if you
  want to find multiple LCPs from a single start point, it is better to first
  create the \code{\link{LcpFinder}} object using \code{\link{lcp_finder}()}
  and then use \code{find_lcp(<LcpFinder>)} for finding LCPs. This is because
  an \code{\link{LcpFinder}} object saves state, so subsequent calls to
  \code{find_lcp(<LcpFinder>)} will run faster.
  
  By default, if the end point falls in the same cell as the start point, the
  path will consist only of the point associated with the cell. When using
  \code{find_lcp} with a \code{\link{LcpFinder}}, setting
  \code{allow_same_cell_path} to \code{TRUE} allows for paths to be found
  within a single cell. In this case, if the start and end points fall in the
  same cell, the path will consist of two points - the point associated with
  the cell and \code{end_point}. If using \code{find_lcp} with a
  \code{\link{Quadtree}}, this will automatically be allowed if
  \code{use_orig_points} is \code{TRUE}.
}
\examples{
####### NOTE #######
# see the "quadtree-lcp" vignette for more details and examples:
# vignette("quadtree-lcp", package = "quadtree")
####################

library(quadtree)
habitat <- terra::rast(system.file("extdata", "habitat.tif", package="quadtree"))

# create a quadtree
qt <- quadtree(habitat, split_threshold = .1, adj_type = "expand")
plot(qt, crop = TRUE, na_col = NULL, border_lwd = .4)

# define our start and end points
start_pt <- c(6989, 34007)
end_pt <- c(33015, 38162)

# create the LCP finder object and find the LCP
lcpf <- lcp_finder(qt, start_pt)
path <- find_lcp(lcpf, end_pt)

# plot the LCP
plot(qt, crop = TRUE, na_col = NULL, border_col = "gray30", border_lwd = .4)
points(rbind(start_pt, end_pt), pch = 16, col = "red")
lines(path[, 1:2], col = "black")

# note that the above path can also be found as follows:
path <- find_lcp(qt, start_pt, end_pt)
}
\seealso{
\code{\link{lcp_finder}()} creates the LCP finder object used as
  input to this function. \code{\link{find_lcps}()} calculates all LCPs
  whose cost-distance is less than some value. \code{\link{summarize_lcps}()}
  outputs a summary matrix of all LCPs that have been calculated so far.
}
