% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/table_to_parquet.R
\name{table_to_parquet}
\alias{table_to_parquet}
\title{Convert an input file to parquet format}
\usage{
table_to_parquet(
  path_to_file,
  path_to_parquet,
  max_memory = NULL,
  max_rows = NULL,
  chunk_size = lifecycle::deprecated(),
  chunk_memory_size = lifecycle::deprecated(),
  columns = "all",
  by_chunk = lifecycle::deprecated(),
  skip = 0,
  partition = "no",
  encoding = NULL,
  chunk_memory_sample_lines = 10000,
  compression = "snappy",
  compression_level = NULL,
  user_na = FALSE,
  ...
)
}
\arguments{
\item{path_to_file}{String that indicates the path to the input file (don't forget the extension).}

\item{path_to_parquet}{String that indicates the path to the directory where the parquet files will be stored.}

\item{max_memory}{Memory size (in Mb) in which data of one parquet file should roughly fit.}

\item{max_rows}{Number of lines that defines the size of the chunk.
This argument can not be filled in if max_memory is used.}

\item{chunk_size}{DEPRECATED use max_rows}

\item{chunk_memory_size}{DEPRECATED use max_memory}

\item{columns}{Character vector of columns to select from the input file (by default, all columns are selected).}

\item{by_chunk}{DEPRECATED use max_memory or max_rows instead}

\item{skip}{By default 0. This argument must be filled in if `by_chunk` is TRUE. Number of lines to ignore when converting.}

\item{partition}{String ("yes" or "no" - by default) that indicates whether you want to create a partitioned parquet file.
If "yes", `"partitioning"` argument must be filled in. In this case, a folder will be created for each modality of the variable filled in `"partitioning"`.
Be careful, this argument can not be "yes" if `max_memory` or `max_rows` argument are not NULL.}

\item{encoding}{String that indicates the character encoding for the input file.}

\item{chunk_memory_sample_lines}{Number of lines to read to evaluate max_memory. Default to 10 000.}

\item{compression}{compression algorithm. Default "snappy".}

\item{compression_level}{compression level. Meaning depends on compression algorithm.}

\item{user_na}{If `TRUE` variables with user defined missing will be read
into [haven::labelled_spss()] objects. If `FALSE`, the default, user-defined missings will be converted to `NA`.}

\item{...}{Additional format-specific arguments,  see \href{https://arrow.apache.org/docs/r/reference/write_parquet.html}{arrow::write_parquet()}
and \href{https://arrow.apache.org/docs/r/reference/write_dataset.html}{arrow::write_dataset()} for more informations.}
}
\value{
Parquet files, invisibly
}
\description{
This function allows to convert an input file to parquet format. \cr

It handles SAS, SPSS and Stata files in a same function. There is only one function to use for these 3 cases.
For these 3 cases, the function guesses the data format using the extension of the input file (in the `path_to_file` argument). \cr

Two conversions possibilities are offered :

\itemize{

\item{Convert to a single parquet file. Argument `path_to_parquet` must then be used;}
\item{Convert to a partitioned parquet file. Additionnal arguments `partition` and `partitioning` must then be used;}

}

To avoid overcharging R's RAM, the conversion can be done by chunk. One of arguments `max_memory` or `max_rows` must then be used.
This is very useful for huge tables and for computers with little RAM because the conversion is then done
with less memory consumption. For more information, see \href{https://ddotta.github.io/parquetize/articles/aa-conversions.html}{here}.
}
\examples{
# Conversion from a SAS file to a single parquet file :

table_to_parquet(
  path_to_file = system.file("examples","iris.sas7bdat", package = "haven"),
  path_to_parquet = tempfile(fileext = ".parquet")
)

# Conversion from a SPSS file to a single parquet file :

table_to_parquet(
  path_to_file = system.file("examples","iris.sav", package = "haven"),
  path_to_parquet = tempfile(fileext = ".parquet"),
)
# Conversion from a Stata file to a single parquet file without progress bar :

table_to_parquet(
  path_to_file = system.file("examples","iris.dta", package = "haven"),
  path_to_parquet = tempfile(fileext = ".parquet")
)

# Reading SPSS file by chunk (using `max_rows` argument)
# and conversion to multiple parquet files :

table_to_parquet(
  path_to_file = system.file("examples","iris.sav", package = "haven"),
  path_to_parquet = tempfile(),
  max_rows = 50,
)

# Reading SPSS file by chunk (using `max_memory` argument)
# and conversion to multiple parquet files of 5 Kb when loaded (5 Mb / 1024)
# (in real files, you should use bigger value that fit in memory like 3000
# or 4000) :

table_to_parquet(
  path_to_file = system.file("examples","iris.sav", package = "haven"),
  path_to_parquet = tempfile(),
  max_memory = 5 / 1024
)

# Reading SAS file by chunk of 50 lines with encoding
# and conversion to multiple files :

table_to_parquet(
  path_to_file = system.file("examples","iris.sas7bdat", package = "haven"),
  path_to_parquet = tempfile(),
  max_rows = 50,
  encoding = "utf-8"
)

# Conversion from a SAS file to a single parquet file and select only
# few columns  :

table_to_parquet(
  path_to_file = system.file("examples","iris.sas7bdat", package = "haven"),
  path_to_parquet = tempfile(fileext = ".parquet"),
  columns = c("Species","Petal_Length")
)

# Conversion from a SAS file to a partitioned parquet file  :

table_to_parquet(
  path_to_file = system.file("examples","iris.sas7bdat", package = "haven"),
  path_to_parquet = tempfile(),
  partition = "yes",
  partitioning =  c("Species") # vector use as partition key
)

# Reading SAS file by chunk of 50 lines
# and conversion to multiple files with zstd, compression level 10

if (isTRUE(arrow::arrow_info()$capabilities[['zstd']])) {
  table_to_parquet(
    path_to_file = system.file("examples","iris.sas7bdat", package = "haven"),
    path_to_parquet = tempfile(),
    max_rows = 50,
    compression = "zstd",
    compression_level = 10
  )
}
}
