% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/nonresponse_weighting_adjustments.R
\name{wt_class_adjust}
\alias{wt_class_adjust}
\title{Adjust weights in a replicate design for nonresponse or unknown eligibility status, using weighting classes}
\usage{
wt_class_adjust(
  survey_design,
  status,
  status_codes,
  wt_class = NULL,
  type = c("UE", "NR")
)
}
\arguments{
\item{survey_design}{A replicate survey design object created with the \code{survey} package.}

\item{status}{A character string giving the name of the variable representing response/eligibility status. \cr
The \code{status} variable should have at most four categories,
representing eligible respondents (ER), eligible nonrespondents (EN),
known ineligible cases (IE), and cases whose eligibility is unknown (UE).}

\item{status_codes}{A named vector, with four entries named 'ER', 'EN', 'IE', and 'UE'. \cr
\code{status_codes} indicates how the values of the \code{status} variable are to be interpreted.}

\item{wt_class}{(Optional) A character string giving the name of the variable which divides sample cases into weighting classes. \cr
If \code{wt_class=NULL} (the default), adjustment is done using the entire sample.}

\item{type}{A character vector including one or more of the following options: \cr
\itemize{
\item \code{'UE'}: Adjust for unknown eligibility. \cr
\item \code{'NR'}: Adjust for nonresponse.
\cr
To sequentially adjust for unknown eligibility and then nonresponse, set \code{type=c('UE', 'NR')}.
}}
}
\value{
A replicate survey design object, with adjusted full-sample and replicate weights
}
\description{
Updates weights in a survey design object to adjust for nonresponse and/or unknown eligibility
using the method of weighting class adjustment. For unknown eligibility adjustments, the weight in each class
is set to zero for cases with unknown eligibility, and the weight of all other cases in the class is
increased so that the total weight is unchanged. For nonresponse adjustments, the weight in each class
is set to zero for cases classified as eligible nonrespondents, and the weight of eligible respondent cases
in the class is increased so that the total weight is unchanged. \cr
\cr
This function currently only works for survey designs with replicate weights,
since the linearization-based estimators included in the \code{survey} package (or Stata or SAS for that matter)
are unable to fully reflect the impact of nonresponse adjustment.
Adjustments are made to both the full-sample weights and all of the sets of replicate weights. \cr
}
\details{
See the vignette "Nonresponse Adjustments" from the svrep package for a step-by-step walkthrough of
nonresponse weighting adjustments in R: \cr
\code{vignette(topic = "nonresponse-adjustments", package = "svrep")}
}
\examples{
library(survey)
# Load an example dataset
data("involvement_survey_str2s", package = "nrba")

# Create a survey design object

involvement_survey_sample <- svydesign(
  data = involvement_survey_str2s,
  weights = ~BASE_WEIGHT,
  strata = ~SCHOOL_DISTRICT,
  ids = ~ SCHOOL_ID + UNIQUE_ID,
  fpc = ~ N_SCHOOLS_IN_DISTRICT + N_STUDENTS_IN_SCHOOL
)

rep_design <- as.svrepdesign(involvement_survey_sample, type = "mrbbootstrap")

# Adjust weights for nonresponse within weighting classes
nr_adjusted_design <- wt_class_adjust(
  survey_design = rep_design,
  status = "RESPONSE_STATUS",
  status_codes = c(
    "ER" = "Respondent",
    "EN" = "Nonrespondent",
    "IE" = "Ineligible",
    "UE" = "Unknown"
  ),
  wt_class = "PARENT_HAS_EMAIL",
  type = "NR"
)
}
\references{
See Chapter 2 of Heeringa, West, and Berglund (2017) or Chapter 13 of Valliant, Dever, and Kreuter (2018)
for an overview of nonresponse adjustment methods based on redistributing weights.
\itemize{
\item Heeringa, S., West, B., Berglund, P. (2017). Applied Survey Data Analysis, 2nd edition. Boca Raton, FL: CRC Press.
"Applied Survey Data Analysis, 2nd edition." Boca Raton, FL: CRC Press.
\item Valliant, R., Dever, J., Kreuter, F. (2018).
"Practical Tools for Designing and Weighting Survey Samples, 2nd edition." New York: Springer.
}
}
\seealso{
\code{\link[svrep:redistribute_weights]{svrep::redistribute_weights()}}, \code{vignette(topic = "nonresponse-adjustments", package = "svrep")}
}
