% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/nonlinearICP.R
\name{nonlinearICP}
\alias{nonlinearICP}
\title{Nonlinear Invariant Causal Prediction}
\usage{
nonlinearICP(X, Y, environment,
  condIndTest = InvariantResidualDistributionTest, argsCondIndTest = NULL,
  alpha = 0.05, varPreSelectionFunc = NULL,
  argsVarPreSelectionFunc = NULL, maxSizeSets = ncol(X),
  condIndTestNames = NULL, speedUp = FALSE, subsampleSize = c(0.1, 0.25,
  0.5, 0.75, 1), retrieveDefiningsSets = TRUE, seed = 1,
  stopIfEmpty = TRUE, testAdditionalSet = NULL, verbose = FALSE)
}
\arguments{
\item{X}{A (nxp)-dimensional matrix (or data frame) with n observations of p variables.}

\item{Y}{A (nx1)-dimensional response vector.}

\item{environment}{Environment variable(s) in an (n x k)-dimensional
matrix or dataframe. Note that not all nonlinear conditional
independence tests may support more than one environmental variable.}

\item{condIndTest}{Function implementing a conditional independence test (see below
for the required interface). Defaults to \code{InvariantResidualDistributionTest} from
the package \code{CondIndTests}.}

\item{argsCondIndTest}{Arguments of \code{condIndTest}. Defaults to \code{NULL}.}

\item{alpha}{Significance level to be used. Defaults to \code{0.05}.}

\item{varPreSelectionFunc}{Variable selection function that is applied
to pre-select a set of variables before running the ICP procedure on the resulting
subset. Should be used with care as causal parents might be excluded in this step.
Defaults to \code{NULL}.}

\item{argsVarPreSelectionFunc}{Arguments of \code{varPreSelectionFunc}.
Defaults to \code{NULL}.}

\item{maxSizeSets}{Maximal size of sets considered as causal parents.
Defaults to \code{ncol(X)}.}

\item{condIndTestNames}{Name of conditional independence test, used for printing.
Defaults to \code{NULL}.}

\item{speedUp}{Use subsamples of sizes specified in \code{subsampleSize} to speed
up the test for sets where the null hypothesis can already be rejected based on
a small number of samples (a larger sample size would potentially further
decrease the p-value but would not change the decision, i.e. the set is rejected
in any case). Applies Bonferroni multiple testing correction.
Defaults to \code{FALSE}.}

\item{subsampleSize}{Size of subsamples used in \code{speedUp} procedure
as fraction of total sample size. Defaults to \code{c(0.1, 0.25, 0.5, 0.75, 1)}.}

\item{retrieveDefiningsSets}{Boolean variable to indicate whether defining sets
should be retrieved. Defaults to \code{TRUE}.}

\item{seed}{Random seed.}

\item{stopIfEmpty}{Stop ICP procedure if retrieved set is empty. If
\code{retrieveDefiningsSets} is \code{TRUE}, setting \code{stopIfEmpty} to
\code{TRUE} results in testing further sets to retrieve the defining sets. However,
 setting \code{stopIfEmpty} to \code{TRUE} in this case will still speedup the procedure as
 some sets will not be tested (namely those where accepting/rejecting
 would not affect the defining sets). Setting \code{stopIfEmpty} to
\code{FALSE} means that all possible subsets of the predictors are tested.}

\item{testAdditionalSet}{If a particular set should be tested, the corresponding
indices can be provided via this argument.}

\item{verbose}{Boolean variable to indicate whether messages should be printed.}
}
\value{
A list with the following elements:
\itemize{
\item \code{retrievedCausalVars} Indices of variables in \eqn{\hat{S}}
\item \code{acceptedSets} List of accepted sets.
\item \code{definingSets}  List of defining sets.
\item \code{acceptedModels} List of accepted models if specified in \code{argsCondIndTest}.
\item \code{pvalues.accepted} P-values of accepted sets.
\item \code{rejectedSets} List of rejected sets.
\item \code{pvalues.rejected} P-values of rejected sets.
\item \code{settings} Settings provided to \code{nonlinearICP}.
}
}
\description{
Nonlinear Invariant Causal Prediction
}
\details{
The function provided as \code{condIndTest} needs to take the following
arguments in the given order: \code{Y, environment, X, alpha, verbose}. Additional
arguments can then be provided via \code{argsCondIndTest}.
}
\examples{
# Example 1
require(CondIndTests)
data("simData")
targetVar <- 2
# choose environments where we did not intervene on var
useEnvs <- which(simData$interventionVar[,targetVar] == 0)
ind <- is.element(simData$environment, useEnvs)
X <- simData$X[ind,-targetVar]
Y <- simData$X[ind,targetVar]
E <- as.factor(simData$environment[ind])
result <- nonlinearICP(X = X, Y = Y, environment = E)
cat(paste("Variable",result$retrievedCausalVars, "was retrieved as the causal
parent of target variable", targetVar))

###################################################

# Example 2
E <- rep(c(1,2), each = 500)
X1 <- E + 0.1*rnorm(1000)
X1 <- rnorm(1000)
X2 <- X1 + E^2 + 0.1*rnorm(1000)
Y <- X1 + X2 + 0.1*rnorm(1000)
resultnonlinICP <- nonlinearICP(cbind(X1,X2), Y, as.factor(E))
summary(resultnonlinICP)
}
\references{
Please cite
C. Heinze-Deml, J. Peters and N. Meinshausen: "Invariant Causal Prediction for Nonlinear Models",
\href{https://arxiv.org/abs/1706.08576}{arXiv:1706.08576}.
}
\seealso{
The function \code{\link[CondIndTests]{CondIndTest}} from the package
 \code{CondIndTests} is a wrapper for a variety of nonlinear conditional independence
 tests that can be used in \code{condIndTest}.
}
