% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dpoisppLocalDetection_normal.R
\name{dpoisppLocalDetection_normal}
\alias{dpoisppLocalDetection_normal}
\alias{rpoisppLocalDetection_normal}
\title{Local evaluation for a Poisson point process detection model}
\usage{
dpoisppLocalDetection_normal(
  x,
  lowerCoords,
  upperCoords,
  s,
  sd,
  baseIntensities,
  habitatGridLocal,
  resizeFactor = 1,
  localObsWindowIndices,
  numLocalObsWindows,
  numMaxPoints,
  numWindows,
  indicator,
  log = 0
)

rpoisppLocalDetection_normal(
  n,
  lowerCoords,
  upperCoords,
  s,
  sd,
  baseIntensities,
  habitatGridLocal,
  resizeFactor = 1,
  localObsWindowIndices,
  numLocalObsWindows,
  numMaxPoints,
  numWindows,
  indicator
)
}
\arguments{
\item{x}{Matrix containing the total number of detections (x[1,1]), the x- and y-coordinates (x[2:(x[1,1]+1),1:2]), 
and the corresponding detection window indices (x[2:(x[1,1]+1),3]) for a set of spatial points (detection locations).}

\item{lowerCoords, upperCoords}{Matrices of lower and upper x- and y-coordinates of all detection windows scaled to the habitat (see (\code{\link{scaleCoordsToHabitatGrid}}). One row for each window. Each window should be of size 1x1.}

\item{s}{Vector of x- and y-coordinates of the isotropic multivariate normal distribution mean (the AC location).}

\item{sd}{Standard deviation of the isotropic multivariate normal distribution.}

\item{baseIntensities}{Vector of baseline detection intensities for all detection windows.}

\item{habitatGridLocal}{Matrix of rescaled habitat grid cells indices, from localIndices returned by the  \code{getLocalObjects} function (}

\item{resizeFactor}{Aggregation factor used in the  \code{getLocalObjects} function to reduce the number of habitat grid cells.}

\item{localObsWindowIndices}{Matrix of indices of local observation windows around each rescaled habitat grid cell, as returned by the getLocalObjects function (object named \code{localIndices}).}

\item{numLocalObsWindows}{Vector of numbers of local observation windows around all habitat grid cells, as returned by the getLocalObjects function (object named \code{numLocalIndices}). 
The ith number gives the number of local (original) observation windows for the ith (rescaled) habitat window.}

\item{numMaxPoints}{Maximum number of points. This value (non-negative integer) is only used when simulating detections to constrain the maximum number of detections.}

\item{numWindows}{Number of detection windows. This value (positive integer) is used to truncate \code{lowerCoords} and \code{upperCoords} 
so that extra rows beyond \code{numWindows} are ignored.}

\item{indicator}{Binary argument specifying whether the individual is available for detection (indicator = 1) or not (indicator = 0).}

\item{log}{Logical argument, specifying whether to return the log-probability of the distribution.}

\item{n}{Integer specifying the number of realisations to generate. Only n = 1 is supported.}
}
\value{
The (log) probability density of the observation matrix \code{x}.
}
\description{
Density and random generation functions of the Poisson point process for detection. 
The \code{dpoisppLocalDetection_normal} distribution is a NIMBLE custom distribution which can be used to model and simulate
Poisson observations (\emph{x}) of a single individual in continuous space over a set of detection windows defined by their upper and lower
coordinates (\emph{lowerCoords,upperCoords}). The distribution assumes that an individual’s detection intensity 
follows an isotropic bivariate normal function centered on the individual's activity center (\emph{s}) with standard deviation (\emph{sd}).
All coordinates (\emph{s} and \emph{trapCoords}) should be scaled to the habitat (\code{\link{scaleCoordsToHabitatGrid}}).
}
\references{
W. Zhang, J. D. Chipperfield, J. B. Illian, P. Dupont, C. Milleret, P. de Valpine and R. Bischof. 2020. 
A hierarchical point process model for spatial capture-recapture data. bioRxiv. DOI 10.1101/2020.10.06.325035 

C. Milleret, P. Dupont, C. Bonenfant, H. Brøseth, Ø. Flagstad, C. Sutherland and R. Bischof. 2019. 
A local evaluation of the individual state-space to scale up Bayesian spatial capture-recapture. Ecology and Evolution 9:352-363

#' @examples 
# Create habitat grid
coordsHabitatGridCenter <- matrix(c(0.5, 3.5,
                                    1.5, 3.5,
                                    2.5, 3.5,
                                    3.5, 3.5,
                                    0.5, 2.5,
                                    1.5, 2.5,
                                    2.5, 2.5,
                                    3.5, 2.5,
                                    0.5, 1.5,
                                    1.5, 1.5,
                                    2.5, 1.5,
                                    3.5, 1.5,
                                    0.5, 0.5,
                                    1.5, 0.5,
                                    2.5, 0.5,
                                    3.5, 0.5), ncol = 2, byrow = TRUE)
colnames(coordsHabitatGridCenter) <- c("x","y")
# Create observation windows
lowerCoords <- matrix(c(1, 1, 2, 1, 1, 2, 2, 2), nrow = 4, byrow = TRUE)
upperCoords <- matrix(c(2, 2, 3, 2, 2, 3, 3, 3), nrow = 4, byrow = TRUE)  
colnames(lowerCoords) <- colnames(upperCoords) <- c("x","y")
# Plot check
plot(coordsHabitatGridCenter[,"y"]~coordsHabitatGridCenter[,"x"],pch=16) 
points(lowerCoords[,"y"]~lowerCoords[,"x"],col="red",pch=16) 
points(upperCoords[,"y"]~upperCoords[,"x"],col="red",pch=16) 

s <- c(1, 1)
sd <- 0.1
baseIntensities <- c(1:4)
windowIndex <- 4
numPoints <- 1
numWindows <- 4
indicator <- 1

# Rescale coordinates
ScaledLowerCoords <- scaleCoordsToHabitatGrid(coordsData =  lowerCoords,
                                              coordsHabitatGridCenter = coordsHabitatGridCenter)$coordsDataScaled
ScaledUpperCoords <- scaleCoordsToHabitatGrid(coordsData =  upperCoords,
                                              coordsHabitatGridCenter = coordsHabitatGridCenter)$coordsDataScaled
ScaledUpperCoords[,2] <- ScaledUpperCoords[,2] + 1.5
ScaledLowerCoords[,2] <- ScaledLowerCoords[,2] - 1.5
habitatMask <- matrix(1, nrow = 4, ncol=4, byrow = TRUE)
# Create local objects 
ObsWindowsLocal <- getLocalObjects(habitatMask = habitatMask,
                                   coords = ScaledLowerCoords,
                                   dmax=3,
                                   resizeFactor = 1,
                                   plot.check = TRUE
)

# Detection locations
x <- matrix(c(1.5, 2, 1.1, 1.5, 1.4, 0.7, 2, 1.3, 1, 1.5), nrow = 5, byrow = TRUE)

# get the window indeces on the third dimension of x
windowIndexes <- 0
for(i in 1:nrow(x)){
  windowIndexes[i] <- getWindowIndex(curCoords = x[i,],
                                     lowerCoords = ScaledLowerCoords,
                                     upperCoords =ScaledUpperCoords)
}
x <- cbind(x, windowIndexes)
# get the total number of detections on x[1,1]
x <- rbind(c(length(windowIndexes),0,0), x)
dpoisppLocalDetection_normal(x, ScaledLowerCoords, ScaledUpperCoords, s, sd, baseIntensities,  
                             ObsWindowsLocal$habitatGrid, ObsWindowsLocal$resizeFactor,
                             ObsWindowsLocal$localIndices,ObsWindowsLocal$numLocalIndices,
                             numMaxPoints = dim(x)[1], numWindows, indicator, log = TRUE)
}
\author{
Wei Zhang, Cyril Milleret and Pierre Dupont
}
