#' ASGN Model Parameter Estimation using MCMC
#'
#' @description
#' Estimates parameters (alpha, mu, sigma2) for the ASGN model using Markov Chain Monte Carlo (MCMC) methods.
#' This function can be used for both cancer and normal groups in the multi-stage DMR detection framework,
#' and also as a standalone function for fitting skewed and potentially bimodal data.
#'
#' @param data A matrix or vector containing the mean methylation levels (M-values)
#' @param priors A list of prior parameters for alpha, mu, and sigma2. If NULL, 
#'   weakly informative priors are automatically generated from the data.
#' @param mcmc A list of MCMC parameters:
#'   \itemize{
#'     \item \code{nburn}: Number of burn-in iterations (default: 5000)
#'     \item \code{niter}: Number of sampling iterations (default: 10000)
#'     \item \code{thin}: Thinning interval (default: 1)
#'   }
#' @param seed Random seed for reproducibility. If NULL, no seed is set.
#' @param return_mcmc Logical indicating whether to return full MCMC samples 
#'   for diagnostic purposes (default: FALSE)
#' @param return_summary Logical indicating whether to return a summary data.frame
#'   with parameter estimates and 95\% credible intervals (default: FALSE)
#'
#' @details
#' This function implements a Metropolis-Hastings within Gibbs sampler to estimate
#' the parameters of the ASGN distribution, which can model skewed and potentially
#' bimodal data. The algorithm updates alpha and mu using Metropolis-Hastings steps 
#' and sigma2 using inverse Gamma sampling.
#'
#' The ASGN (Alpha-skewed generalized normal) distribution is particularly useful for
#' modeling methylation data that often exhibits skewness and bimodal issue. 
#'
#' @return
#' A list that may contain the following elements:
#' \itemize{
#'   \item \code{posteriors}: Vector of posterior means for alpha, mu, and sigma2 (always present)
#'   \item \code{mcmc_samples}: List containing full MCMC chains for alpha, mu, and sigma2 (only if return_mcmc = TRUE)
#'   \item \code{summary}: Data.frame containing parameter estimates with 95\% credible intervals (only if return_summary = TRUE)
#' }
#'
#' @examples
#' \donttest{
#' # Generate sample data
#' set.seed(2021)
#' dt <- rgamma(1000, shape = 2, rate = 1)
#' dt <- as.matrix(dt, ncol=1)
#'
#' result <- asgn_func(dt, return_mcmc = TRUE, return_summary = TRUE)
#' print(result$summary)
#' }
#'
#' @author
#' Zhexuan Yang, Duchwan Ryu, and Feng Luan
#'
#' @seealso
#' \code{\link{mmcmcBayes}} for the main DMR detection function,
#' \code{\link{traceplot_asgn}} for MCMC diagnostic plots
#' 
#' @importFrom stats sd var rnorm runif quantile
#' @importFrom MCMCpack rinvgamma
#'
#' @export
asgn_func <- function(data, 
                      priors = NULL, 
                      mcmc = list(nburn = 5000, niter = 10000, thin = 1), 
                      seed = NULL,
                      return_mcmc = FALSE,
                      return_summary = FALSE) {
  
  # Ensure MCMCpack is installed
  if (!requireNamespace("MCMCpack", quietly = TRUE)) {
    stop("MCMCpack package is required but not installed. Install it using install.packages('MCMCpack')")
  }
  
  # Set seed if provided
  if (!is.null(seed)) {
    set.seed(seed)
  }
  
  ybar <- data  # Input is already the mean methylation level
  
  # Initialize priors based on Stage 1 (NULL) or previous stage priors
  if (is.null(priors)) {
    prior_mean_mu <- mean(ybar, na.rm = TRUE)  
    prior_sd_mu <- sd(ybar, na.rm = TRUE)  
    
    prior_mean_sigma2 <- var(ybar, na.rm = TRUE)  
    prior_sd_sigma2 <- prior_mean_sigma2 / 2  # Weakly informative
    
    priors <- list(
      alpha = 1,  # Default prior for alpha
      mu = prior_mean_mu,
      sigma2 = prior_mean_sigma2
    )
  } else {
    # If priors are provided (starting Stage 2), compute prior SDs from given priors
    prior_mean_mu <- priors$mu
    prior_sd_mu <- sqrt(priors$sigma2)  # Use the provided sigma2 to compute an appropriate SD for mu
    
    prior_mean_sigma2 <- priors$sigma2
    prior_sd_sigma2 <- prior_mean_sigma2 / 2  # Maintain the same regularization rule
  }
  
  # Set MCMC parameters
  nburn <- ifelse(is.null(mcmc$nburn), 5000, mcmc$nburn)
  niter <- ifelse(is.null(mcmc$niter), 10000, mcmc$niter)
  thin  <- ifelse(is.null(mcmc$thin), 1, mcmc$thin)
  
  # Extract priors for this stage
  alpha_pri <- as.numeric(priors$alpha);siga <- 1
  mu_pri <- as.numeric(prior_mean_mu);sigm <- (prior_sd_mu)^2
  sigma2_pri <- as.numeric(prior_mean_sigma2)
  
  # Regularized inverse gamma prior
  As <- (sigma2_pri^2) / (sigma2_pri^2 / 2) + 2  
  Bs <- sigma2_pri * ((sigma2_pri^2) / (sigma2_pri^2 / 2) + 1)
  
  # Initialize MCMC values
  alpha_t <- 1
  mu_t <- mu_pri
  sig_t <- sigma2_pri
  
  ## MCMC Info
  total_iter <- nburn + niter * thin  # Total iterations including thinning
  
  n <- length(ybar)
  
  ## Containers for posterior samples
  alpha_samples <- numeric(niter)
  mu_samples <- numeric(niter)
  sig_samples <- numeric(niter)
  
  ## MCMC Sampling
  save_index <- 1  # Tracks where to save samples after burn-in
  for (k in 1:total_iter) {
    
    ## --- Update alpha using Metropolis-Hastings ---
    alpha_c <- rnorm(1, alpha_t, sd=1)
    
    log_prior_alpha_c <- - (alpha_c - alpha_pri)^2 / (2 * siga)
    log_prior_alpha_t <- - (alpha_t - alpha_pri)^2 / (2 * siga)
    
    ## Include the missing term from the full conditional
    log_gamma_term_c <- -n * log(4 * gamma(3/2) * (alpha_c^2) + 4 * gamma(1/2))
    log_gamma_term_t <- -n * log(4 * gamma(3/2) * (alpha_t^2) + 4 * gamma(1/2))
    
    log_likelihood_c <- sum(log(((1 - alpha_c * ybar)^2 + 1)) - ((ybar - mu_t)^2) / (2 * sig_t))
    log_likelihood_t <- sum(log(((1 - alpha_t * ybar)^2 + 1)) - ((ybar - mu_t)^2) / (2 * sig_t))
    
    ## Compute the Metropolis-Hastings acceptance ratio
    log_alpha_ratio <- (log_prior_alpha_c + log_likelihood_c + log_gamma_term_c) - 
      (log_prior_alpha_t + log_likelihood_t + log_gamma_term_t)
    
    if (log(runif(1,0,1)) < log_alpha_ratio) {
      alpha_t <- alpha_c
    }
    
    ## --- Update mu using Metropolis-Hastings ---
    mu_c <- rnorm(1, mu_t, sd=1)
    
    log_prior_mu_c <- - (mu_c - mu_pri)^2 / (2 * sigm)
    log_prior_mu_t <- - (mu_t - mu_pri)^2 / (2 * sigm)
    
    log_likelihood_mu_c <- sum(log(((1 - alpha_t * ybar)^2 + 1)) - ((ybar - mu_c)^2) / (2 * sig_t))
    log_likelihood_mu_t <- sum(log(((1 - alpha_t * ybar)^2 + 1)) - ((ybar - mu_t)^2) / (2 * sig_t))
    
    log_mu_ratio <- (log_prior_mu_c + log_likelihood_mu_c) - (log_prior_mu_t + log_likelihood_mu_t)
    
    if (log(runif(1,0,1)) < log_mu_ratio) {
      mu_t <- mu_c
    }
    
    ## --- Update sigma2 using Inverse Gamma Sampling with Stability Check ---
    sig_c <- MCMCpack::rinvgamma(1, shape = (sig_t + 2), scale = ((sig_t + 1) * sig_t))
    
    ## Prevent numerical instability (avoid very small values leading to large variances)
    if (sig_c < 1e-300) sig_c <- 1e-300  
    
    log_prior_sig_c <- -As * log(sig_c) - (Bs / sig_c)
    log_prior_sig_t <- -As * log(sig_t) - (Bs / sig_t)
    
    log_likelihood_sig_c <- sum(log(((1 - alpha_t * ybar)^2 + 1)) - ((ybar - mu_t)^2) / (2 * sig_c))
    log_likelihood_sig_t <- sum(log(((1 - alpha_t * ybar)^2 + 1)) - ((ybar - mu_t)^2) / (2 * sig_t))
    
    log_sigma_ratio <- (log_prior_sig_c + log_likelihood_sig_c) - (log_prior_sig_t + log_likelihood_sig_t)
    
    if (log(runif(1,0,1)) < log_sigma_ratio) {
      sig_t <- sig_c
    }
    
    ## Store posterior samples after burn-in with thinning
    if (k > nburn && (k - nburn) %% thin == 0) {
      alpha_samples[save_index] <- alpha_t
      mu_samples[save_index] <- mu_t
      sig_samples[save_index] <- sig_t
      save_index <- save_index + 1
    }
  }
  
  ## Compute posterior means for next stage priors
  alpha_post <- mean(alpha_samples)
  mu_post <- mean(mu_samples)
  sig_post <- mean(sig_samples)
  
  ## Create the return variables
  posterior_means <- c(alpha_post, mu_post, sig_post)
  
  ## Initialize result list - ALWAYS includes posteriors for mmcmcBayes compatibility
  result <- list(posteriors = posterior_means)
  
  ## Add summary data.frame if requested
  if (return_summary) {
    result$summary <- data.frame(
      parameter = c("alpha", "mu", "sigma2"),
      estimate = c(alpha_post, mu_post, sig_post),
      lower_ci = c(
        quantile(alpha_samples, 0.025),
        quantile(mu_samples, 0.025),
        quantile(sig_samples, 0.025)
      ),
      upper_ci = c(
        quantile(alpha_samples, 0.975),
        quantile(mu_samples, 0.975),
        quantile(sig_samples, 0.975)
      ),
      stringsAsFactors = FALSE
    )
    
    # Format for nice printing
    row.names(result$summary) <- NULL
  }
  
  ## Add MCMC samples if requested
  if (return_mcmc) {
    result$mcmc_samples <- list(
      alpha = alpha_samples,
      mu = mu_samples, 
      sigma2 = sig_samples
    )
  }
  
  return(result)
}