% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/PSO.Algorithm.R
\name{PSO}
\alias{PSO}
\title{Optimization using Prticle Swarm Optimization}
\usage{
PSO(FUN, optimType = "MIN", numVar, numPopulation = 40,
  maxIter = 500, rangeVar, Vmax = 2, ci = 1.49445, cg = 1.49445,
  w = 0.729)
}
\arguments{
\item{FUN}{an objective function or cost function,}

\item{optimType}{a string value that represent the type of optimization.
There are two option for this arguments: \code{"MIN"} and \code{"MAX"}.
The default value is \code{"MIN"}, which the function will do minimization.
Otherwise, you can use \code{"MAX"} for maximization problem.
The default value is \code{"MIN"}.}

\item{numVar}{a positive integer to determine the number variables.}

\item{numPopulation}{a positive integer to determine the number populations. The default value is 40.}

\item{maxIter}{a positive integer to determine the maximum number of iterations. The default value is 500.}

\item{rangeVar}{a matrix (\eqn{2 \times n}) containing the range of variables,
where \eqn{n} is the number of variables, and first and second rows
are the lower bound (minimum) and upper bound (maximum) values, respectively.
If all variable have equal upper bound, you can define \code{rangeVar} as
matrix (\eqn{2 \times 1}).}

\item{Vmax}{a positive integer to determine the maximum particle's velocity. The default value is 2.}

\item{ci}{a positive integer to determine individual cognitive. The default value is 1.49445.}

\item{cg}{a positive integer to determine group cognitive. The default value is 1.49445.}

\item{w}{a positive integer to determine inertia weight. The default value is 0.729.}
}
\value{
\code{Vector [v1, v2, ..., vn]} where \code{n} is number variable
        and \code{vn} is value of \code{n-th} variable.
}
\description{
This is the internal function that implements Particle Swarm Optimization
Algorithm. It is used to solve continuous optimization tasks.
Users do not need to call it directly,
but just use \code{\link{metaOpt}}.
}
\details{
This algorithm was proposed by (Kennedy & Eberhart, 1995), inspired by
the behaviour of the social animals/particles, like a flock of birds in
a swarm. The inertia weight that proposed by Shi and Eberhart is used to
increasing the performance of PSO.

In order to find the optimal solution, the algorithm follow the following steps.
\itemize{
\item Initialization: Initialize the first population of particles and its corresponding
      velocity. Then, calculate the fitness of particles and find the best position as
      Global Best and Local Best.
\item Update Velocity: Every particle move around search space with specific velocity.
      In every iteration, the velocity is depend on two things, Global best and Local best.
      Global best is the best position of particle obtained so far, and Local best is the best solution
      in current iteration.
\item Update particle position. After calculating the new velocity, then the particle move around search
      with the new velocity.
\item Update Global best and local best if the new particle become fitter.
\item Check termination criteria, if termination criterion is satisfied, return the
      Global best as the optimal solution for given problem. Otherwise, back to Update Velocity steps.
}
}
\examples{
##################################
## Optimizing the schewefel's problem 1.2 function

# define schewefel's problem 1.2 function as objective function
schewefels1.2 <- function(x){
  dim <- length(x)
  result <- 0
    for(i in 1:dim){
       result <- result + sum(x[1:i])^2
   }
  return(result)
}

## Define parameter
Vmax <- 2
ci <- 1.5
cg <- 1.5
w <- 0.7
numVar <- 5
rangeVar <- matrix(c(-10,10), nrow=2)

## calculate the optimum solution using Particle Swarm Optimization Algorithm
resultPSO <- PSO(schewefels1.2, optimType="MIN", numVar, numPopulation=20,
                 maxIter=100, rangeVar, Vmax, ci, cg, w)

## calculate the optimum value using schewefel's problem 1.2 function
optimum.value <- schewefels1.2(resultPSO)

}
\references{
Kennedy, J. and Eberhart, R. C. Particle swarm optimization.
Proceedings of IEEE International Conference on Neural Networks, Piscataway, NJ. pp. 1942-1948, 1995

Shi, Y. and Eberhart, R. C. A modified particle swarm optimizer.
Proceedings of the IEEE Congress on Evolutionary Computation (CEC 1998), Piscataway, NJ. pp. 69-73, 1998
}
\seealso{
\code{\link{metaOpt}}
}
