% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{ricker_allee_reproduction_model}
\alias{ricker_allee_reproduction_model}
\title{Ricker reproduction model with Allee effects}
\usage{
ricker_allee_reproduction_model(
  abundance,
  reproduction_rate,
  carrying_capacity,
  allee_threshold,
  overcomp_factor = as.numeric(c(1))
)
}
\arguments{
\item{abundance}{\verb{<numeric>} vector (or matrix) of abundances.}

\item{reproduction_rate}{\verb{<numeric>} vector (or matrix) of reproduction rates.}

\item{carrying_capacity}{\verb{<numeric>} vector (or matrix) of carrying capacities.}

\item{allee_threshold}{\verb{<numeric>} vector (or matrix) of Allee thresholds.}

\item{overcomp_factor}{\verb{<numeric>} overcompensation factor (default: 1.0).
Higher values lead to stronger overcompensation. Can also be a vector or matrix.}
}
\value{
\verb{<numeric>} vector (or matrix) of abundances.
}
\description{
An implementation of the Ricker reproduction model with Allee effects based on
(Cabral and Schurr, 2010) with variable overcompensation and an extension to
handle negative reproduction rates.
}
\details{
\subsection{Equations:}{

If \eqn{reproduction\_rate >= 0} (based on: Cabral and Schurr, 2010):
\deqn{N_{t+1} = N_t e^{b r \frac{(K - N_t)(N_t - C)}{(K - C)^2})}}{N_t1 = N_t * e^(b * r * ((K - N_t) * (N_t - C) / (K - C)^2))}

If \eqn{reproduction\_rate < 0}:
\deqn{N_{t+1} = N_t \cdot e^{r}}{N_t1 = N_t * e^(r)}

With:
\itemize{
\item \eqn{N_t} = abundance at time t
\item \eqn{N_{t+1}} = abundance at time t+1
\item \eqn{r} = reproduction rate
\item \eqn{K} = carrying capacity
\item \eqn{C} = (critical) Allee threshold
\item \eqn{b} = overcompensation factor
}

Note that:
\itemize{
\item \code{abundance} should generally be greater than 0.
\item \code{reproduction_rate}, \code{carrying_capacity} and \code{allee_threshold} should either all have the same size as the input abundance or all be of length 1.
\item \code{carrying_capacity} should be greater than 0. If it is 0 or less, the abundance will be set to 0.
\item \code{allee_threshold} should be less than \code{carrying_capacity}. If it is greater than or equal, the abundance will be set to 0.
}

Important Note:
To optimize performance, the functions modifies the abundance in-place.
This mean the input abundance will be modified (See Examples).
Since the result of this function is usually assigned to the same variable as the input abundance, this is unnoticable in most use cases.
Should you wish to keep the input abundance unchanged, you can \code{rlang::duplicate()} it before passing it to this function.
}
}
\examples{
ricker_allee_reproduction_model(
    abundance = 50,
    reproduction_rate = 2,
    carrying_capacity = 100,
    allee_threshold = -100
)
ricker_allee_reproduction_model(
    abundance = 50,
    reproduction_rate = 2,
    carrying_capacity = 100,
    allee_threshold = -100,
    overcomp_factor = 4
)
ricker_allee_reproduction_model(
    abundance = matrix(10, 5, 5),
    reproduction_rate =  0.25,
    carrying_capacity =  100,
    allee_threshold =  20
)
ricker_allee_reproduction_model(
    abundance = matrix(10, 5, 5),
    reproduction_rate =  matrix(seq(-0.5, 0.5, length.out = 25), 5, 5),
    carrying_capacity =  matrix(100, 5, 5),
    allee_threshold =  matrix(20, 5, 5)
)
ricker_allee_reproduction_model(
    abundance = matrix(10, 5, 5),
    reproduction_rate =  matrix(1, 5, 5),
    carrying_capacity =  matrix(100, 5, 5),
    allee_threshold =  matrix(seq(0, 100, length.out = 25), 5, 5)
)
ricker_allee_reproduction_model(
    abundance = matrix(10, 5, 5),
    reproduction_rate =  matrix(seq(0, -2, length.out = 25), 5, 5),
    carrying_capacity =  matrix(100, 5, 5),
    allee_threshold =  matrix(20, 5, 5)
)
# Note that the input abundance is modified in-place
abu <- 10
res <- ricker_allee_reproduction_model(
    abundance = abu,
    reproduction_rate = 0.25,
    carrying_capacity = 100,
    allee_threshold = -100
)
stopifnot(identical(abu, res))
}
\references{
Cabral, J.S. and Schurr, F.M. (2010)
Estimating demographic models for the range dynamics of plant species.
\emph{Global Ecology and Biogeography}, \strong{19}, 85--97.
\doi{10.1111/j.1466-8238.2009.00492.x}
}
