% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/optimize.R
\name{optimize_weights}
\alias{optimize_weights}
\title{Optimize MaxEnt OT constraint weights}
\usage{
optimize_weights(
  input,
  bias_input = NA,
  mu = NA,
  sigma = NA,
  control_params = NA,
  upper_bound = DEFAULT_UPPER_BOUND,
  encoding = "unknown",
  model_name = NA,
  allow_negative_weights = FALSE
)
}
\arguments{
\item{input}{The input data frame/data table/tibble. This should contain one
or more OT tableaux consisting of mappings between underlying and surface
forms with observed frequency and violation profiles. Constraint violations
must be numeric.

For an example of the data frame format, see inst/extdata/sample_data_frame.csv.
You can read this file into a data frame using read.csv or into a tibble
using dplyr::read_csv.

This function also supports the legacy OTSoft file format. You can use this
format by passing in a file path string to the OTSoft file rather than a
data frame.

For examples of OTSoft format, see inst/extdata/sample_data_file.txt.}

\item{bias_input}{(optional)
A data frame/data table/tibble containing the bias mus and sigmas. Each row
corresponds to an individual constraint, and consists of three columns:
\code{Constraint}, which contains the constraint name, \code{Mu}, which contains the
mu, and \code{Sigma}, which contains the sigma. If this argument is provided,
the mu and sigma arguments will be ignored.
Like the \code{input} argument, this function also supports the legacy OTSoft
file format for this argument. In this case, \code{bias_input} should be a path
to the bias parameters in OTSoft format.

For examples of OTSoft bias format, see inst/extdata/sample_bias_file_otsoft.txt.
Each row in this file should be the name of the constraint, followed by the
mu, followed by the sigma (separated by tabs).}

\item{mu}{(optional) A scalar or vector that will serve as the mu for each
constraint in the bias term. Constraint weights will also be initialized to
this value. If a vector, its length must equal the number of constraints in
the input file. This value will not be used if \code{bias_file} is provided.}

\item{sigma}{(optional) A scalar or vector that will serve as the sigma for
each constraint in the bias term. If a vector, its length must equal the
number of constraints in the input file. This value will not be used if
\code{bias_file} is provided.}

\item{control_params}{(optional) A named list of control parameters that
will be passed to the \link[stats]{optim} function. See the documentation
of that function for details. Note that some parameter settings may
interfere with optimization. The parameter \code{fnscale} will be overwritten
with \code{-1} if specified, since this must be treated as a maximization
problem.}

\item{upper_bound}{(optional) The maximum value for constraint weights.
Defaults to 100.}

\item{encoding}{(optional) The character encoding of the input file. Defaults
to "unknown".}

\item{model_name}{(optional) A name for the model. If not provided, the name
of the variable will be used if the input is a data frame. If the input
is a path to an OTSoft file, the filename will be used.}

\item{allow_negative_weights}{(optional) Whether the optimizer should allow
negative weights. Defaults to FALSE.}
}
\value{
An object with the following named attributes:
\itemize{
\item \code{weights}: A named list of the optimal constraint weights
\item \code{log_lik}: the log likelihood of the data under the discovered
weights
\item \code{k}: the number of constraints
\item \code{n}: the number of data points in the training set
}
}
\description{
Optimizes constraint weights given a data set and optional biases. If no
bias arguments are provided, the bias term(s) will not be included in the
optimization.
}
\details{
The objective function \eqn{J(w)} that is optimized is defined as

\deqn{J(w) = \sum_{i=1}^{n}{\ln P(y_i|x_i; w)}
- \sum_{k=1}^{m}{\frac{(w_k - \mu_k)^2}{2\sigma_k^2}}}

The first term in this equation calculates the natural logarithm of the
conditional likelihood of the training data under the weights \eqn{w}. \eqn{n}
is the number of data points (i.e., the sample size or the sum of the frequency
column in the input),\eqn{x_i} is the input form of the \eqn{i}th data
point, and \eqn{y_i} is the observed surface form corresponding to
\eqn{x_i}.\eqn{P(y_i|x_i; w)} represents the probability of realizing
underlying \eqn{x_i} as surface \eqn{y_i} given weights \eqn{w}. This
probability is defined as

\deqn{P(y_i|x_i; w) = \frac{1}{Z_w(x_i)}\exp(-\sum_{k=1}^{m}{w_k f_k(y_i, x_i)})}

where \eqn{f_k(y_i, x_i)} is the number of violations of constraint \eqn{k}
incurred by mapping underlying \eqn{x_i} to surface \eqn{y_i}. \eqn{Z_w(x_i)}
is a normalization term defined as

\deqn{Z(x_i) = \sum_{y\in\mathcal{Y}(x_i)}{\exp(-\sum_{k=1}^{m}{w_k f_k(y, x_i)})}}

where \eqn{\mathcal{Y}(x_i)} is the set of observed surface realizations of
input \eqn{x_i}.

The second term of the equation for calculating the objective function is
the optional bias term, where \eqn{w_k} is the weight of constraint \eqn{k}, and
\eqn{\mu_k} and \eqn{\sigma_k} parameterize a normal distribution that
serves as a prior for the value of \eqn{w_k}. \eqn{\mu_k} specifies the mean
of this distribution (the expected weight of constraint \eqn{k} before
seeing any data) and \eqn{sigma_k} reflects certainty in this value: lower
values of \eqn{\sigma_k} penalize deviations from \eqn{\mu_k} more severely,
and thus require greater amounts of data to move \eqn{w_k} away from
\eqn{mu_k}. While increasing \eqn{\sigma_k} will improve the fit to the
training data, it may result in overfitting, particularly for small data
sets.

A general bias with \eqn{\mu_k = 0} for all \eqn{k} is commonly used as a
form of simple regularization to prevent overfitting (see, e.g., Goldwater
and Johnson 2003). Bias terms have also been used to model proposed
phonological learning biases; see for example Wilson (2006), White (2013),
and Mayer (2021, Ch. 4). The choice of \eqn{\sigma} depends on the sample
size. As the number of data points increases, \eqn{\sigma} must decrease in
order for the effect of the bias to remain constant: specifically,
\eqn{n\sigma^2} must be held constant, where \eqn{n} is the number of tokens.

Optimization is done using the \link[stats]{optim} function from the R-core
statistics library. By default it uses \code{L-BFGS-B} optimization, which is a
quasi-Newtonian method that allows upper and lower bounds on variables.
Constraint weights are restricted to finite, non-negative values.

If no bias parameters are specified (either the \code{bias_file} argument or the
mu and sigma parameters), optimization will be done without the bias term.
}
\examples{
  # Get paths to toy data and bias files.
  df_file <- system.file(
      "extdata", "sample_data_frame.csv", package = "maxent.ot"
  )
  bias_file <- system.file(
       "extdata", "sample_bias_data_frame.csv", package = "maxent.ot"
  )
  # Fit weights to data with no biases
  tableaux_df <- read.csv(df_file)
  optimize_weights(tableaux_df)

  # Fit weights with biases specified in file
  bias_df <- read.csv(bias_file)
  optimize_weights(tableaux_df, bias_df)

  # Fit weights with biases specified in vector form
  optimize_weights(
      tableaux_df, mu = c(1, 2), sigma = c(100, 200)
  )

  # Fit weights with biases specified as scalars
  optimize_weights(tableaux_df, mu = 0, sigma = 1000)

  # Fit weights with mix of scalar and vector biases
  optimize_weights(tableaux_df, mu = c(1, 2), sigma = 1000)

  # Pass additional arguments to optim function
  optimize_weights(tableaux_df, control_params = list(maxit = 500))

}
