#include <RcppArmadillo.h>
#include <fstream>
using namespace Rcpp;
using namespace std;

//' Calculate Lambda Values for Poisson Distribution Mean
//'
//' This function computes the convolution of the epidemic curve with the serial interval distribution
//' to estimate the mean of the Poisson distribution for each day. This mean is crucial
//' in models where the number of new cases follows a Poisson process. The calculation is effectively
//' a weighted sum of past case counts, where weights are given by the serial interval distribution,
//' representing the expected number of new cases caused by an individual case on each subsequent day.
//'
//' @param curve NumericVector representing the daily counts of new cases, estimated from the epidemic data.
//'              This vector should include both historically observed data and nowcasted estimates.
//' @param si NumericVector representing the serial interval distribution, a probability distribution
//'           describing the time delay between successive cases in an infectious disease transmission chain.
//' @return NumericVector containing the estimated mean values for the Poisson distribution
//'         at each day, excluding the last day as the serial interval cannot be fully applied.
//' @details The function applies the serial interval to the epidemic curve via convolution, 
//'          essentially calculating the expected number of secondary cases generated by each primary case
//'          over the serial interval. The length of the resulting vector is one less than that of `curve`
//'          because the last day's value cannot be calculated without full serial interval data.
//' @examples
//' curve <- c(1, 2, 3, 4, 5, 6, 7, 8, 9, 10)
//' si <- c(0.5, 0.3, 0.2)  # Declining probability over three days
//' lambda_values <- lambda(curve, si)
//' @export
// [[Rcpp::export]]
NumericVector lambda(NumericVector curve, NumericVector si) {
  // Description:
  //    create a lambda function to compute mean of poisson dist.
  //    function used in getr and updater
  //    curve is the estimated epidemic curve (i.e., back-calculated counts & nowcasted counts)
  //
  // Arguments
  //
  // Returns:
  //
  int k = si.size();
  int n = curve.size();
  NumericVector c;
  NumericVector sic;
  NumericVector result (n-1);
  for (int i = 0; i < (n-1); ++i){
    if (i < k) {
      c = curve[seq(0, i)];
      sic = rev(si[seq(0, i)]);
      result[i] = sum(c * sic);
    } else {
      c = curve[seq(i - k + 1, i)];
      sic = rev(si);
      result[i] = sum(c * sic);
    }
  }
  return result;
}
