% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tree_detection.R
\name{tree_detection}
\alias{tree_detection}
\title{Tree detection}
\usage{
tree_detection(las, res = 1, ROI = NULL, normalize = FALSE, crown = FALSE, ...)
}
\arguments{
\item{las}{An object of class \code{\link[terra]{SpatRaster-class}},
\code{\link[lidR]{LAS-class}} or \code{\link[lidR]{LAScatalog-class}}}

\item{res}{numeric. The size of a grid cell in point cloud coordinates units,
used to rasterize the point cloud. In case the \code{las} argument is a \code{SpatRaster}
\code{res} is not used.}

\item{ROI}{spatial polygons in sf/sfc format, in the same CRS as argument \code{las}. geometric object that defines the
region where tree detection has to be performed. In case the input is of class
\code{\link[lidR]{LAScatalog-class}}, the chunk buffer set with 
\code{\link[lidR]{engine_options}} is applied to the point cloud to prevent 
border effects, but only treetops lying within the \code{ROI} are returned.}

\item{normalize}{boolean. Should the point cloud be normalized before detection
(not applicable if \code{las} argument is a \code{SpatRaster}) ?}

\item{crown}{Parameter passed to \code{\link{tree_extraction}}}

\item{...}{Parameters passed to \code{\link{tree_segmentation}}}
}
\value{
A sf collection of POINTs with 7 fields: tree id, local maximum stats
 (height, dominance radius), segment stats (surface and volume), coordinates 
 (x and y). In case argument \code{crown} is \code{TRUE}, a \code{crown} field 
 containing the WKT geometry of the 2D crown is also present.
}
\description{
Performs tree detection by applying the functions \code{\link{tree_segmentation}}
and \code{\link{tree_extraction}} to objects of class \code{\link[terra]{SpatRaster-class}},
\code{\link[lidR]{LAS-class}} or \code{\link[lidR]{LAScatalog-class}}
}
\examples{
# load canopy height model
data(chm_chablais3)
chm_chablais3 <- terra::rast(chm_chablais3)
# create polygon of region of interest
ROI <- sf::st_polygon(list(cbind(
 c(974360, 974360, 974380, 974380, 974360),
 c(6581640, 6581680, 6581680, 6581640, 6581640)
)))
# convert to sfc and set projection
ROI = sf::st_sfc(ROI)
sf::st_crs(ROI) <- terra::crs(chm_chablais3)
#
# tree detection
trees <- tree_detection(chm_chablais3)
# plot results
# canopy height model background
terra::plot(chm_chablais3)
# detected trees
plot(trees["h"], add = TRUE, cex = trees$h/20, col = "black")
#
# tree detection in ROI and minimum tree height set to 10
trees_ROI <- tree_detection(chm_chablais3, ROI = ROI, hmin = 10, crown = TRUE)
# create polygons from WKT field
trees_ROI_crowns <- sf::st_as_sf(sf::st_drop_geometry(trees_ROI), wkt = "crown")
# plot results
# canopy height model background 
terra::plot(chm_chablais3)
# detected trees
plot(trees_ROI["h"], add = TRUE, cex = trees_ROI$h/20, col = "black")
# corresponding crowns
plot(sf::st_geometry(trees_ROI_crowns), add = TRUE, border = "black", col = NA)
# add ROI
plot(ROI, add = TRUE, border = "red", col = NA)
}
\references{
Monnet, J.-M. 2011. Using airborne laser scanning for mountain 
forests mapping: Support vector regression for stand parameters estimation 
and unsupervised training for treetop detection. Ph.D. thesis. University of 
Grenoble, France. Section 6.2 
\url{https://theses.hal.science/tel-00652698/document}

Monnet, J.-M., Mermin, E., Chanussot, J., Berger, F. 2010. Tree top detection 
using local maxima filtering: a parameter sensitivity analysis. Silvilaser 2010, 
the 10th International Conference on LiDAR Applications for Assessing Forest 
Ecosystems, September 14-17, Freiburg, Germany, 9 p. 
\url{https://hal.science/hal-00523245/document}
}
\seealso{
\code{\link{tree_segmentation}}, \code{\link{tree_extraction}}
}
