% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/methods.R
\name{wald_univariate}
\alias{wald_univariate}
\title{Univariate Wald Tests and Confidence Intervals for Lagrangian Multiplier Smoothing Splines}
\usage{
wald_univariate(object, scale_vcovmat_by = 1, cv, ...)
}
\arguments{
\item{object}{A fitted lgspline model object containing coefficient estimates and
variance-covariance matrix (requires return_varcovmat = TRUE in fitting).}

\item{scale_vcovmat_by}{Numeric; Scaling factor for variance-covariance matrix.
Adjusts standard errors and test statistics. Default 1.}

\item{cv}{Numeric; Critical value for confidence interval construction. If missing,
defaults to value specified in lgspline() fit (`object$critical_value`) or
`qnorm(0.975)` as a fallback. Common choices:
\itemize{
  \item qnorm(0.975) for normal-based 95% intervals
  \item qt(0.975, df) for t-based 95% intervals, where df = N - trace(XUGX)
}}

\item{...}{Additional arguments passed to the internal `wald_univariate` method.}
}
\value{
A data frame with rows for each coefficient (across all partitions) and columns:
\describe{
  \item{estimate}{Numeric; Coefficient estimate.}
  \item{std_error}{Numeric; Standard error.}
  \item{statistic}{Numeric; Wald or t-statistic (estimate/std_error).}
  \item{p_value}{Numeric; Two-sided p-value based on normal or t-distribution.}
  \item{lower_ci}{Numeric; Lower confidence bound (estimate - cv*std_error).}
  \item{upper_ci}{Numeric; Upper confidence bound (estimate + cv*std_error).}
}
}
\description{
Performs coefficient-specific Wald tests and constructs confidence intervals for fitted
lgspline models. (Wrapper for internal wald_univariate method). For Gaussian family
with identity-link, a t-distribution replaces a normal distribution (and t-intervals, t-tests etc.)
over Wald when mentioned.
}
\details{
For each coefficient, provides:
\itemize{
  \item Point estimates
  \item Standard errors from the model's variance-covariance matrix
  \item Two-sided test statistics and p-values
  \item Confidence intervals using specified critical values
}
}
\examples{

## Simulate some data and fit using default settings
set.seed(1234)
t <- runif(1000, -10, 10)
y <- 2*sin(t) + -0.06*t^2 + rnorm(length(t))
# Ensure varcovmat is returned for Wald tests
model_fit <- lgspline(t, y, return_varcovmat = TRUE)

## Use default critical value (likely qnorm(0.975) if not set in fit)
wald_default <- wald_univariate(model_fit)
print(wald_default)

## Specify t-distribution critical value
eff_df <- NA
if(!is.null(model_fit$N) && !is.null(model_fit$trace_XUGX)) {
   eff_df <- model_fit$N - model_fit$trace_XUGX
}
if (!is.na(eff_df) && eff_df > 0) {
  wald_t <- wald_univariate(
    model_fit,
    cv = stats::qt(0.975, eff_df)
  )
  print(wald_t)
} else {
  warning("Effective degrees of freedom invalid.")
}


}
\seealso{
\code{\link{lgspline}}
}
