% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/kld-estimation-nearest-neighbours.R
\name{kld_est_brnn}
\alias{kld_est_brnn}
\title{Bias-reduced generalized k-nearest-neighbour KL divergence estimation}
\usage{
kld_est_brnn(X, Y, max.k = 100, warn.max.k = TRUE, eps = 0)
}
\arguments{
\item{X, Y}{\code{n}-by-\code{d} and \code{m}-by-\code{d} matrices, representing \code{n} samples from
the true distribution \eqn{P} and \code{m} samples from the approximate distribution
\eqn{Q}, both in \code{d} dimensions. Vector input is treated as a column matrix.
\code{Y} can be left blank if \code{q} is specified (see below).}

\item{max.k}{Maximum numbers of nearest neighbours to compute (default: \code{100}).
A larger \code{max.k} may yield a more accurate KL-D estimate (see \code{warn.max.k}),
but will always increase the computational cost.}

\item{warn.max.k}{If \code{TRUE} (the default), warns if \code{max.k} is such that more
than \code{max.k} neighbours are within the neighbourhood \eqn{\delta} for some
data point(s). In this case, only the first \code{max.k} neighbours are counted.
As a consequence, \code{max.k} may required to be increased.}

\item{eps}{Error bound in the nearest neighbour search. A value of \code{eps = 0}
(the default) implies an exact nearest neighbour search, for \code{eps > 0}
approximate nearest neighbours are sought, which may be somewhat faster for
high-dimensional problems.}
}
\value{
A scalar, the estimated Kullback-Leibler divergence \eqn{\hat D_{KL}(P||Q)}.
}
\description{
This is the bias-reduced generalized k-NN based KL divergence estimator from
Wang et al. (2009) specified in Eq.(29).
}
\details{
Finite sample bias reduction is achieved by an adaptive choice of the number
of nearest neighbours. Fixing the number of nearest neighbours upfront, as
done in \code{\link[=kld_est_nn]{kld_est_nn()}}, may result in very different distances
\eqn{\rho^l_i,\nu^k_i} of a datapoint \eqn{x_i} to its \eqn{l}-th nearest
neighbours in \eqn{X} and \eqn{k}-th nearest neighbours in \eqn{Y},
respectively, which may lead to unequal biases in NN density estimation,
especially in a high-dimensional setting.
To overcome this issue, the number of neighbours \eqn{l,k} are here chosen
in a way to render \eqn{\rho^l_i,\nu^k_i} comparable, by taking the largest
possible number of neighbours \eqn{l_i,k_i} smaller than
\eqn{\delta_i:=\max(\rho^1_i,\nu^1_i)}.

Since the bias reduction explicitly uses both samples \code{X} and \code{Y}, one-sample
estimation is not possible using this method.

Reference:
Wang, Kulkarni and Verdú, "Divergence Estimation for Multidimensional
Densities Via k-Nearest-Neighbor Distances", IEEE Transactions on Information
Theory, Vol. 55, No. 5 (2009). DOI: https://doi.org/10.1109/TIT.2009.2016060
}
\examples{
# KL-D between one or two samples from 1-D Gaussians:
set.seed(0)
X <- rnorm(100)
Y <- rnorm(100, mean = 1, sd = 2)
q <- function(x) dnorm(x, mean = 1, sd =2)
kld_gaussian(mu1 = 0, sigma1 = 1, mu2 = 1, sigma2 = 2^2)
kld_est_nn(X, Y)
kld_est_nn(X, q = q)
kld_est_nn(X, Y, k = 5)
kld_est_nn(X, q = q, k = 5)
kld_est_brnn(X, Y)


# KL-D between two samples from 2-D Gaussians:
set.seed(0)
X1 <- rnorm(100)
X2 <- rnorm(100)
Y1 <- rnorm(100)
Y2 <- Y1 + rnorm(100)
X <- cbind(X1,X2)
Y <- cbind(Y1,Y2)
kld_gaussian(mu1 = rep(0,2), sigma1 = diag(2),
             mu2 = rep(0,2), sigma2 = matrix(c(1,1,1,2),nrow=2))
kld_est_nn(X, Y)
kld_est_nn(X, Y, k = 5)
kld_est_brnn(X, Y)
}
