% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/containers.R
\name{iv-containers}
\alias{iv-containers}
\alias{iv_containers}
\alias{iv_identify_containers}
\alias{iv_identify_container}
\alias{iv_locate_containers}
\title{Containers}
\usage{
iv_containers(x)

iv_identify_containers(x)

iv_identify_container(x)

iv_locate_containers(x)
}
\arguments{
\item{x}{\verb{[iv]}

An interval vector.}
}
\value{
\itemize{
\item For \code{iv_containers()}, an iv with the same type as \code{x}.
\item For \code{iv_identify_containers()}, a list-of containing ivs with the same size
as \code{x}.
\item For \code{iv_identify_container()}, an iv with the same type as \code{x}.
\item For \code{iv_locate_containers()}, a two column data frame with a \code{key} column
containing the result of \code{iv_containers()} and a \code{loc} list-column containing
integer vectors.
}
}
\description{
This family of functions revolves around computing interval \emph{containers}.
A container is defined as the widest interval that isn't contained by any
other interval.
\itemize{
\item \code{iv_containers()} returns all of the containers found within \code{x}.
\item \code{iv_identify_containers()} identifies the containers that each interval in
\code{x} falls in. It replaces \code{x} with a list of the same size where each element
of the list contains the containers that the corresponding interval in \code{x}
falls in. This is particularly useful alongside \code{\link[tidyr:unnest]{tidyr::unnest()}}.
\item \code{iv_identify_container()} is similar in spirit to
\code{iv_identify_containers()}, but is useful when you suspect that each interval
in \code{x} is contained within exactly 1 container. It replaces \code{x} with an iv of
the same size where each interval is the container that the corresponding
interval in \code{x} falls in. If any interval falls in more than one container,
an error is thrown.
\item \code{iv_locate_containers()} returns a two column data frame with a \code{key}
column containing the result of \code{iv_containers()} and a \code{loc} list-column
containing integer vectors that map each interval in \code{x} to the container
that it falls in.
}
}
\examples{
library(dplyr, warn.conflicts = FALSE)
library(tidyr)

x <- iv_pairs(
  c(4, 6),
  c(1, 5),
  c(2, 3),
  c(NA, NA),
  c(NA, NA),
  c(9, 12),
  c(9, 14)
)
x

# Containers are intervals which aren't contained in any other interval.
# They are always returned in ascending order.
# If any missing intervals are present, a single one is retained.
iv_containers(x)

# `iv_identify_container()` is useful alongside `group_by()` and
# `summarize()` if you know that each interval is contained within exactly
# 1 container
df <- tibble(x = x)
df <- mutate(df, container = iv_identify_container(x))
df

df \%>\%
  group_by(container) \%>\%
  summarize(n = n())

# If any interval is contained within multiple containers,
# then you can't use `iv_identify_container()`
y <- c(x, iv_pairs(c(0, 3), c(8, 13)))
y

try(iv_identify_container(y))

# Instead, use `iv_identify_containers()` to identify every container
# that each interval falls in
df <- tibble(y = y, container = iv_identify_containers(y))
df

# You can use `tidyr::unchop()` to see the containers that each interval
# falls in
df \%>\%
  mutate(row = row_number(), .before = 1) \%>\%
  unchop(container)

# A more programmatic interface to `iv_identify_containers()` is
# `iv_locate_containers()`, which returns the containers you get from
# `iv_containers()` alongside the locations in the input that they contain.
iv_locate_containers(y)
}
