% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calc_clint_point.R
\name{calc_clint_point}
\alias{calc_clint_point}
\title{Calculate a Point Estimate of Intrinsic Hepatic Clearance (Clint) (Level-3)}
\usage{
calc_clint_point(
  FILENAME,
  data.in,
  good.col = "Verified",
  output.res = FALSE,
  sig.figs = 3,
  INPUT.DIR = NULL,
  OUTPUT.DIR = NULL,
  verbose = TRUE
)
}
\arguments{
\item{FILENAME}{A string used to identify the input level-2 file,
"<FILENAME>-Clint-Level2.tsv" (if importing from a .tsv file), and/or used 
to identify the output level-3 file, "<FILENAME>-Clint-Level3.tsv" (if exporting).}

\item{data.in}{(Data Frame) A level-2 data frame generated from the 
\code{format_clint} function with a verification column added by 
\code{sample_verification}. Complement with manual verification if needed.}

\item{good.col}{(Character) Column name indicating which rows have been
verified, data rows valid for analysis are indicated with a "Y".
(Defaults to "Verified".)}

\item{output.res}{(Logical) When set to \code{TRUE}, the result 
table (level-3) will be exported to the user's per-session temporary directory
or \code{OUTPUT.DIR} (if specified) as a .tsv file. 
(Defaults to \code{FALSE}.)}

\item{sig.figs}{(Numeric) The number of significant figures to round the exported result table (level-3). 
(Note: console print statements are also rounded to specified significant figures.)
(Defaults to \code{3}.)}

\item{INPUT.DIR}{(Character) Path to the directory where the input level-2 file exists. 
If \code{NULL}, looking for the input level-2 file in the current working
directory. (Defaults to \code{NULL}.)}

\item{OUTPUT.DIR}{(Character) Path to the directory to save the output file. 
If \code{NULL}, the output file will be saved to the user's per-session 
temporary directory or \code{INPUT.DIR} if specified. (Defaults to \code{NULL}.)}

\item{verbose}{(\emph{logical}) Indicate whether printed statements should be shown.
(Default is TRUE.)}
}
\value{
A level-3 data frame with one row per chemical, contains a point estimate of intrinsic 
clearance (Clint), estimates of Clint of assays performed at 1 and 10 uM (if tested), 
the p-value and the Akaike Information Criterion (AIC) of the linear 
regression fit for all chemicals in the input data frame.
}
\description{
This function calculates a point estimate of intrinsic hepatic clearance (Clint) 
using mass spectrometry (MS) peak area data collected as part of \emph{in vitro} measurements 
of chemical clearance, as characterized by the disappearance of parent compound over 
time when incubated with primary hepatocytes \insertCite{shibata2002prediction}{invitroTKstats}.
}
\details{
The input to this function should be "level-2" data. Level-2 data is level-1,
data formatted with the \code{\link{format_clint}} function, and curated
with a verification column. "Y" in the verification column indicates the
data row is valid for analysis. 

The data frame of observations should be annotated according to
these types:
\tabular{rrrrr}{
  Blank \tab Blank\cr
  Hepatocyte incubation concentration vs. time \tab Cvst\cr
}

Clint is calculated using \code{\link{lm}} to perform a linear regression of
MS response as a function of time.

If the output level-3 result table is chosen to be exported and an output 
directory is not specified, it will be exported to the user's R session
temporary directory. This temporary directory is a per-session directory 
whose path can be found with the following code: \code{tempdir()}. For more 
details, see \url{https://www.collinberke.com/til/posts/2023-10-24-temp-directories/}.

As a best practice, \code{INPUT.DIR} (when importing a .tsv file) and/or \code{OUTPUT.DIR} should be 
specified to simplify the process of importing and exporting files. This 
practice ensures that the exported files can easily be found and will not be 
exported to a temporary directory.
}
\examples{
## Load example level-2 data
level2 <- invitroTKstats::clint_L2

## scenario 1: 
## input level-2 data from the R session and do not export the result table
level3 <- calc_clint_point(data.in = level2, output.res = FALSE)

## scenario 2: 
## import level-2 data from a 'tsv' file and export the result table to 
## same location as INPUT.DIR 
\dontrun{
## Refer to sample_verification help file for how to export level-2 data to a directory.
## Unless a different path is specified in OUTPUT.DIR,
## the result table will be saved to the directory specified in INPUT.DIR.
## Will need to replace FILENAME and INPUT.DIR with name prefix and location of level-2 'tsv'.
level3 <- calc_clint_point(# e.g. replace with "Examples" from "Examples-Clint-Level2.tsv"
                           FILENAME="<level-2 FILENAME prefix>",
                           INPUT.DIR = "<level-2 FILE LOCATION>",
                           output.res = TRUE)
}

## scenario 3: 
## input level-2 data from the R session and export the result table to the 
## user's temporary directory
## Will need to replace FILENAME with desired level-2 filename prefix. 
\dontrun{
level3 <- calc_clint_point(# e.g. replace with "MYDATA"
                           FILENAME = "<desired level-2 FILENAME prefix>",
                           data.in = level2,
                           output.res = TRUE)
# To delete, use the following code. For more details, see the link in the 
# "Details" section. 
file.remove(list.files(tempdir(), full.names = TRUE, 
pattern = "<desired level-2 FILENAME prefix>-Clint-Level3.tsv"))
}

}
\references{
\insertRef{shibata2002prediction}{invitroTKstats}
}
\author{
John Wambaugh
}
