% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/independence_weights_estimation.R
\name{independence_weights}
\alias{independence_weights}
\title{Construction of distance covariance optimal weights weights}
\usage{
independence_weights(
  A,
  X,
  lambda = 0,
  decorrelate_moments = FALSE,
  preserve_means = FALSE,
  dimension_adj = TRUE
)
}
\arguments{
\item{A}{vector indicating the value of the treatment or exposure variable. Should be a numeric vector.}

\item{X}{matrix of covariates with number of rows equal to the length of \code{A} and each column is a
\strong{pre-treatment} covariate to be balanced between treatment groups.}

\item{lambda}{tuning parameter for the penalty on the sum of squares of the weights}

\item{decorrelate_moments}{logical scalar. Whether or not to add constraints that result in exact decorrelation of
weighted first order moments of \code{X} and \code{A}. Defaults to \code{FALSE}.}

\item{preserve_means}{logical scalar. Whether or not to add constraints that result in exact preservation of
weighted first order moments of \code{X} and \code{A}. Defaults to \code{FALSE}.}

\item{dimension_adj}{logical scalar. Whether or not to add adjustment to energy distance terms that account for
the dimensionality of \code{X}. Defaults to \code{TRUE}.}
}
\value{
An object of class \code{"independence_weights"} with elements:
\item{weights}{A vector of length \code{nrow(X)} containing the estimated sample weights }
\item{A}{Treatment vector}
\item{opt}{The optimization object returned by \code{osqp::solve_osqp()}}
\item{objective}{The value of the objective function at its optimal value. This is the weighted dependence statistic plus any ridge penalty on the weights.}
\item{D_unweighted}{The value of the weighted dependence distance using all weights = 1 (i.e. unweighted)}
\item{D_w}{The value of the weighted dependence distance of Huling, et al. (2021) using the optimal estimated weights. This is the weighted dependence statistic without the ridge penalty on the weights.}
\item{distcov_unweighted}{The unweighted distance covariance term. This is the standard distance covariance of Szekely et al (2007). This term
is always equal to \code{D_unweighted}.}
\item{distcov_weighted}{The weighted distance covariance term. This term itself does not directly measure weighted dependence but is a critical component of it.  }
\item{energy_A}{The weighted energy distance between \code{A} and its weighted version}
\item{energy_X}{The weighted energy distance between \code{X} and its weighted version}
\item{ess}{The estimated effective sample size of the weights using Kish's effective sample size formula.}

An object of class \code{"independence_weights"}.
\item{weights}{the estimated weights, the distance covariance optimal weights (DCOWs)}
\item{A}{the treatment vector}
\item{opt}{the object returned by whatever optimization routine was used}
\item{objective}{the value of the optimized objective function}
\item{distcov_unweighted}{the unweighted distance covariance between treatment and covariates}
\item{distcov_weighted}{the weighted distance covariance between treatment and covariates}
\item{energy_A}{the (energy) distance between the treatment distribution and the weighted treatment distribution. Smaller values mean the marginal distribution of the treatment is preserved after weighting}
\item{energy_x}{the (energy) distance between the covariate distribution and the weighted covariate distribution. Smaller values mean the marginal distribution of the covariates is preserved after weighting}
\item{ess}{the expected sample size after weighting. Kish's approximation is used}
}
\description{
Constructs independence-inducing weights (distance covariance optimal weights) for
estimation of causal quantities for continuous-valued treatments
}
\examples{

simdat <- simulate_confounded_data(seed = 999, nobs = 500)

y <- simdat$data$Y
A <- simdat$data$A
X <- as.matrix(simdat$data[c("Z1", "Z2", "Z3", "Z4", "Z5")])

dcows <- independence_weights(A, X)

print(dcows)

# distribution of response:
quantile(y)

## create grid
trt_vec <- seq(min(simdat$data$A), 50, length.out=500)

## estimate ADRF
adrf_hat <- weighted_kernel_est(A, y, dcows$weights, trt_vec)$est

## estimate naively without weights
adrf_hat_unwtd <- weighted_kernel_est(A, y, rep(1, length(y)), trt_vec)$est

ylims <- range(c(simdat$data$Y, simdat$true_adrf(trt_vec)))
plot(x = simdat$data$A, y = simdat$data$Y, ylim = ylims, xlim = c(0,50))
## true ADRF
lines(x = trt_vec, y = simdat$true_adrf(trt_vec), col = "blue", lwd=2)
## estimated ADRF
lines(x = trt_vec, y = adrf_hat, col = "red", lwd=2)
## naive estimate
lines(x = trt_vec, y = adrf_hat_unwtd, col = "green", lwd=2)

}
\references{
Szekely, G. J., Rizzo, M. L., & Bakirov, N. K. (2007). Measuring and testing dependence by correlation of distances.
Annals of Statistics 35(6) 2769-2794 \doi{10.1214/009053607000000505}

Huling, J. D., Greifer, N., & Chen, G. (2021). Independence weights for causal inference with continuous exposures.
arXiv preprint arXiv:2107.07086. \url{https://arxiv.org/abs/2107.07086}
}
\seealso{
\code{\link[independenceWeights]{print.independence_weights}} for printing of fitted energy balancing objects
}
