#' Plurality formula
#'
#' @rdname ElecFuns
#'
#' @section Details: The plurality function returns the number of seats according 
#' to the seat allocation formula—plurality. In a single-member district decided
#' by plurality system, voters get a single vote, cast at the party level, to
#' fill the only contested seat, and that seat goes to the top vote-earner
#' regardless of level of support. In a multiple non-transferable vote system,
#' the votes are cast at the candidate level and m is greater than 1. The number
#' of candidates should always be greater or equal to m. 
#' 
#' @return For plurality, a matrix with all candidates participating (1 if a seat was awarded, 0 
#' if not). If m = 1, candidates can be interpreted as parties.
#' @export
#'
#' @examples 
#' plurality (v=example, m=3)

plurality <- function(v, 
                      m,
                      ...){
  n_cand <- ncol(v)
  if(nrow(v) > 1){
    stop("Plurality undefined for ranked votes (perhaps you meant to use a_v()?).")
  }
  if(m > n_cand){
    stop("The number of seats to be awarded is greater than the number of candidates")
  }
  seats <- rep(0, length(v))
  names(seats) <- colnames(v)
  seated <- max_n(v, m)
  seats[seated] <- 1
  return(seats)
}