% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/noteworthy.R
\name{noteworthy}
\alias{noteworthy}
\title{Find noteworthy (unusual) points in a 2D plot}
\usage{
noteworthy(x, y = NULL, n = length(x), method = "mahal", level = NULL, ...)
}
\arguments{
\item{x, y}{The x and y coordinates of a set of points. Alternatively, a single argument \code{x} can be provided,
since \code{\link[grDevices]{xy.coords}(x, y)} is used for construction of the coordinates.}

\item{n}{Maximum number of points to identify. If set to 0, no points are identified.}

\item{method}{Method of point identification. See Details.}

\item{level}{Where appropriate, if supplied, the identified points are filtered so that only those for which the
criterion is \verb{< level}}

\item{...}{Other arguments, silently ignored}
}
\description{
This function extends the logic used by \code{\link[car]{showLabels}} to provide a more general
collection of methods to identify unusual or "noteworthy" points in a two-dimensional display.
Standard methods include Mahalanobis  and Euclidean distance from the centroid, absolute value of distance from
the mean of X or Y, absolute value of Y and absolute value of the residual in a model \code{Y ~ X}.
}
\details{
The \code{method} argument determines how the points to be identified are selected:
\describe{
\item{\code{"mahal"}}{Treat (x, y) as if it were a bivariate sample,
and select cases according to their Mahalanobis distance from \verb{(mean(x), mean(y))}.}
\item{\code{"dsq"}}{Similar to \code{"mahal"} but uses squared Euclidean distance.}
\item{\code{"x"}}{Select points according to their value of \code{abs(x - mean(x))}.}
\item{\code{"y"}}{Select points according to their value of \code{abs(y - mean(y))}.}
\item{\code{"r"}}{Select points according to their value of \code{abs(y)}, as may be appropriate
in residual plots, or others with a meaningful origin at 0, such as a chi-square QQ plot.}
\item{\code{"ry"}}{Fit the linear model, \code{y ~ x} and select points according to their absolute residuals.}
\item{case IDs}{\code{method} can be an integer vector of case numbers in \code{1:length{x}}, in which case those cases
will be labeled.}
\item{numeric vector}{\code{method} can be a vector of the same length as x consisting of values to determine the points
to be labeled. For example, for a linear model \code{mod}, setting \code{method=cooks.distance(mod)} will label the
\code{n} points corresponding to the largest values of Cook's distance. Warning: If missing data are present,
points may be incorrectly selected.}
}

In the case of \code{method == "mahal"} a value for \code{level} can be supplied.
This is used as a filter to select cases whose criterion value
exceeds \code{level}. In this case, the number of points identified will be less than or equal to \code{n}.
}
\examples{
# example code
set.seed(47)
x <- c(runif(100), 1.5, 1.6, 0)
y <- c(2*x[1:100] + rnorm(100, sd = 1.2), -2, 6, 6 )
z <- y - x
mod <- lm(y ~ x)

# testing function to compare noteworthy with car::showLabels() 
testnote <- function(x, y, n, method=NULL, ...)  {
  plot(x, y)
  abline(lm(y ~ x))
  if (!is.null(method))
    car::showLabels(x, y, n=n, method = method) |> print()
  ids <- noteworthy(x, y, n=n, method = method, ...)
  text(x[ids], y[ids], labels = ids, col = "red")
  ids
  }

  # Mahalanobis distance 
testnote(x, y, n = 5, method = "mahal")
testnote(x, y, n = 5, method = "mahal", level = .99)
  # Euclidean distance 
testnote(x, y, n = 5, method = "dsq")

testnote(x, y, n = 5, method = "y")
testnote(x, y, n = 5, method = "ry")

  # a vector of criterion values
testnote(x, y, n = 5, method = Mahalanobis(data.frame(x,y)))
testnote(x, y, n = 5, method = z)
  # vector of case IDs
testnote(x, y, n = 4, method = seq(10, 60, 10))
testnote(x, y, n = 4, method = which(cooks.distance(mod) > .25))

  # test use of xy.coords  
noteworthy(data.frame(x,y), n=4)
noteworthy(y ~ x, n=4)
}
\keyword{utilities}
