% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/functions.R
\name{fut}
\alias{fut}
\title{Design of non-binding futility analysis at multiple points}
\usage{
fut(
  alpha,
  beta,
  t,
  gamma,
  side = 2,
  increment = 1e-04,
  si = 0,
  scale = "CP",
  seed = 12345
)
}
\arguments{
\item{alpha}{Type I error.}

\item{beta}{Type II error (1 - power).}

\item{t}{A numeric vector of information times in \eqn{(0, 1)} for futility looks.}

\item{gamma}{A numeric vector of probabilities (whose meaning depends on
\code{scale}) at information times \code{t}.}

\item{side}{\code{1}- or \code{2}-sided test.}

\item{increment}{Error for the numerical solution of the sample size inflation factor.}

\item{si}{\code{0}: without sample size inflation;
\code{1}: with sample size inflation.}

\item{scale}{Character string specifying the scaled used: \code{"CP"}, conditional power;
 \code{"PP"}, predictive power; \code{"CPd"}: condition power
under current estimate.}

\item{seed}{Seed number for the randomized evaluation of multivariate normal distribution.}
}
\value{
An object of class \code{fut} with the following components.
\code{gamma1}: conditional power at information times \code{t} converted from
the supplied \code{gamma} and \code{scale};
\code{theta}: local alternative associated with the actual power when the
futility rules of enforced;
\code{IF}: sample size inflation factor if \code{si}=1;
\code{loss}: power loss if \code{si}=0.
}
\description{
Design of non-binding futility looks at multiple information times based
on conditional power (CP), predictive power (PP), or condition power
under current estimate (CPd) (Gallo, Mao, and Shih, 2014).
}
\examples{
## load the package
library(grpseq)
## two-sided level 0.05 test with 80\% power;
## evenly spaced three futility looks with predictive power 20\%;
## inflate sample size to recoup power.
obj1 <- fut(alpha=0.05,beta=0.2,t=(1:3)/4,gamma=0.2*rep(1,3),side=2,scale="PP",si=1)
obj1
## print the summary results
summary(obj1)

## do the same thing without sample size inflation
obj2 <- fut(alpha=0.05,beta=0.2,t=(1:3)/4,gamma=0.2*rep(1,3),side=2,scale="PP",si=0)
obj2
## print the summary results
summary(obj2)
oldpar <- par(mfrow = par("mfrow"))
par(mfrow=c(1,2))
## plot the futility boundaries by z-value
plot(obj2,scale='z',lwd=2,main="")
## plot the futility boundaries by B-value
plot(obj2,scale='b',lwd=2,main="")
par(oldpar)
## plot the power curve as a function of the (local)
## effect size in units of the hypothesized effect size
## ref=TRUE requests the power curve for the original one-time analysis
powerplot(obj2,lwd=2, ref=TRUE)
}
\references{
Gallo, P.,  Mao, L., and Shih, V.H. (2014).
Alternative views on setting clinical trial futility criteria.
Journal of Biopharmaceutical Statistics, 24, 976-993.
}
\seealso{
\code{\link{print.fut}}, \code{\link{summary.fut}}, \code{\link{plot.fut}},
\code{\link{powerplot}}
}
\keyword{fut}
