% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/means.R
\name{lehmer_mean}
\alias{lehmer_mean}
\alias{contraharmonic_mean}
\title{Lehmer mean}
\usage{
lehmer_mean(r)

contraharmonic_mean(x, w = NULL, na.rm = FALSE)
}
\arguments{
\item{r}{A finite number giving the order of the Lehmer mean.}

\item{x}{A strictly positive numeric vector.}

\item{w}{A strictly positive numeric vector of weights, the same length as
\code{x}. The default is to equally weight each element of \code{x}.}

\item{na.rm}{Should missing values in \code{x} and \code{w} be removed? By
default missing values in \code{x} or \code{w} return a missing value.}
}
\value{
\code{lehmer_mean()} returns a function:

\preformatted{function(x, w = NULL, na.rm = FALSE){...}}

\code{contraharmonic_mean()} returns a numeric value for the Lehmer mean of
order 2.
}
\description{
Calculate a weighted Lehmer mean.
}
\details{
The function \code{lehmer_mean()} returns a function to compute the Lehmer
mean of order \code{r} of \code{x} with weights \code{w}, which is
calculated as the arithmetic mean of \code{x} with weights \eqn{wx^{r-1}}.
This is also called the counter-harmonic mean or generalized anti-harmonic
mean. See Bullen (2003, p. 245) for details.

The Lehmer mean of order 2 is sometimes called the contraharmonic (or
anti-harmonic) mean. The function \code{contraharmonic_mean()} simply calls
\code{lehmer_mean(2)()}. See von der Lippe
(2015) for more details on the use of these means for making price indexes.
}
\note{
\code{lehmer_mean()} can be defined on the extended real line, so that
\code{r = -Inf / Inf} returns \code{\link[=min]{min()}}/\code{\link[=max]{max()}}, to agree with the
definition in, e.g., Bullen (2003). This is not implemented, and \code{r}
must be finite.
}
\examples{
x <- 2:3
w <- c(0.25, 0.75)

# The Pythagorean means are special cases of the Lehmer mean.

all.equal(lehmer_mean(1)(x, w), arithmetic_mean(x, w))
all.equal(lehmer_mean(0)(x, w), harmonic_mean(x, w))
all.equal(lehmer_mean(0.5)(x), geometric_mean(x))

# When r < 1, the generalized mean is larger than the corresponding
# Lehmer mean.

lehmer_mean(-1)(x, w) < generalized_mean(-1)(x, w)

# The reverse is true when r > 1.

lehmer_mean(3)(x, w) > generalized_mean(3)(x, w)

# This implies the contraharmonic mean is larger than the quadratic
# mean, and therefore the Pythagorean means.

contraharmonic_mean(x, w) > arithmetic_mean(x, w)
contraharmonic_mean(x, w) > geometric_mean(x, w)
contraharmonic_mean(x, w) > harmonic_mean(x, w)

# ... and the logarithmic mean

contraharmonic_mean(2:3) > logmean(2, 3)

# The difference between the arithmetic mean and contraharmonic mean
# is proportional to the variance of x.

weighted_var <- function(x, w) {
  arithmetic_mean((x - arithmetic_mean(x, w))^2, w)
}

arithmetic_mean(x, w) + weighted_var(x, w) / arithmetic_mean(x, w)
contraharmonic_mean(x, w)

}
\references{
Bullen, P. S. (2003). \emph{Handbook of Means and Their Inequalities}.
Springer Science+Business Media.

von der Lippe, P. (2015). Generalized Statistical Means and New Price Index
Formulas, Notes on some unexplored index formulas, their interpretations and
generalizations. Munich Personal RePEc Archive paper no. 64952.
}
\seealso{
Other means: 
\code{\link{extended_mean}()},
\code{\link{generalized_mean}()},
\code{\link{nested_mean}()}
}
\concept{means}
