#' Visualization of RSCU (Relative Synonymous Codon Usage) with ggplot2
#'
#' @description
#' This function generates a customized visualization of Relative Synonymous
#' Codon Usage (RSCU) across distinct biological categories. It represents
#' codon usage through color-coded blocks using species-specific shapes,
#' and optionally incorporates a secondary plot displaying codon labels.
#' Also applicable for extended visualizations of similar data structures
#' (e.g., energy consumption data).
#'
#' @details
#' Key features:
#' - Hierarchical grouping with automatic position calculation
#' - Flexible color management with 8 predefined themes or custom palettes
#' - Dual-axis annotation system for biological information
#' - Interactive subplot arrangements with smart label positioning
#'
#' @param data A data frame containing the input data.
#' @param x Primary and secondary x-axis variables: a character vector
#' (e.g., c("Group", "Subgroup")) or a single variable ("Group").
#' @param y Primary and secondary y-axis variables: a character vector
#' (e.g., c("ValueVar", "Subgroup")) or a single variable ("ValueVar").
#' @param fill Variable for color filling.
#' @param rev (Optional)  logical value. If TRUE, the color order will
#' be reversed. The default is FALSE.
#' @param blocks_colors (Optional) A custom color vector to fill the plot.
#' If "NULL", the default color scheme will be used.
#' @param theme (Optional) A string specifying the color theme for the
#' blocks. Available themes are: "theme1"-"theme8".
#' @param shape (Optional) logical value. If TRUE, a vector of shape
#' values for species in the plot is applied  (default: FALSE).
#' @param size (Optional) Size of shape markers (default: 1).
#' @param merge (Optional) logical value. If TRUE, the main plot will be
#' merged with the secondary annotation plot. Default is FALSE.
#' @param border_color (Optional) Color for bar borders (default: NA)
#' @param width (Optional) Thickness of bar plot borders.
#' (e.g., 0.1 for thin, 0.5 for thick). (Default: 0.1).
#' @param sub_axis (Optional) Annotation type for secondary plot
#' ("ysub" or "fill", default: "fill").
#'
#' @return A ggplot object or patchwork composite plot when merge=TRUE
#'
#' @examples
#' # Example 1: Basic Usage
#'
#' library(ggmRSCU)
#'
#' ggmRSCU(
#'   data = rscu_data,
#'   x = c(AA, Species),
#'   y = c(RSCU, Codon),
#'   fill = Fill,
#'   shape = TRUE,
#'   merge = TRUE
#' )
#'
#'
#' # Example 2: Visualize RSCU from data loaded using `read_codonw` function
#' \donttest{
#' # Load data from `read_codonw` output
#' example_dir <- system.file("extdata", "codonw", package = "ggmRSCU")
#' data <- read_codonw(example_dir)
#'
#' col <- c("#FF6F61", "#6B5B95", "#88B04B",
#'          "#F7CAC9", "#92A8D1", "#9b59b6")
#'
#' ggmRSCU(
#'   data = data,
#'   x = c(AA, Species),
#'   y = c(RSCU, Codon),
#'   fill = Fill,
#'   blocks_colors = col,
#'   shape = TRUE,
#'   merge = TRUE,
#'   sub_axis = "fill"
#' )
#' }
#'
#'
#' # Example 3: Visualize RSCU from data loaded using `read_mega` function
#' \donttest{
#' # Load data from `read_mega` output
#' example_dir <- system.file("extdata", "mega", package = "ggmRSCU")
#' df <- read_mega(example_dir)
#'
#' ggmRSCU(
#'   data = df,
#'   x = c(AA, Species),
#'   y = c(RSCU, Codon),
#'   fill = Fill
#' )
#' }
#'
#'
#' # Example 4: Additional Dataset (scoring_data)
#'
#' col <- c("#4E79A7", "#A0CBE8", "#F28E2B",
#'          "#FFBE7D", "#59A14F", "#9b59b6")
#'
#' ggmRSCU(
#'   data = scoring_data,
#'   x = c(Sample, Replicate),
#'   y = c(Score, CellType),
#'   fill = CellType,
#'   blocks_colors = col,
#'   shape = TRUE,
#'   sub_axis = "Fill"
#' )
#'
#'
#' # Example 5: Additional Dataset (hydrogen_data)
#'
#' library(tidyr)
#'
#' df_long <- hydrogen_data %>%
#'   pivot_longer(cols = -c(Year, type),
#'   names_to = "category",
#'   values_to = "value")
#'
#' col <- c("#4E79A7", "#A0CBE8", "#F28E2B",
#'          "#FFBE7D", "#59A14F", "#9b59b6")
#'
#' ggmRSCU(
#'   data = df_long,
#'   x = c(Year, category),
#'   y = c(value, type),
#'   fill = type,
#'   shape = TRUE,
#'   blocks_colors = col
#' )
#'
#'
#' # Example 6: Additional Dataset (energy_data)
#'
#' long_data <- energy_data %>%
#'   pivot_longer(cols = 3:7,
#'   names_to = "Source",
#'   values_to = "Production")
#'
#' col <- c("#4E79A7", "#A0CBE8", "#F28E2B",
#'          "#FFBE7D", "#59A14F", "#9b59b6")
#'
#' ggmRSCU(
#'   data = long_data,
#'   x = c(Scenario, Year),
#'   y = c(Production, Source),
#'   fill = Source,
#'   blocks_colors = col,
#'   shape = TRUE
#' )
#'
#'
#' # Example 7: Additional Dataset (financial_data)
#'
#' ggmRSCU(
#'   data = financial_data,
#'   x = c(Month, Category),
#'   y = c(Amount),
#'   fill = Subcategory
#' )
#'
#'
#' # Example 8: Additional Dataset (genomic_data)
#'
#' ggmRSCU(
#'   data = genomic_data,
#'   x = c(Species, haplotypes),
#'   y = c(Mb),
#'   fill = Region,
#'   shape = TRUE
#' )
#'
#'
#' @export
ggmRSCU <- function(data,
                    x,
                    y,
                    fill,
                    rev = FALSE,
                    theme = "theme1",
                    shape = FALSE,
                    size = 1,
                    width = 0.1,
                    sub_axis = "fill",
                    merge = FALSE,
                    border_color = NA,
                    blocks_colors = NULL) {
  parse_vars <- function(expr, name, allow_secondary = TRUE) {
    tryCatch(
      {
        if (is.character(expr)) {
          return(expr)
        }

        if (is.call(expr)) {
          if (expr[[1]] == quote(c) || expr[[1]] == quote(list)) {
            elements <- lapply(expr[-1], function(e) {
              if (is.character(e)) {
                return(e)
              }
              all.vars(e)
            })
            parsed <- unlist(elements)
          } else if (expr[[1]] == quote(`~`)) {
            parsed <- all.vars(expr)
          } else {
            parsed <- all.vars(expr)
          }
        } else if (is.name(expr)) {
          parsed <- as.character(expr)
        } else if (is.atomic(expr)) {
          parsed <- as.character(expr)
        } else {
          stop("Unsupported input type for ", name)
        }

        parsed <- unique(parsed)
        if (allow_secondary) {
          if (length(parsed) > 2) {
            stop(name, " accepts maximum 2 variables, got ", length(parsed))
          }
          return(parsed)
        } else {
          if (length(parsed) != 1) {
            stop(name, " requires exactly 1 variable, got ", length(parsed))
          }
          return(parsed)
        }
      },
      error = function(e) {
        stop("Error parsing ", name, ": ", conditionMessage(e))
      }
    )
  }


  x <- parse_vars(substitute(x), "x", allow_secondary = TRUE)
  y <- parse_vars(substitute(y), "y", allow_secondary = TRUE)
  fill <- parse_vars(substitute(fill), "fill", allow_secondary = FALSE)


  xpri <- x[1]
  xsub <- if (length(x) >= 2) x[2] else NULL
  ypri <- y[1]
  ysub <- if (length(y) >= 2) y[2] else NULL
  fill_var <- fill[1]



  validate_params <- function() {
    all_vars <- unique(c(xpri, xsub, ypri, ysub, fill_var))
    missing_vars <- setdiff(all_vars, names(data))
    if (length(missing_vars) > 0) {
      stop("Missing variables in data: ", paste(missing_vars, collapse = ", "))
    }

    color_schemes <- c(paste0("theme", 1:8))
    if (!is.null(theme) && !theme %in% color_schemes) {
      stop("Invalid theme. Available: ", paste(color_schemes, collapse = ", "))
    }
  }
  validate_params()

  suppressMessages({
    requireNamespace("dplyr", quietly = TRUE)
    requireNamespace("ggplot2", quietly = TRUE)
    if (merge) requireNamespace("patchwork", quietly = TRUE)
  })

  data <- data %>%
    dplyr::mutate(
      !!xpri := factor(.data[[xpri]], levels = unique(.data[[xpri]])),
      !!ypri := as.numeric(.data[[ypri]])
    )

  if (!is.null(xsub)) {
    data <- data %>%
      dplyr::mutate(!!xsub := factor(.data[[xsub]],
                      levels = unique(.data[[xsub]])
                    ))
  }

  if (!is.null(ysub)) {
    data <- data %>%
      dplyr::mutate(!!ysub := factor(.data[[ysub]],
                      levels = unique(.data[[ysub]])
                    ))
  }

  data <- data %>%
    dplyr::mutate(
      !!fill_var := {
        fill_data <- dplyr::pull(data, !!fill_var)
        fill_numeric <- suppressWarnings(as.numeric(as.character(fill_data)))
        if (!any(is.na(fill_numeric))) {
          factor(fill_numeric, levels = sort(unique(fill_numeric)))
        } else {
          factor(fill_data, levels = unique(fill_data))
        }
      }
    )

  if (!is.null(xsub) && xsub != xpri) {
    block_info <- data %>%
      dplyr::distinct(.data[[xpri]], .data[[xsub]]) %>%
      dplyr::group_by(.data[[xpri]]) %>%
      dplyr::summarise(n_sub = dplyr::n(), .groups = "drop") %>%
      dplyr::mutate(
        block_start = cumsum(dplyr::lag(n_sub + 1, default = 0)),
        center_x = block_start + (n_sub - 1) / 2
      )
  } else {
    block_info <- data %>%
      dplyr::distinct(.data[[xpri]]) %>%
      dplyr::mutate(
        n_sub = 1L,
        block_start = dplyr::row_number() - 1,
        center_x = block_start + (n_sub - 1) / 2
      )
  }

  if (!is.null(xsub) && xsub != xpri) {
    pos_data <- data %>%
      dplyr::distinct(.data[[xpri]], .data[[xsub]]) %>%
      dplyr::group_by(.data[[xpri]]) %>%
      dplyr::mutate(sub_idx = dplyr::row_number() - 1) %>%
      dplyr::ungroup() %>%
      dplyr::left_join(block_info, by = xpri) %>%
      dplyr::mutate(global_x = block_start + sub_idx)
  } else {
    pos_data <- data %>%
      dplyr::distinct(.data[[xpri]]) %>%
      dplyr::mutate(sub_idx = 0L) %>%
      dplyr::left_join(block_info, by = xpri) %>%
      dplyr::mutate(global_x = block_start + sub_idx)
  }

  group_vars <- c(xpri, xsub)
  group_vars <- group_vars[!sapply(group_vars, is.null)]

  plot_data <- data %>%
    dplyr::left_join(pos_data, by = c(xpri, if (!is.null(xsub)) xsub)) %>%
    dplyr::group_by(across(all_of(group_vars))) %>%
    dplyr::arrange(.data[[fill_var]]) %>%
    dplyr::mutate(
      cumulative = cumsum(.data[[ypri]]),
      y_start = dplyr::lag(cumulative, default = 0)
    ) %>%
    dplyr::ungroup() %>%
    dplyr::left_join(block_info %>% dplyr::select(
      all_of(xpri),
      center_x
    ), by = xpri)

  color_schemes <- list(
    theme1 = c(
      "#FFBE7D", "#9b59b6", "#A0CBE8", "#59A14F",
      "#c8e6c9", "#8c564b", "#d9d5e1", "#f0b2b3"
    ),
    theme2 = c(
      "#4E79A7", "#A0CBE8", "#F28E2B", "#FFBE7D",
      "#59A14F", "#9b59b6", "#2ecc71", "#f39c12"
    ),
    theme3 = c(
      "#1f77b4", "#ff7f0e", "#2ca02c", "#d62728",
      "#9467bd", "#8c564b", "#d9d5e1", "#f0b2b3"
    ),
    theme4 = c(
      "#e377c2", "#7f7f7f", "#a0e3f4", "#9b59b6",
      "#d9d5e1", "#c8e6c9", "#f39c12", "#2ca02c"
    ),
    theme5 = c(
      "#ff6f61", "#76daff", "#9b59b6", "#f39c12",
      "#2ca02c", "#e377c2", "#d9d5e1", "#d62728"
    ),
    theme6 = c(
      "#47cf73", "#6a5acd", "#e67e22", "#34495e",
      "#ff7f0e", "#7f7f7f", "#d9d5e1", "#d62728"
    ),
    theme7 = c(
      "#1f77b4", "#d62728", "#8c564b", "#a0e3f4",
      "#f0b2b3", "#c8e6c9", "#d9d5e1", "#e377c2"
    ),
    theme8 = c(
      "#e74c3c", "#3498db", "#1abc9c", "#f39c12",
      "#9467bd", "#d9d5e1", "#7f7f7f", "#a0e3f4"
    )
  )

  if (is.null(blocks_colors)) {
    blocks_colors <- color_schemes[[theme]]
  }

  fill_levels <- levels(dplyr::pull(data, !!fill_var))
  n_fill <- length(fill_levels)
  blocks_colors <- if (n_fill > length(blocks_colors)) {
    grDevices::colorRampPalette(blocks_colors)(n_fill)
  } else {
    blocks_colors[1:n_fill]
  }
  if (rev) blocks_colors <- rev(blocks_colors)

  shared_x_scale <- ggplot2::scale_x_continuous(
    expand = c(0.01, 0.01),
    breaks = block_info$center_x,
    labels = block_info[[xpri]],
    limits = c(min(pos_data$global_x) - 0.5, max(pos_data$global_x) + 0.5)
  )

  base_plot <- ggplot2::ggplot(plot_data) +
    ggplot2::geom_rect(
      ggplot2::aes(
        xmin = global_x - 0.45,
        xmax = global_x + 0.45,
        ymin = y_start,
        ymax = cumulative,
        fill = .data[[fill_var]]
      ),
      color = border_color,
      linewidth = width
    ) +
    shared_x_scale +
    ggplot2::labs(
      y = rlang::as_label(rlang::sym(ypri)),
      x = rlang::as_label(rlang::sym(xpri))
    ) +
    ggplot2::scale_fill_manual(values = blocks_colors) +
    ggplot2::theme(
      axis.text.x = ggplot2::element_text(angle = 0, hjust = 0.5),
      legend.title = ggplot2::element_text(size = 10),
      plot.margin = ggplot2::margin(0.5, 0.5, 0.5, 0.5, "cm")
    )

  if (shape && !is.null(xsub)) {
    unique_sub <- unique(dplyr::pull(plot_data, .data[[xsub]]))
    if (length(unique_sub) > 1) {
      top_points <- plot_data %>%
        dplyr::group_by(global_x) %>%
        dplyr::slice_max(order_by = cumulative, n = 1) %>%
        dplyr::ungroup()

      base_plot <- base_plot +
        ggplot2::geom_point(
          data = top_points,
          ggplot2::aes(
            x = global_x,
            y = cumulative + 0.015 * max(cumulative),
            shape = .data[[xsub]]
          ),
          size = size,
          stroke = 0.5,
          fill = "black"
        ) +
        ggplot2::scale_shape_manual(
          values = seq_len(length(unique_sub)),
          name = xsub
        )
    }
  }

  sub_plot <- if (!is.null(ysub)) {
    if (!is.null(xsub)) {
      first_sub_data <- data %>%
        dplyr::group_by(.data[[xpri]]) %>%
        dplyr::filter(.data[[xsub]] == levels(.data[[xsub]])[1]) %>%
        dplyr::distinct(.data[[xpri]], .data[[ysub]], .data[[fill_var]])
    } else {
      first_sub_data <- data %>%
        dplyr::group_by(.data[[xpri]]) %>%
        dplyr::distinct(.data[[xpri]], .data[[ysub]], .data[[fill_var]]) %>%
        dplyr::slice(1)
    }

    sub_plot_data <- block_info %>%
      dplyr::select(all_of(xpri), center_x) %>%
      dplyr::left_join(first_sub_data, by = xpri) %>%
      dplyr::filter(!is.na(.data[[ysub]])) %>%
      dplyr::mutate(global_x = center_x)

    sub_plot <- if (sub_axis == "fill") {
      ggplot2::ggplot(sub_plot_data) +
        ggplot2::geom_label(
          ggplot2::aes(
            x = global_x,
            y = .data[[fill_var]],
            label = .data[[ysub]],
            fill = .data[[fill_var]]
          ),
          size = 2.8,
          label.size = 0,
          color = "black"
        )
    } else {
      ggplot2::ggplot(sub_plot_data) +
        ggplot2::geom_label(
          ggplot2::aes(
            x = global_x,
            y = .data[[ysub]],
            label = .data[[ysub]],
            fill = .data[[fill_var]]
          ),
          size = 2.8,
          label.size = 0,
          color = "black"
        )
    }

    sub_plot +
      shared_x_scale +
      ggplot2::scale_y_discrete(expand = c(0.1, 0.1)) +
      ggplot2::scale_fill_manual(values = blocks_colors) +
      ggplot2::labs(y = NULL, x = NULL) +
      ggplot2::theme(
        legend.position = "none",
        axis.line = ggplot2::element_blank(),
        axis.text = ggplot2::element_blank(),
        axis.ticks = ggplot2::element_blank()
      )
  } else {
    NULL
  }

  if (merge) {
    if (!is.null(sub_plot)) {
      patchwork::wrap_plots(
        base_plot,
        sub_plot,
        ncol = 1,
        heights = c(3.2, 1)
      )
    } else {
      base_plot +
        ggplot2::theme(plot.margin = ggplot2::margin(0.5, 0.5, 0.5, 0.5, "cm"))
    }
  } else {
    base_plot
  }
}
