% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dist2land.R
\name{dist2land}
\alias{dist2land}
\title{Calculate distance to the closest land for coordinates}
\usage{
dist2land(
  data,
  lon = NULL,
  lat = NULL,
  shapefile = "DecimalDegree",
  proj.in = 4326,
  bind = TRUE,
  dist.col = "ldist",
  binary = FALSE,
  verbose = TRUE
)
}
\arguments{
\item{data}{Data frame or \link[sf:sf]{sf} object containing geographic coordinates.}

\item{lon, lat}{Either the names of the longitude and latitude columns in \code{data} or \code{NULL} to \link[=guess_coordinate_columns]{guess the longitude and/or latitude columns} in \code{data}.}

\item{shapefile}{Land shape to which distances should be calculated. Either a character argument referring to a name of pre-made shapefiles in \code{\link{shapefile_list}}, a single \link[sf]{sf} or \code{sp} polygons object object or \code{NULL} to enable automatic definition of the land shapes based on \code{data}. Set to \code{"DecimalDegree"} by default which enables great circle distances using \link[sf]{s2} features assuming a spherical Earth (as a contrast to earlier versions of the function which used flat Earth).}

\item{proj.in}{\code{\link[sf:st_crs]{coordinate reference system}} of \code{data}.}

\item{bind}{Logical indicating whether \code{x} should be returned with the distances (\code{TRUE}, default) or should the distances be returned as vector (\code{FALSE}).}

\item{dist.col}{The name of the distance column, if \code{bind = TRUE}. Defaults to "ldist".}

\item{binary}{Logical indicating whether binary (TRUE = the position is in the ocean, FALSE = the position is on land) should be returned instead of distances. Speeds up the function considerably.}

\item{verbose}{Logical indicating whether information about the process should be returned as messages. Set to \code{FALSE} to make the function silent.}
}
\value{
Returns a vector if \code{bind = FALSE}, otherwise a data frame. The distances are given in a new column defined by the \code{dist.col} argument. The distances are \strong{kilometers} if \code{binary = FALSE}, otherwise logical (TRUE = the position is in the ocean, FALSE = the position is on land).
}
\description{
Calculates the closest distance to land for coordinates in a data frame
}
\details{
The function calculates great circle spherical distances using the \code{\link[sf]{st_distance}} function by default. The function can be slow for large datasets. If you only want to use the function to remove (wrong) observations reported on land, set the \code{binary} argument to \code{TRUE}. This speeds up the calculations by a factor of ten.
}
\examples{
# Simple example:
dt <- data.frame(lon = seq(-20, 80, length.out = 41), lat = 50:90)
dt <- dist2land(dt, verbose = FALSE)
\donttest{
qmap(dt, color = ldist) + scale_color_viridis_c()

# Datasets covering the entire Earth seem to work now, except 0,0 lon/lat point
lon = deg_to_dd(seq(0,360,30)); lat = c(80,50,20,0,-20,-50,-80)

dt <- data.frame(
 lon = rep(lon, length(lat)), lat = rep(lat, each = length(lon)))

qmap(dist2land(dt, verbose = FALSE), color = ldist) +
 scale_color_viridis_c()
}
\dontrun{
dt <- data.frame(
  lon = deg_to_dd(seq(0,360,length.out = 1e3)), 
  lat = rep(60, 1000))
  
# The distance calculation is slow for large datasets
system.time(dist2land(dt))
# user  system elapsed 
# 12.677   0.146  12.849 

# binary = TRUE speeds the function up
system.time(dist2land(dt, binary = TRUE))
# user  system elapsed 
# 1.239   0.120   1.369 
}
}
\author{
Mikko Vihtakari
}
