#' Z_curve_12bit_DNA (Zcurve12bit_DNA)
#'
#' These group of functions (Zcurve (9, 12, 36, 48, 144)_bit) function calculates the Z-curves. Z-curves are based on freqiencies of nucleotides, di-nucleotides, or tri-nucleotides and their positions on the sequences.
#' For more information about the methods please refer to reference part.
#'
#' @references Gao,F. and Zhang,C.T. Comparison of various algorithms for recognizing short coding sequences of human genes. Bioinformatics, (2004).
#'
#' @param seqs is a FASTA file containing nucleotide sequences. The sequences start
#' with '>'. Also, seqs could be a string vector. Each element of the vector is a nucleotide sequence.
#'
#' @param ORF (Open Reading Frame) is a logical parameter. If it is set to true, ORF region of each sequence is considered instead of the original sequence (i.e., 3-frame).
#'
#' @param reverseORF is a logical parameter. It is enabled only if ORF is true.
#' If reverseORF is true, ORF region will be searched in the sequence and also in the reverse complement of the sequence (i.e., 6-frame).
#'
#'
#' @param label is an optional parameter. It is a vector whose length is equivalent to the number of sequences. It shows the class of
#' each entry (i.e., sequence).
#'
#' @return This function returns a feature matrix. The number of rows is equal to the number of sequences and
#' the number of columns is 12.
#'
#'
#' @export
#'
#' @examples
#'
#' fileLNC<-system.file("extdata/Athaliana_LNCRNA.fa",package="ftrCOOL")
#' mat<-Zcurve12bit_DNA(seqs=fileLNC,ORF=TRUE,reverseORF=FALSE)


Zcurve12bit_DNA<-function(seqs,ORF=FALSE,reverseORF=TRUE,label=c()){


  if(length(seqs)==1&&file.exists(seqs)){
    seqs<-fa.read(seqs,alphabet="dna")
    seqs_Lab<-alphabetCheck(seqs,alphabet = "dna",label)

    seqs<-seqs_Lab[[1]]
    label<-seqs_Lab[[2]]
  }
  else if(is.vector(seqs)){
    seqs<-sapply(seqs,toupper)

    seqs_Lab<-alphabetCheck(seqs,alphabet = "dna",label)

    seqs<-seqs_Lab[[1]]
    label<-seqs_Lab[[2]]

  }
  else {
    stop("ERROR: Input sequence is not in the correct format. It should be a FASTA file or a string vector.")
  }
  flag=0
  if(ORF==TRUE){
    if(length(label)==length(seqs)){
      names(label)=names(seqs)
      flag=1
    }
    seqs=maxORF(seqs,reverse=reverseORF)
    if(flag==1)
      label=label[names(seqs)]
  }


  numSeqs<-length(seqs)
  lenSeqs<-sapply(seqs,nchar)

  DiNuc<-nameKmer(k=2,type = "dna")
  featureMatrix<-matrix(ncol = 12,nrow = numSeqs)
  for(n in 1:numSeqs){
    seq<-seqs[n]
    lenSeq<-lenSeqs[n]
    seqChars<-unlist(strsplit(seq,split = ""))
    temp<-seqChars[1:(lenSeq-1)]
    temp2<-seqChars[2:lenSeq]
    Dimer<-paste(temp,temp2,sep = "")
    tabdimer<-table(Dimer)
    freqDim<-rep(0,16)
    names(freqDim)<-DiNuc
    freqDim[names(tabdimer)]<-tabdimer

    Nucs<-c("A","C","G","T")
    mat<-matrix(nrow = 4,ncol = 3)
    colnames(mat)<-c("x","y","z")
    rownames(mat)<-c("A","C","G","T")
    for(i in 1:4){
      Pxa=freqDim[paste(Nucs[i],"A",sep="")]
      Pxg=freqDim[paste(Nucs[i],"G",sep="")]
      Pxc=freqDim[paste(Nucs[i],"C",sep="")]
      Pxt=freqDim[paste(Nucs[i],"T",sep="")]
      mat[i,"x"]<-(Pxa+Pxg)-(Pxc+Pxt)
      mat[i,"y"]<-(Pxa+Pxc)-(Pxg+Pxt)
      mat[i,"z"]<-(Pxa+Pxt)-(Pxg+Pxc)
    }

    mat<-t(mat)
    vecMat<-as.vector(mat)
    featureMatrix[n,]<-vecMat/(lenSeq-1)

  }
  tempName1<-rep(c("x","y",'z'),4)
  tempName2<-rep(c("A","C","G","T"),each=3)
  tmp<-paste0(tempName2,".",tempName1)
  colnames(featureMatrix)<-tmp

  if(length(label)==numSeqs){
    featureMatrix<-as.data.frame(featureMatrix)
    featureMatrix<-cbind(featureMatrix,label)
  }

  return(featureMatrix)

}
