% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mle_foot.R
\name{mle_foot}
\alias{mle_foot}
\title{Fit football models with Maximum Likelihood}
\usage{
mle_foot(
  data,
  model,
  predict = 0,
  maxit = 1000,
  method = "BFGS",
  interval = "profile",
  hessian = FALSE,
  sigma_y = 1
)
}
\arguments{
\item{data}{A data frame containing match data with columns:
\itemize{
  \item \code{periods}:  Time point of each observation (integer >= 1).
  \item \code{home_team}: Home team's name (character string).
  \item \code{away_team}: Away team's name (character string).
  \item \code{home_goals}: Goals scored by the home team (integer >= 0).
  \item \code{away_goals}: Goals scored by the away team (integer >= 0).
}}

\item{model}{A character specifying the model to fit. Options are:
\itemize{
  \item \code{"double_pois"}: Double Poisson model.
  \item \code{"biv_pois"}: Bivariate Poisson model.
  \item \code{"skellam"}: Skellam model.
  \item \code{"student_t"}: Student's t model.
  }}

\item{predict}{An integer specifying the number of out-of-sample matches for prediction. If missing, the function fits the model to the entire dataset without making predictions.}

\item{maxit}{An integer specifying the maximum number of optimizer iterations  default is 1000).}

\item{method}{A character specifying the optimization method. Options are
  \itemize{
    \item \code{"Nelder-Mead"}.
    \item \code{"BFGS"} (default).
    \item \code{"CG"}.
    \item \code{"L-BFGS-B"}.
    \item \code{"SANN"}.
    \item \code{"Brent"}.
  }
For further details see \code{{optim}} function in \code{\link[stats]{stats}} package.}

\item{interval}{A character specifying the interval type for confidence intervals. Options are
\itemize{
  \item \code{"profile"} (default).
  \item \code{"Wald"}.
  }}

\item{hessian}{A logical value indicating to include the computation of the Hessian (default FALSE).}

\item{sigma_y}{A positive numeric value indicating the scale parameter for Student t likelihood (default 1).}
}
\value{
A named list containing:
\itemize{
  \item{\code{att}}: A matrix of attack ratings, with MLE and 95\% confidence intervals (for \code{"double_pois"}, \code{"biv_pois"} and \code{"skellam"} models).
  \item{\code{def}}: A matrix of defence ratings, with MLE and 95\% confidence intervals (for \code{"double_pois"}, \code{"biv_pois"} and \code{"skellam"} models).
  \item{\code{abilities}}: A matrix of combined ability, with MLE and 95\% confidence intervals (for \code{"student_t"} only).
  \item{\code{home_effect}}: A matrix with with MLE and 95\% confidence intervals for the home effect estimate.
  \item{\code{corr}}: A matrix with MLE and 95\% confidence intervals for the bivariate Poisson correlation parameter (for \code{"biv_pois"} only).
  \item{\code{model}}: The name of the fitted model (character).
  \item{\code{predict}}: The number of out-of-sample matches used for prediction (integer).
  \item{\code{sigma_y}}: The scale parameter used in the Student t likelihood (for \code{"student_t"} only).
  \item{\code{team1_prev}}: Integer indices of home teams in the out-of-sample matches (if \code{predict > 0}).
  \item{\code{team2_prev}}: Integer indices of away teams in the out-of-sample matches (if \code{predict > 0}).
  \item{\code{logLik}}: The maximized log likelihood (numeric).
  \item{\code{aic}}: Akaike Information Criterion (numeric).
  \item{\code{bic}}: Bayesian Information Criterion (numeric).
}
}
\description{
Fits football goal-based models using maximum likelihood estimation.
Supported models include: double Poisson, bivariate Poisson, Skellam, and Student's t.
}
\details{
MLE can be obtained only for static models, with no time-dependence.
Likelihood optimization is performed via the \code{BFGS} method
of the \code{{optim}} function in \code{\link[stats]{stats}} package.
}
\examples{
\dontrun{
library(dplyr)

data("italy")
italy <- as_tibble(italy)
italy_2000_2002 <- italy \%>\%
  dplyr::select(Season, home, visitor, hgoal, vgoal) \%>\%
  dplyr::filter(Season == "2000" | Season == "2001" | Season == "2002")

colnames(italy_2000_2002) <- c("periods", "home_team", "away_team", "home_goals", "away_goals")

mle_fit <- mle_foot(
  data = italy_2000_2002,
  model = "double_pois"
)
}

}
\references{
Baio, G. and Blangiardo, M. (2010). Bayesian hierarchical model for the prediction of football
results. Journal of Applied Statistics 37(2), 253-264.

Egidi, L., Pauli, F., and Torelli, N. (2018). Combining historical data
and bookmakers' odds in modelling football scores. Statistical Modelling, 18(5-6), 436-459.

Gelman, A. (2014). Stan goes to the World Cup. From
"Statistical Modeling, Causal Inference, and Social Science" blog.

Karlis, D. and Ntzoufras, I. (2003). Analysis of sports data by using bivariate poisson models.
Journal of the Royal Statistical Society: Series D (The Statistician) 52(3), 381-393.

Karlis, D. and Ntzoufras,I. (2009).  Bayesian modelling of football outcomes: Using
the Skellam's distribution for the goal difference. IMA Journal of Management Mathematics 20(2), 133-145.

Owen, A. (2011). Dynamic Bayesian forecasting models
of football match outcomes with estimation of the
evolution variance parameter. IMA Journal of Management Mathematics, 22(2), 99-113.
}
\author{
Leonardo Egidi \email{legidi@units.it} and Roberto Macrì Demartino \email{roberto.macridemartino@deams.units.it}
}
