% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fmrHP.R
\name{fmrHP}
\alias{fmrHP}
\title{Finite Mixture Effects Model with Heterogeneity Pursuit}
\usage{
fmrHP(y, X, m, intercept = FALSE, lambda, equal.var = FALSE,
      ic.type = c("ALL", "BIC", "AIC", "GIC"),
      B = NULL, prob = NULL, rho = NULL, w = NULL,
      control = list(), report = FALSE)
}
\arguments{
\item{y}{a vector of response (\eqn{n \times 1})}

\item{X}{a matrix of covariate (\eqn{n \times p})}

\item{m}{number of components}

\item{intercept}{indicating whether intercept should be included}

\item{lambda}{value of tuning parameter}

\item{equal.var}{indicating whether variances of different components are equal}

\item{ic.type}{the information criterion to be used; currently supporting "AIC", "BIC", and "GIC".}

\item{B}{initial values for the rescaled coefficients with first column being the 
common effect, and the rest \code{m} columns being the heterogeneity for 
corresponding components}

\item{prob}{initial values for prior probabilitis for different components}

\item{rho}{initial values for rho vector (\eqn{1 / \sigma}), the reciprocal of standard deviation}

\item{w}{weight matrix for penalty function. Default option is NULL}

\item{control}{a list of parameters for controlling the fitting process}

\item{report}{indicating whether printing the value of objective function during EM algorithm 
for validation checking of initial value.}
}
\value{
A list consisting of
\item{y}{vector of response}
\item{X}{matrix of covariates}
\item{m}{number of components}
\item{B.hat}{estimated rescaled coefficient (\eqn{p \times m + 1 \times nlambda})}
\item{pi.hat}{estimated prior probabilities (\eqn{m \times nlambda})}
\item{rho.hat}{estimated rho values (\eqn{m \times nlambda})}
\item{lambda}{lambda used in model fitting}
\item{plik}{value of penalized log-likelihood}
\item{loglik}{value of log-likelihood}
\item{conv}{indicator of convergence of EM algorithm}
\item{IC}{values of information criteria}
\item{df}{degree of freedom}
}
\description{
Produce solution for specified lambda of regularized finite mixture effects model with lasso 
or adaptive lasso; compute the degrees of freeom, likelihood and information criteria (AIC, BIC and GIC) 
of the estimators. Model fitting is conducted by EM algorithm and Bregman coordinate descent.
}
\details{
The available elements for argument \code{control} include
\itemize{
   \item{epsilon}: Convergence threshold for generalized EM algorithm.
       Defaults value is 1E-6.
   
   \item{maxit}: Maximum number of passes over the data for all lambda values.
       Default is 1000.
   
   \item{inner.eps}: Convergence threshold for Bregman coordinate descent algorithm.
       Defaults value is 1E-6.
   
   \item{inner.maxit}: Maximum number of iteration for Bregman coordinate descent algorithm.
       Defaults value is 200.
   
   \item{n.ini}: Number of initial values for EM algorithm. Default is 10. In EM algorithm, it is 
       preferable to start from several different initial values.
}
}
\examples{
library(fmerPack)
## problem settings
n <- 100; m <- 3; p <- 5;
sigma2 <- c(0.1, 0.1, 0.4); rho <- 1 / sqrt(sigma2)
phi <- rbind(c(1, 1, 1), c(1, 1, 1), c(0, -3, 3), c(-3, 3, 0), c(3, 0, -3))
beta <- t(t(phi) / rho)
## generate response and covariates
z <- rmultinom(n, 1, prob= rep(1 / m, m))
X <- matrix(rnorm(n * p), nrow = n, ncol = p)
y <- MASS::mvrnorm(1, mu = rowSums(t(z) * X[, 1:(nrow(beta))] \%*\% beta), 
                   Sigma = diag(colSums(z * sigma2)))
fmrHP(y, X, m = m, lambda = 0.01, control = list(n.ini = 10))
}
