% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/distGDM2.R, R/distGower.R, R/distSimMatch.R
\name{distGDM2}
\alias{distGDM2}
\alias{distGower}
\alias{distances}
\alias{distSimMatch}
\title{Distance Functions for K-Centroids Clustering of (Ordinal) Categorical/Mixed Data}
\usage{
distGDM2(x, centers, genDist, xrange = NULL)

distGower(x, centers, genDist)

distSimMatch(x, centers)
}
\arguments{
\item{x}{A numeric matrix or data frame.}

\item{centers}{A numeric matrix with \code{ncol(centers)} equal to
\code{ncol(x)} and \code{nrow(centers)} smaller or equal to \code{row(x)}.}

\item{genDist}{Additional information on \code{x} required for distance
calculation.  Filled automatically if used within
\code{\link[flexclust:kcca]{flexclust::kcca()}}.
\itemize{
\item For \code{distGower}: A character vector of variable specific
distances to be used with length equal to \code{ncol(x)}. The
following options are possible:
\itemize{
\item \code{distEuclidean}: Euclidean distance between the scaled variables.
\item \code{distManhattan}: absolute distance between the scaled variables.
\item \code{distJaccard}: counts of zero if both binary variables are
equal to 1, and 1 otherwise.
\item \code{distSimMatch}: Simple Matching Distance, i.e. the number of
agreements between variables.
}
\item For \code{distGDM2}: Function creating a distance function that will
be primed on \code{x}.
\item For \code{distSimMatch}: not used.
}}

\item{xrange}{Range specification for the variables. Currently only
used for \code{distGDM2} (as \code{distGower} expects \code{x} to be already
scaled). Possible values are:
\itemize{
\item \code{NULL} (default): defaults to \code{"all"}.
\item \code{"all"}: uses the same minimum and maximum value for each column
of \code{x} by determining the whole range of values in the data
object \code{x}.
\item \code{"columnwise"}: uses different minimum and maximum values for
each column of \code{x} by determining the columnwise ranges of
values in the data object \code{x}.
\item A vector of \code{c(min, max)}: specifies the same minimum and maximum
value for each column of \code{x}.
\item A list of vectors \code{list(c(min1, max1), c(min2, max2),...)} with
length \code{ncol(x)}: specifies different minimum and maximum
values for each column of \code{x}.
}}
}
\value{
A matrix of dimensions \code{c(nrow(x), nrow(centers))} that contains the distance
between each row of \code{x} and each row of \code{centers}.
}
\description{
Functions to calculate the distance between a matrix \code{x} and a
matrix \code{c}, which can be used for K-centroids clustering via
\code{\link[flexclust:kcca]{flexclust::kcca()}}.

\code{distSimMatch} implements Simple Matching Distance (most frequently
used for categorical, or symmetric binary data) for K-centroids
clustering.

\code{distGower} implements Gower's Distance after Gower (1971) and
Kaufman & Rousseeuw (1990) for mixed-type data with missings for K-centroids
clustering.

\code{distGDM2} implements GDM2 distance for ordinal data introduced by
Walesiak et al. (1993) and adapted to K-centroids clustering by
Ernst et al. (2025).

These functions are designed for use with \code{\link[flexclust:kcca]{flexclust::kcca()}} or
functions that are built upon it. Their use is easiest via the
wrapper \code{\link[=kccaExtendedFamily]{kccaExtendedFamily()}}.  However, they can also easily be
used to obtain a distance matrix of \code{x}, see Examples.
}
\details{
\itemize{
\item \code{distSimMatch}: Simple Matching Distance between two observations
is calculated as the proportion of disagreements acros all
variables. Described, e.g., in Kaufman & Rousseeuw (1990), p. 24.
If this is used in K-centroids analysis in combination with mode
centroids (as implemented in \code{centMode}), this results in the
\strong{kModes} algorithm.  A wrapper for this algorithm is obtained
with \code{kccaExtendedFamily(which='kModes')}.
\item \code{distGower}: Distances are calculated for each column (Euclidean
distance, \code{distEuclidean}, is recommended for numeric, Manhattan
distance, \code{distManhattan} for ordinal, Simple Matching Distance,
\code{distSimMatch} for categorical, and Jaccard distance,
\code{distJaccard} for asymmetric binary variables), and they are
summed up as:

\deqn{d(x_i, x_k) = \frac{\sum_{j=1}^p \delta_{ikj} d(x_{ij},
  x_{kj})}{\sum_{j=1}^p \delta_{ikj}}}

where \eqn{p} is the number of variables and with the weight
\eqn{\delta_{ikj}} being 1 if both values \eqn{x_{ij}} and
\eqn{x_{kj}} are not missing, and in the case of asymmetric
binary variables, at least one of them is not 0.  Please note
that for calculating Gower's distance, scaling of numeric/ordered
variables is required (as f.i. by \code{.ScaleVarSpecific}).  A
wrapper for K-centroids analysis using Gower's distance in
combination with a numerically optimized centroid is found in
\code{kccaExtendedFamily(which='kGower')}.
\item \code{distGDM2}: GDM2 distance for ordinal variables conducts only
relational operations on the variables, such as \eqn{\leq},
\eqn{\geq} and \eqn{=}. By translating \eqn{x} to its relative
frequencies and empirical cumulative distributions, we are able
to extend this principle to compare two arbitrary values, and
thus use it within K-centroids clustering. For more details, see
Ernst et al. (2025).  A wrapper for this algorithm in
combination with a numerically optimized centroid is found in
\code{kccaExtendedFamily(which='kGDM2')}.
}

The distances functions presented here can also be used in clustering algorithms that
rely on distance matrices (such as hierarchical clustering and PAM), if applied
accordingly, see Examples.
}
\examples{
# Example 1: Simple Matching Distance
set.seed(123)
dat <- data.frame(question1 = factor(sample(LETTERS[1:4], 10, replace=TRUE)),
                  question2 = factor(sample(LETTERS[1:6], 10, replace=TRUE)),
                  question3 = factor(sample(LETTERS[1:4], 10, replace=TRUE)),
                  question4 = factor(sample(LETTERS[1:5], 10, replace=TRUE)),
                  state = factor(sample(state.name[1:10], 10, replace=TRUE)),
                  gender = factor(sample(c('M', 'F', 'N'), 10, replace=TRUE,
                                         prob=c(0.45, 0.45, 0.1))))
datmat <- data.matrix(dat)
initcenters <- datmat[sample(1:10, 3),]
distSimMatch(datmat, initcenters)
## within kcca
flexclust::kcca(dat, k=3, family=kccaExtendedFamily('kModes'))
## as a distance matrix
as.dist(distSimMatch(datmat, datmat))

# Example 2: GDM2 distance
flexclust::kcca(dat, k=3, family=kccaExtendedFamily('kGDM2'))

# Example 3: Gower's distance
# Ex. 3.1: single variable type case with no missings:
flexclust::kcca(datmat, 3, kccaExtendedFamily('kGower'))

# Ex. 3.2: single variable type case with missing values:
nas <- sample(c(TRUE, FALSE), prod(dim(dat)), replace = TRUE,
   prob=c(0.1, 0.9)) |> 
   matrix(nrow = nrow(dat))
dat[nas] <- NA
flexclust::kcca(dat, 3, kccaExtendedFamily('kGower', cent=centMode))

#Ex. 3.3: mixed variable types (with or without missings): 
dat <- data.frame(cont = sample(1:100, 10, replace=TRUE)/10,
                  bin_sym = as.logical(sample(0:1, 10, replace=TRUE)),
                  bin_asym = as.logical(sample(0:1, 10, replace=TRUE)),                     
                  ord_levmis = factor(sample(1:5, 10, replace=TRUE),
                                      levels=1:6, ordered=TRUE),
                  ord_levfull = factor(sample(1:4, 10, replace=TRUE),
                                       levels=1:4, ordered=TRUE),
                  nom = factor(sample(letters[1:4], 10, replace=TRUE),
                               levels=letters[1:4]))
dat[nas] <- NA
flexclust::kcca(dat, 3, kccaExtendedFamily('kGower'))

}
\references{
\itemize{
\item Ernst, D, Ortega Menjivar, L, Scharl, T, Grün, B (2025).
\emph{Ordinal Clustering with the flex-Scheme.}
Austrian Journal of Statistics. \emph{Submitted manuscript}.
\item Gower, JC (1971).
\emph{A General Coefficient for Similarity and Some of Its Properties.}
Biometrics, 27(4), 857-871.
\doi{10.2307/2528823}
\item Kaufman, L, Rousseeuw, P (1990).
\emph{Finding Groups in Data: An Introduction to Cluster Analysis.}
Wiley Series in Probability and Statistics.
\doi{10.1002/9780470316801}
\item Leisch, F (2006). \emph{A Toolbox for K-Centroids Cluster Analysis.}
Computational Statistics and Data Analysis, 17(3), 526-544.
\doi{10.1016/j.csda.2005.10.006}
\item Kaufman, L, Rousseeuw, P (1990.) \emph{Finding Groups in Data: An Introduction to Cluster Analysis.}
Wiley Series in Probability and Statistics, New York: John Wiley & Sons.
\doi{10.1002/9780470316801}
\item Walesiak, M (1993). \emph{Statystyczna Analiza Wielowymiarowa w Badaniach Marketingowych.}
Wydawnictwo Akademii Ekonomicznej, 44-46.
\item Weihs, C, Ligges, U, Luebke, K, Raabe, N (2005). \emph{klaR Analyzing German Business Cycles}.
In Baier D, Decker, R, Schmidt-Thieme, L (eds.). Data Analysis and Decision Support,
335-343. Berlin: Springer-Verlag.
\doi{10.1007/3-540-28397-8_36}
}
}
\seealso{
\code{\link[flexclust:kcca]{flexclust::kcca()}},
\code{\link[klaR:kmodes]{klaR::kmodes()}},
\code{\link[cluster:daisy]{cluster::daisy()}},
\code{\link[clusterSim:dist.GDM]{clusterSim::dist.GDM()}}
}
