#include "filters.h"
#include <Rcpp.h>
using namespace Rcpp;

// Use C++ 11
// ==========

// [[Rcpp::plugins("cpp11")]]

//' Compute the Wavelet Filter
//'
//' Compute the wavelet filter.
//'
//' @param wavelet A character string indicating the wavelet filter desired
//' @return Wavelet filter vector (a numeric vector)
//' @references
//' Percival, D. B. and A. T. Walden (2000) Wavelet Methods for Time Series Analysis, Cambridge
//' University Press.
//' @export
// [[Rcpp::export]]
NumericVector wavelet_filter(String wavelet) {

  NumericVector g = scaling_filter(wavelet);
  int L = g.size();

  NumericVector h(L);

  for(int i = 0; i <= L-1; ++i)
  {
    h(L-i-1) = g(i) / pow(-1, i+1);
  }

  return h;
}

//' Scaling Filter
//'
//' Compute the scaling filter.
//'
//' @param wavelet A character string indicating the scaling filter desired
//' @return Scaling filter vector (a numeric vector)
//' @details The argument `wavelet` can take one of the following values:
//'
//' `c("haar", "d1", "sym1", "bior1.1", "rbio1.1",
//' "d2", "sym2", "d3", "sym3", "d4", "d5", "d6", "d7", "d8", "d9", "d10", "d11",
//' "sym4", "sym5", "sym6", "sym7", "sym8", "sym9", "sym10",
//' "coif1", "coif2", "coif3", "coif4", "coif5",
//' "bior1.3", "bior1.5", "bior2.2", "bior2.4", "bior2.6", "bior2.8", "bior3.1", "bior3.3",
//' "bior3.5", "bior3.7", "bior3.9", "bior4.4", "bior5.5", "bior6.8",
//' "rbio1.3", "rbio1.5", "rbio2.2", "rbio2.4", "rbio2.6", "rbio2.8", "rbio3.1", "rbio3.3",
//' "rbio3.5", "rbio3.7", "rbio3.9", "rbio4.4", "rbio5.5", "rbio6.8",
//' "la8", "la10", "la12", "la14", "la16", "la18", "la20",
//' "bl14", "bl18", "bl20",
//' "fk4", "fk6", "fk8", "fk14", "fk18", "fk22",
//' "b3spline",
//' "mb4.2", "mb8.2", "mb8.3", "mb8.4", "mb10.3", "mb12.3", "mb14.3", "mb16.3", "mb18.3", "mb24.3", "mb32.3",
//' "beyl",
//' "vaid",
//' "han2.3", "han3.3", "han4.5", "han5.5")`
//'
//' @references
//' Percival, D. B. and A. T. Walden (2000) Wavelet Methods for Time Series Analysis, Cambridge
//' University Press.
//'
//' Wasilewski, F. (2008). Wavelet browser by pywavelets. Wavelet Properties Browser.
//' Retrieved November 17, 2022, from http://wavelets.pybytes.com/
//'
//' Gregory R. Lee, Ralf Gommers, Filip Wasilewski, Kai Wohlfahrt, Aaron O’Leary (2019).
//' PyWavelets: A Python package for wavelet analysis. Journal of Open Source Software,
//' 4(36), 1237, https://doi.org/10.21105/joss.01237.
//'
//' Olhede, S., &amp; Walden, A. T. (2004). The Hilbert spectrum via wavelet projections.
//' Proceedings of the Royal Society of London. Series A: Mathematical, Physical and
//' Engineering Sciences, 460(2044), 955–975. https://doi.org/10.1098/rspa.2003.1199
//'
//' Maheswaran, R., &amp; Khosa, R. (2012). Comparative study of different wavelets for
//' hydrologic forecasting. Computers &amp; Geosciences, 46, 284–295.
//' https://doi.org/10.1016/j.cageo.2011.12.015
//' @export
// [[Rcpp::export]]
NumericVector scaling_filter(String wavelet)
{
  NumericVector g;

  // Haar

  if (wavelet == "haar")
  {
    g = {sqrt(2) * 0.5, sqrt(2) * 0.5};
  }

  else if (wavelet == "d1")
  {
    g = {sqrt(2) * 0.5, sqrt(2) * 0.5};
  }
  else if (wavelet == "sym1")
  {
    g = {sqrt(2) * 0.5, sqrt(2) * 0.5};
  }
  else if (wavelet == "bior1.1")
  {
    g = {sqrt(2) * 0.5, sqrt(2) * 0.5};
  }
  else if (wavelet == "rbio1.1")
  {
    g = {sqrt(2) * 0.5, sqrt(2) * 0.5};
  }

  // Debauchies
  else if (wavelet == "d2")
  {
    g = {0.4829629131445341, 0.8365163037378077, 0.2241438680420134,
         -0.1294095225512603};
  }
  else if (wavelet == "sym2")
  {
    g = {0.4829629131445341, 0.8365163037378077, 0.2241438680420134,
         -0.1294095225512603};
  }
  else if (wavelet == "d3")
  {
    g = {0.3326705529500827, 0.8068915093110928, 0.4598775021184915,
         -0.1350110200102546, -0.0854412738820267, 0.0352262918857096};
  }
  else if (wavelet == "sym3")
  {
    g = {0.3326705529500827, 0.8068915093110928, 0.4598775021184915,
         -0.1350110200102546, -0.0854412738820267, 0.0352262918857096};
  }
  else if (wavelet == "d4")
  {
    g = {0.230377813308855, 0.714846570552542, 0.630880767929590,
         -0.0279837694169839, -0.187034811718881, 0.0308413818359870,
         0.0328830116669829, -0.0105974017849973};
  }
  else if (wavelet == "d5")
  {
    g = {0.160102397974125, 0.603829269797473, 0.724308528438574,
         0.138428145901103, -0.242294887066190, -0.0322448695850295,
         0.0775714938400652, -0.00624149021301171, -0.0125807519990155,
         0.00333572528500155};
  }
  else if (wavelet == "d6")
  {
    g = {0.111540743350080, 0.494623890398385, 0.751133908021578,
         0.315250351709243, -0.226264693965169, -0.129766867567096,
         0.0975016055870794, 0.0275228655300163, -0.0315820393180312,
         0.000553842200993802, 0.00477725751101065, -0.00107730108499558};
  }
  else if (wavelet == "d7")
  {
    g = {0.0778520540850624, 0.396539319482306, 0.729132090846555,
         0.469782287405359, -0.143906003929106, -0.224036184994166,
         0.0713092192670500, 0.0806126091510659, -0.0380299369350346,
         -0.0165745416310156, 0.0125509985560138, 0.000429577973004703,
         -0.00180164070399983, 0.000353713800001040};
  }
  else if (wavelet == "d8")
  {
    g = {0.0544158422430816, 0.312871590914466, 0.675630736298013,
         0.585354683654869, -0.0158291052560239, -0.284015542962428,
         0.000472484573997973, 0.128747426620186, -0.0173693010020221,
         -0.0440882539310647, 0.0139810279170155, 0.00874609404701566,
         -0.00487035299301066, -0.000391740372995977,
         0.000675449405998557, -0.000117476784002282};
  }
  else if (wavelet == "d9")
  {
    g = {0.0380779473631673, 0.243834674637667, 0.604823123676779,
         0.657288078036639, 0.133197385822089, -0.293273783272587,
         -0.0968407832208790, 0.148540749334760, 0.0307256814783229,
         -0.0676328290595240, 0.000250947114991938, 0.0223616621235152,
         -0.00472320475789483, -0.00428150368190472,
         0.00184764688296113, 0.000230385763995413,
         -0.000251963188998179, 3.93473199950261e-05};
  }
  else if (wavelet == "d10")
  {
    g = {0.0266700579009508, 0.188176800077621, 0.527201188930920,
         0.688459039452592, 0.281172343660427, -0.249846424326489,
         -0.195946274376597, 0.127369340335743, 0.0930573646038066,
         -0.0713941471658608, -0.0294575368219457, 0.0332126740589332,
         0.00360655356698839, -0.0107331754829796, 0.00139535174699408,
         0.00199240529499085, -0.000685856695004683,
         -0.000116466854994386, 9.35886700010899e-05,
         -1.32642030023549e-05};
  }
  else if (wavelet == "d11")
  {
    g = {0.0186942977614714, 0.144067021150627, 0.449899764356052,
         0.685686774916211, 0.411964368947914, -0.162275245027494,
         -0.274230846817954, 0.0660435881966801, 0.149812012466376,
         -0.0464799551166885, -0.0664387856950288, 0.0313350902190449,
         0.0208409043601805, -0.0153648209062023, -0.00334085887301471,
         0.00492841765605901, -0.000308592858815184,
         -0.000893023250666286, 0.000249152523552826,
         5.44390746993690e-05, -3.46349841869855e-05,
         4.49427427723659e-06};
  }

  //  Symlets
  else if (wavelet == "sym4")
  {
    g = {0.0322231006040427, -0.0126039672620378, -0.0992195435768472,
         0.297857795605277, 0.803738751805916, 0.497618667632015,
         -0.0296355276459985, -0.0757657147892733};
  }
  else if (wavelet == "sym5")
  {
    g = {0.0195388827352867, -0.0211018340247589, -0.175328089908450,
         0.0166021057645223, 0.633978963458212, 0.723407690402421,
         0.199397533977394, -0.0391342493023831, 0.0295194909257746,
         0.0273330683450780};
  }
  else if (wavelet == "sym6")
  {
    g = {-0.00780070832503415, 0.00176771186424280, 0.0447249017706658,
         -0.0210602925123006, -0.0726375227864625, 0.337929421727622,
         0.787641141030194, 0.491055941926747, -0.0483117425856330,
         -0.117990111148191, 0.00349071208421747, 0.0154041093270274};
  }
  else if (wavelet == "sym7")
  {
    g = {0.0102681767085113, 0.00401024487153366, -0.107808237703818,
         -0.140047240442962, 0.288629631751515, 0.767764317003164,
         0.536101917091763, 0.0174412550868558, -0.0495528349371273,
         0.0678926935013727, 0.0305155131659636, -0.0126363034032519,
         -0.00104738488868292, 0.00268181456825788};
  }
  else if (wavelet == "sym8")
  {
    g = {0.00188995033275946, -0.000302920514721367,
         -0.0149522583370482, 0.00380875201389062, 0.0491371796736075,
         -0.0272190299170560, -0.0519458381077090, 0.364441894835331,
         0.777185751700524, 0.481359651258372, -0.0612733590676585,
         -0.143294238350810, 0.00760748732491761, 0.0316950878114930,
         -0.000542132331791148, -0.00338241595100613};
  }
  else if (wavelet == "sym9")
  {
    g = {0.00106949003290860, -0.000473154498680074,
         -0.0102640640276331, 0.00885926749340024, 0.0620777893028850,
         -0.0182337707793965, -0.191550831297283, 0.0352724880352735,
         0.617338449140935, 0.717897082764410, 0.238760914607304,
         -0.0545689584308325, 0.000583462746125848, 0.0302248788582752,
         -0.0115282102076793, -0.0132719677818171, 0.000619780888985554,
         0.00140091552591467};
  }
  else if (wavelet == "sym10")
  {
    g = {-0.000459329421004650, 5.70360836185006e-05,
         0.00459317358531176, -0.000804358932016547,
         -0.0203549398123110, 0.00576491203358172, 0.0499949720773757,
         -0.0319900568824271, -0.0355367404738153, 0.383826761067083,
         0.769510037021100, 0.471690666938436, -0.0708805357832397,
         -0.159494278884913, 0.0116098939037126, 0.0459272392310921,
         -0.00146538258130490, -0.00864129927702228,
         9.56326707228934e-05, 0.000770159809114479};
  }

  // coiflets
  else if (wavelet == "coif1")
  {
    g = {-0.0727326195128539, 0.337897662457809, 0.852572020212255,
         0.384864846864203, -0.0727326195128539, -0.0156557281354645};
  }
  else if (wavelet == "coif2")
  {
    g = {0.0163873364635221, -0.0414649367817592, -0.0673725547219630,
         0.386110066821162, 0.812723635445542, 0.417005184421693,
         -0.0764885990783064, -0.0594344186464569, 0.0236801719463341,
         0.00561143481939450, -0.00182320887070299,
         -0.000720549445364512};
  }
  else if (wavelet == "coif3")
  {
    g = {-0.00379351286449101, 0.00778259642732542, 0.0234526961418363,
         -0.0657719112818555, -0.0611233900026729, 0.405176902409617,
         0.793777222625621, 0.428483476377619, -0.0717998216193120,
         -0.0823019271068860, 0.0345550275730616, 0.0158805448636159,
         -0.00900797613666158, -0.00257451768875022, 0.00111751877089060,
         0.000466216960112886, -7.09833031381413e-05, -3.45997728362126e-05};
  }
  else if (wavelet == "coif4")
  {
    g = {0.000892313668582315, -0.00162949201260173,
         -0.00734616632764209, 0.0160689439647763, 0.0266823001560531,
         -0.0812666996808788, -0.0560773133167548, 0.415308407030430,
         0.782238930920499, 0.434386056491469, -0.0666274742634250,
         -0.0962204420339880, 0.0393344271233375, 0.0250822618448641,
         -0.0152117315279463, -0.00565828668661072, 0.00375143615727846,
         0.00126656192929894, -0.000589020756244338,
         -0.000259974552487713, 6.23390344610071e-05,
         3.12298758653456e-05, -3.25968023688337e-06,
         -1.78498500308826e-06};
  }
  else if (wavelet == "coif5")
  {
    g = {-0.000212080839803798, 0.000358589687895738,
         0.00217823635810902, -0.00415935878138605, -0.0101311175198498,
         0.0234081567858392, 0.0281680289709364, -0.0919200105596962,
         -0.0520431631762438, 0.421566206690852, 0.774289603652956,
         0.437991626171837, -0.0620359639629036, -0.105574208703339,
         0.0412892087501817, 0.0326835742671118, -0.0197617789425726,
         -0.00916423116248185, 0.00676418544805308, 0.00243337321265767,
         -0.00166286370201308, -0.000638131343045111,
         0.000302259581813063, 0.000140541149702034,
         -4.13404322725125e-05, -2.13150268099558e-05,
         3.73465517514140e-06, 2.06376185136468e-06,
         -1.67442885768230e-07, -9.51765727381917e-08};
  }

  // Biorthogonal
  else if (wavelet == "bior1.3")
  {
    g = {0, 0, 0.707106781186548, 0.707106781186548, 0, 0};
  }
  else if (wavelet == "bior1.5")
  {
    g = {0, 0, 0, 0, 0.707106781186548, 0.707106781186548, 0, 0, 0, 0};
  }
  else if (wavelet == "bior2.2")
  {
    g = {0, 0.353553390593274, 0.707106781186548, 0.353553390593274, 0, 0};
  }
  else if (wavelet == "bior2.4")
  {
    g = {0, 0, 0, 0.353553390593274, 0.707106781186548, 0.353553390593274,
         0, 0, 0, 0};
  }
  else if (wavelet == "bior2.6")
  {
    g = {0, 0, 0, 0, 0, 0.353553390593274, 0.707106781186548,
         0.353553390593274, 0, 0, 0, 0, 0, 0};
  }
  else if (wavelet == "bior2.8")
  {
    g = {0, 0, 0, 0, 0, 0, 0, 0.353553390593274, 0.707106781186548,
         0.353553390593274, 0, 0, 0, 0, 0, 0, 0, 0};
  }
  else if (wavelet == "bior3.1")
  {
    g = {0.176776695296637, 0.530330085889911, 0.530330085889911,
         0.176776695296637};
  }
  else if (wavelet == "bior3.3")
  {
    g = {0, 0, 0.176776695296637, 0.530330085889911, 0.530330085889911,
         0.176776695296637, 0, 0};
  }
  else if (wavelet == "bior3.5")
  {
    g = {0, 0, 0, 0, 0.176776695296637, 0.530330085889911,
         0.530330085889911, 0.176776695296637, 0, 0, 0, 0};
  }
  else if (wavelet == "bior3.7")
  {
    g = {0, 0, 0, 0, 0, 0, 0.176776695296637, 0.530330085889911,
         0.530330085889911, 0.176776695296637, 0, 0, 0, 0, 0, 0};
  }
  else if (wavelet == "bior3.9")
  {
    g = {0, 0, 0, 0, 0, 0, 0, 0, 0.176776695296637, 0.530330085889911,
         0.530330085889911, 0.176776695296637, 0, 0, 0, 0, 0, 0, 0, 0};
  }
  else if (wavelet == "bior4.4")
  {
    g = {0, -0.0645388826286971, -0.0406894176091641, 0.418092273221617,
         0.788485616405583, 0.418092273221617, -0.0406894176091641,
         -0.0645388826286971, 0, 0};
  }
  else if (wavelet == "bior5.5")
  {
    g = {0.0134567094591187, -0.00269496688011151, -0.136706584664329, -0.0935046974009389, 0.476803265798484, 0.899506109748648,
         0.476803265798484, -0.0935046974009389, -0.136706584664329,
         -0.00269496688011151, 0.0134567094591187, 0};
  }
  else if (wavelet == "bior6.8")
  {
    g = {0, 0, 0, 0.0144262825056244, 0.0144675048967901,
         -0.0787220010626288, -0.0403679790303399, 0.417849109150275,
         0.758907729453654, 0.417849109150275, -0.0403679790303399,
         -0.0787220010626288, 0.0144675048967901, 0.0144262825056244,
         0, 0, 0, 0};
  }

  // Reverse Biorthogonal
  else if (wavelet == "rbio1.3")
  {
    g = {-0.0883883476483185, 0.0883883476483185, 0.707106781186548,
         0.707106781186548, 0.0883883476483185, -0.0883883476483185};
  }
  else if (wavelet == "rbio1.5")
  {
    g = {0.0165728151840597, -0.0165728151840597, -0.121533978016438,
         0.121533978016438, 0.707106781186548, 0.707106781186548,
         0.121533978016438, -0.121533978016438, -0.0165728151840597,
         0.0165728151840597};
  }
  else if (wavelet == "rbio2.2")
  {
    g = {-0.176776695296637, 0.353553390593274, 1.06066017177982,
         0.353553390593274, -0.176776695296637, 0};
  }
  else if (wavelet == "rbio2.4")
  {
    g = {0.0331456303681194, -0.0662912607362388, -0.176776695296637,
         0.419844651329513, 0.994368911043583, 0.419844651329513,
         -0.176776695296637, -0.0662912607362388, 0.0331456303681194, 0};
  }
  else if (wavelet == "rbio2.6")
  {
    g = {-0.00690533966002488, 0.0138106793200498, 0.0469563096881692,
         -0.107723298696388, -0.169871355636612, 0.447466009969612,
         0.966747552403483, 0.447466009969612, -0.169871355636612,
         -0.107723298696388, 0.0469563096881692, 0.0138106793200498,
         -0.00690533966002488, 0};
  }
  else if (wavelet == "rbio2.8")
  {
    g = {0.00151054305063044, -0.00302108610126088,
         -0.0129475118625466, 0.0289161098263542, 0.0529984818906909,
         -0.134913073607736, -0.163829183434090, 0.462571440475917,
         0.951642121897179, 0.462571440475917, -0.163829183434090,
         -0.134913073607736, 0.0529984818906909, 0.0289161098263542,
         -0.0129475118625466, -0.00302108610126088, 0.00151054305063044,
         0};
  }
  else if (wavelet == "rbio3.1")
  {
    g = {-0.353553390593274, 1.06066017177982, 1.06066017177982,
         -0.353553390593274};
  }
  else if (wavelet == "rbio3.3")
  {
    g = {0.0662912607362388, -0.198873782208717, -0.154679608384557,
         0.994368911043583, 0.994368911043583, -0.154679608384557,
         -0.198873782208717, 0.0662912607362388};
  }
  else if (wavelet == "rbio3.5")
  {
    g = {-0.0138106793200498, 0.0414320379601493, 0.0524805814161891,
         -0.267927178808965, -0.0718155324642587, 0.966747552403483,
         0.966747552403483, -0.0718155324642587, -0.267927178808965,
         0.0524805814161891, 0.0414320379601493, -0.0138106793200498};
  }
  else if (wavelet == "rbio3.7")
  {
    g = {0.00302108610126088, -0.00906325830378265,
         -0.0168317654213106, 0.0746639850740190, 0.0313329787073629,
         -0.301159125922835, -0.0264992409453455, 0.951642121897179,
         0.951642121897179, -0.0264992409453455, -0.301159125922835,
         0.0313329787073629, 0.0746639850740190, -0.0168317654213106,
         -0.00906325830378265, 0.00302108610126088};
  }
  else if (wavelet == "rbio3.9")
  {
    g = {-0.000679744372783699, 0.00203923311835110,
         0.00506031921961198, -0.0206189126411055, -0.0141127879301758,
         0.0991347824942322, 0.0123001362694193, -0.320191968360779,
         0.00205002271156989, 0.942125700678207, 0.942125700678207,
         0.00205002271156989, -0.320191968360779, 0.0123001362694193,
         0.0991347824942322, -0.0141127879301758, -0.0206189126411055,
         0.00506031921961198, 0.00203923311835110, -0.000679744372783699};
  }
  else if (wavelet == "rbio4.4")
  {
    g = {0.0378284555072640, -0.0238494650195568, -0.110624404418437,
         0.377402855612831, 0.852698679008894, 0.377402855612831,
         -0.110624404418437, -0.0238494650195568, 0.0378284555072640, 0};
  }
  else if (wavelet == "rbio5.5")
  {
    g = {0, 0.0396870883474054, 0.00794810863724032, -0.0544637884682369,
         0.345605281956033, 0.736660181428211, 0.345605281956033,
         -0.0544637884682369, 0.00794810863724032, 0.0396870883474054, 0,
         0};
  }
  else if (wavelet == "rbio6.8")
  {
    g = {0.00190883173648129, -0.00191428612908877, -0.0169906398676023,
         0.0119345652797293, 0.0497329034909408, -0.0772631731672041,
         -0.0940592034957365, 0.420796284609827, 0.825922997458402,
         0.420796284609827, -0.0940592034957365, -0.0772631731672041,
         0.0497329034909408, 0.0119345652797293, -0.0169906398676023,
         -0.00191428612908877, 0.00190883173648129, 0};
  }

  // Least Asymmetric
  else if (wavelet == "la8")
  {
    g = {-0.07576571478935668, -0.02963552764596039, 0.49761866763256290,
         0.80373875180538600, 0.29785779560560505, -0.09921954357695636,
         -0.01260396726226383, 0.03222310060407815};
  }
  else if (wavelet == "la10")
  {
    g = {0.0195388827353869, -0.0211018340249298, -0.1753280899081075,
         0.0166021057644243, 0.6339789634569490, 0.7234076904038076,
         0.1993975339769955, -0.0391342493025834, 0.0295194909260734,
         0.0273330683451645};
  }
  else if (wavelet == "la12")
  {
    g = {0.0154041093273377, 0.0034907120843304, -0.1179901111484105,
         -0.0483117425859981, 0.4910559419276396, 0.7876411410287941,
         0.3379294217282401, -0.0726375227866000, -0.0210602925126954,
         0.0447249017707482, 0.0017677118643983, -0.0078007083247650};
  }
  else if (wavelet == "la14")
  {
    g = {0.0102681767084968, 0.0040102448717033, -0.1078082377036168,
         -0.1400472404427030, 0.2886296317509833, 0.7677643170045710,
         0.5361019170907720, 0.0174412550871099, -0.0495528349370410,
         0.0678926935015971, 0.0305155131659062, -0.0126363034031526,
         -0.0010473848889657, 0.0026818145681164};
  }
  else if (wavelet == "la16")
  {
    g = {-0.0033824159513594, -0.0005421323316355, 0.0316950878103452,
         0.0076074873252848, -0.1432942383510542, -0.0612733590679088,
         0.4813596512592012, 0.7771857516997478, 0.3644418948359564,
         -0.0519458381078751, -0.0272190299168137, 0.0491371796734768,
         0.0038087520140601, -0.0149522583367926, -0.0003029205145516,
         0.0018899503329007};
  }
  else if (wavelet == "la18")
  {
    g = {0.0010694900326538, -0.0004731544985879, -0.0102640640276849,
         0.0088592674935117, 0.0620777893027638, -0.0182337707798257,
         -0.1915508312964873, 0.0352724880359345, 0.6173384491413523,
         0.7178970827642257, 0.2387609146074182, -0.0545689584305765,
         0.0005834627463312, 0.0302248788579895, -0.0115282102079848,
         -0.0132719677815332, 0.0006197808890549, 0.0014009155255716};
  }
  else if (wavelet == "la20")
  {
    g = {0.0007701598091030, 0.0000956326707837, -0.0086412992759401,
         -0.0014653825833465, 0.0459272392237649, 0.0116098939129724,
         -0.1594942788575307, -0.0708805358108615, 0.4716906668426588,
         0.7695100370143388, 0.3838267612253823, -0.0355367403054689,
         -0.0319900568281631, 0.0499949720791560, 0.0057649120455518,
         -0.0203549398039460, -0.0008043589345370, 0.0045931735836703,
         0.0000570360843390, -0.0004593294205481};
  }

  // Best Localized
  else if (wavelet == "bl14")
  {
    g = {0.0120154192834842, 0.0172133762994439, -0.0649080035533744,
         -0.0641312898189170, 0.3602184608985549, 0.7819215932965554,
         0.4836109156937821, -0.0568044768822707, -0.1010109208664125,
         0.0447423494687405, 0.0204642075778225, -0.0181266051311065,
         -0.0032832978473081, 0.0022918339541009};
  }
  else if (wavelet == "bl18")
  {
    g = {0.0002594576266544, -0.0006273974067728, -0.0019161070047557,
         0.0059845525181721, 0.0040676562965785, -0.0295361433733604,
         -0.0002189514157348, 0.0856124017265279, -0.0211480310688774,
         -0.1432929759396520, 0.2337782900224977, 0.7374707619933686,
         0.5926551374433956, 0.0805670008868546, -0.1143343069619310,
         -0.0348460237698368, 0.0139636362487191, 0.0057746045512475};
  }
  else if (wavelet == "bl20")
  {
    g = {0.0008625782242896, 0.0007154205305517, -0.0070567640909701,
         0.0005956827305406, 0.0496861265075979, 0.0262403647054251,
         -0.1215521061578162, -0.0150192395413644, 0.5137098728334054,
         0.7669548365010849, 0.3402160135110789, -0.0878787107378667,
         -0.0670899071680668, 0.0338423550064691, -0.0008687519578684,
         -0.0230054612862905, -0.0011404297773324, 0.0050716491945793,
         0.0003401492622332, -0.0004101159165852};
  }

  // Fejer-Korovkin
  else if (wavelet == "fk4")
  {
    g = {0.653927556, 0.753272493, 0.053179229, -0.046165715};
  }
  else if (wavelet == "fk6")
  {
    g = {0.427915032, 0.812919643, 0.356369511, -0.146438681,
         -0.077177757, 0.040625814};
  }
  else if (wavelet == "fk8")
  {
    g = {0.349238112, 0.78268362, 0.475265135, -0.099683328,
         -0.159978097, 0.043106668, 0.042581632, -0.019000179};
  }
  else if (wavelet == "fk14")
  {
    g = {0.260371769, 0.686891477, 0.611554654, 0.051421654,
         -0.245613928, -0.048575339, 0.124282561, 0.02222674,
         -0.063997373, -0.005074373, 0.029779712, -0.003297479,
         -0.009270613, 0.003514101};
  }
  else if (wavelet == "fk18")
  {
    g = {0.221451519, 0.633556364, 0.650983107, 0.142345179,
         -0.24619798, -0.113622552, 0.127848451, 0.070702923,
         -0.075240623, -0.039723868, 0.045936604, 0.018695401,
         -0.026632496, -0.006157329, 0.013639313, -0.000326361,
         -0.004679885, 0.001635794};
  }
  else if (wavelet == "fk22")
  {
    g = {0.193896108, 0.589452191, 0.670084963, 0.215629849,
         -0.228028856, -0.164465715, 0.111549144, 0.110155265,
         -0.066084517, -0.071841682, 0.043542368, 0.044775212,
         -0.029742881, -0.025970873, 0.020284486, 0.012964249,
         -0.012885991, -0.004838433, 0.007173803, 0.000361286,
         -0.002676992, 0.000880577};
  }

  // B-Splines
  else if (wavelet == "b3spline")
  {
    //g = {(1 / 16) * sqrt(2), (1 / 4) * sqrt(2), (3 / 8) * sqrt(2), (1 / 4) * sqrt(2), (1 / 16) * sqrt(2)};
    g = {0.08838835, 0.35355339, 0.53033009, 0.35355339, 0.08838835};
  }

  // Morris Minimum Bandwidth
  else if (wavelet == "mb4.2")
  {
    g = {0.480175521177872,0.837254536926641,0.226931210008674,-0.130147705740091};
  }
  else if (wavelet == "mb8.2")
  {
    g = {-0.167361910339972,
         0.0184775111415796,
         0.572577135375022,
         0.735133145418075,
         0.294785518212471,
         -0.110867306849616,
         0.00710601543902461,
         0.0643634539765099};
  }
  else if (wavelet == "mb8.3")
  {
    g = {0.174738301528803,
         0.767850906718004,
         0.522624904572497,
         0.0827074307236156,
         -0.205512701798051,
         -0.0944661408264937,
         0.215256301883299,
         -0.0489854404285785};
  }
  else if (wavelet == "mb8.4")
  {
    g = {-0.017878642811265,
         0.0988792655479264,
         -0.204974032230429,
         0.356656656081236,
         0.871243836995708,
         0.240954237888012,
         0.0587156892325444,
         0.0106165516693628};
  }
  else if (wavelet == "mb10.3")
  {
    g = {0.318839182047848,0.71695475963206,0.557975268583356,-0.0173542690228727,-0.235939286715503,-0.00328322781513876,0.11723599339906,-0.0118927993303792,-0.0510043771282126,0.0226823187228783};
  }
  else if (wavelet == "mb12.3")
  {
    g = {0.0735689785628174,0.0217556695749993,-0.204186296011186,-0.0943402181570476,0.423717891722598,0.741896785506919,0.447711891253871,0.0149735197074897,-0.0509758090041788,0.0812216384133213,0.0172700996626256,-0.0584005888591345};
  }
  else if (wavelet == "mb14.3")
  {
    g = {0.0696019121215304,0.0746908812866416,-0.105358896782709,-0.0585155403701382,0.401571212321394,0.739761185948378,0.487266535050877,-0.0251806648306476,-0.14584574925662,0.000996950184524842,0.0362655265763531,-0.0585578505294787,-0.0363909870488386,0.033909047701829};
  }
  else if (wavelet == "mb16.3")
  {
    g = {-0.0130276995981101,0.021736769329445,0.113611596495209,-0.0577656982179929,-0.227835892971518,0.118872496332917,0.634922780413343,0.670164579326173,0.234534192764883,-0.0565665682549847,-0.0198798593867286,0.0547462783111386,-0.0248387592337521,-0.0498469784622765,0.00962042670322061,0.00576589882212848};
  }
  else if (wavelet == "mb18.3")
  {
    g = {0.00284545207880089,0.00206211508181306,0.0596010225966329,0.0440016066824009,-0.080041050346094,-0.0196974874679406,0.443037967969708,0.757328287127109,0.425764261420699,-0.07140820707311,-0.160084160692969,0.0314475319227541,0.031186421823759,-0.0573614317475332,-0.0147522355930391,0.0201122776252006,-0.000450679070866691,0.000621870035770601};
  }
  else if (wavelet == "mb24.3")
  {
    g = {-2.13270601315487e-05,0.000474573602927246,0.000745604104599005,-0.00487905303009478,-0.00148299500914735,0.0419957602590366,-0.00265828201639671,-0.00655951304046013,0.101951200628851,0.168945601042083,0.12435310076703,0.194914701202265,0.458110102825696,0.617638503809693,0.255673101577033,-0.309111101906647,-0.362242402234369,-0.00457544802822209,0.147934200912482,0.0102715400633565,-0.0164485901014575,-0.00206233501272081,0.00119300600735865,5.36130103306936e-05};
  }
  else if (wavelet == "mb32.3")
  {
    g = {2.93466392602757e-09,-4.3684828898861e-09,-5.81335485346602e-08,7.33798481503552e-08,2.16825594534599e-06,-2.96733192520412e-06,0.000942256576249066,-0.00141251796439545,-0.0174823895593312,0.0217999794505001,0.102929597405511,-0.0750493581082728,-0.291494692652456,-0.0850474978562553,0.462622188338941,0.704669882237781,0.396384290008563,0.0262388393386123,0.0523345286808328,0.136948196548023,-0.00465941588255271,-0.0278076492990681,0.00685175882729155,0.00777939780390907,-0.00138053896520153,-0.00104955297354451,5.78586585415892e-05,4.04951389792617e-05,-7.80833580317965e-07,-5.40284586381349e-07,7.77739580395954e-09,5.22470786830371e-09};
  }

  // Beylkin
  else if (wavelet == "beyl")
  {
    g = {0.0993057653740066,0.424215360813028,0.699825214057047,0.44971825114903,-0.110927598348007,-0.264497231446018,0.0269003088040018,0.15553873187701,-0.0175207462670012,-0.0885436306230059,0.0196798660440013,0.0429163872740029,-0.0174604086960012,-0.014365807969001,0.0100404118450007,0.0014842347820001,-0.00273603162600018,0.000640485329000043};
  }

  // Vaidyanathan
  else if (wavelet == "vaid")
  {
    g = {-6.29061182563947e-05,0.000343631906400586,-0.00045395662185025,-0.000944897139851241,0.00284383455859099,0.000708137506886249,-0.00883910344502669,0.00315384706885455,0.0196872150902418,-0.0148534480655401,-0.0354703987515713,0.0387426194509084,0.0558925239188085,-0.0777097512187318,-0.0839288847080799,0.131971661954894,0.13508422767958,-0.194450472558547,-0.26349480356196,0.201612162596737,0.635601062462602,0.572797795545627,0.250184130524708,0.0457993342976703};
  }

  // Han Linear Phase Moments
  else if (wavelet == "han2.3")
  {
    g = {0.226584265197043,0.74568755893446,0.607491641385736,-0.077161555495825,-0.126969125396231,0.0385807777479125};
  }
  else if (wavelet == "han3.3")
  {
    g = {-0.0734258721583418,-0.0331456303681194,0.485442659419839,0.806543672290906,0.310052469415027,-0.0994368911043583,-0.0149624754899767,0.0331456303681194};
  }
  else if (wavelet == "han4.5")
  {
    g = {-0.0405752145881833,0.0917217293065359,0.611048711324784,0.750698956448033,0.149947083689531,-0.170465189933196,-0.00932249747851275,0.0369169294136873,-0.00399130176127642,-0.00176564404830787};
  }
  else if (wavelet == "han5.5")
  {
    g = {0.0121757931563527,0.00819283800760616,-0.0879766216764665,-0.0417424913174735,0.450595721518061,0.792926667660111,0.380663551412108,-0.0897113928709131,-0.0548434962331059,0.0487472028328823,0.00701553575493472,-0.012084344405046,-0.000523702745444865,0.00077830127948939};
  }

  g = g / sqrt(2);
  //g = g/1.414214;

  return g;
}
