% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/semw.R
\name{semw}
\alias{semw}
\title{A Markov Chain Monte Carlo (MCMC) sampler for the panel spatial error model (SEM) with unknown spatial weight matrix}
\usage{
semw(
  Y,
  tt,
  Z,
  niter = 100,
  nretain = 50,
  W_prior = W_priors(n = nrow(Y)/tt),
  rho_prior = rho_priors(),
  beta_prior = beta_priors(k = ncol(Z)),
  sigma_prior = sigma_priors()
)
}
\arguments{
\item{Y}{numeric \eqn{N \times 1} matrix containing the dependent variables, where \eqn{N = nT} is the number of
spatial (\eqn{n}) times the number of time observations (\eqn{T}, with \code{tt=T}). Note that the observations
have organized such that \eqn{Y = [Y_1',...,Y_T']'}.}

\item{tt}{single number greater or equal to 1. Denotes the number of time observations. \eqn{tt = T}.}

\item{Z}{numeric \eqn{N \times k_3} design matrix of independent variables.
The default value is a \eqn{N \times 1} vector of ones (i.e. an intercept for the model).}

\item{niter}{single number greater or equal to 1, indicating the total number of draws.
Will be automatically coerced to integer. The default value is 100.}

\item{nretain}{single number greater or equal to 0, indicating the number of draws
kept after the burn-in. Will be automatically coerced to integer. The default value is 50.}

\item{W_prior}{list containing prior settings for estimating the spatial weight matrix \eqn{W}.
Generated by the smart constructor \code{\link{W_priors}}.}

\item{rho_prior}{list of prior settings for estimating \eqn{\rho},
generated by the smart constructor \code{\link{rho_priors}}}

\item{beta_prior}{list containing priors for the slope coefficients \eqn{\beta},
generated by the smart constructor \code{\link{beta_priors}}.}

\item{sigma_prior}{list containing priors for the error variance \eqn{\sigma^2},
generated by the smart constructor \code{\link{sigma_priors}}}
}
\value{
List with posterior samples for the slope parameters, \eqn{\rho}, \eqn{\sigma^2}, \eqn{W},
and average direct, indirect, and total effects.
}
\description{
The sampler uses independent  Normal-inverse-Gamma priors for the slope and variance parameters, as well as a four-parameter
beta prior for the spatial autoregressive parameter \eqn{\rho}.
This is a wrapper function calling \code{\link{sdemw}} with no spatially lagged exogenous variables.
}
\details{
The considered panel spatial error model (SEM) with unknown (\eqn{n} by \eqn{n}) spatial weight
matrix \eqn{W} takes the form:

\deqn{
 Y_t = Z \beta + \varepsilon_t,
 }

with \eqn{\varepsilon_t \sim N(0,(I_n - \rho W) \sigma^2)} and \eqn{W = f(\Omega)}. The \eqn{n} by \eqn{n}
matrix \eqn{\Omega} is an unknown binary adjacency matrix with zeros on the main diagonal and
\eqn{f(\cdot)} is the (optional) row-standardization function. \eqn{\rho} is a scalar spatial autoregressive parameter.

\eqn{Y_t} (\eqn{n \times 1}) collects the \eqn{n} cross-sectional (spatial) observations for time
\eqn{t=1,...,T}. \eqn{Z_t} (\eqn{n \times k_3}) is a matrix of explanatory variables.
\eqn{\beta} (\eqn{k_3 \times 1}) is an unknown slope parameter vector.

After vertically staking the \eqn{T} cross-sections  \eqn{Y=[Y_1',...,Y_T']'} (\eqn{N \times 1}),
and \eqn{Z=[Z_1', ..., Z_T']'} (\eqn{N \times k_3}), with \eqn{N=nT}. The final model can be expressed as:

\deqn{
 Y =  Z \beta + \varepsilon,
}

where \eqn{\tilde{W}=I_T \otimes W} and \eqn{\varepsilon \sim N(0,\sigma^2 (I_n \otimes (I_n - \rho W)  )}.
Note that the input data matrices have to be ordered first by the cross-sectional spatial units
and then stacked by time.

Estimation usually even works well in cases of \eqn{n >> T}. However, note that for applications with \eqn{n > 200} the
estimation process becomes computationally demanding and slow. Consider in this case reducing \code{niter} and
\code{nretain} and carefully check whether the posterior chains have converged.
}
\examples{
n = 20; tt = 10
dgp_dat = sim_dgp(n =n, tt = tt, rho = .5, beta3 = c(.5,1),
            sigma2 = .05,n_neighbor = 3,intercept = TRUE,spatial_error = TRUE)
res = semw(Y = dgp_dat$Y,tt = tt,Z = dgp_dat$Z,niter = 20,nretain = 10)
}
