% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/estim.r
\name{estimatetransprobs}
\alias{estimatetransprobs}
\title{Estimate Transition Probabilities from Pairdata}
\usage{
estimatetransprobs(
  dynamicvariables,
  pairdata,
  statespace,
  factors = character(),
  by = character(),
  prior = "nochange"
)
}
\arguments{
\item{dynamicvariables}{The names of the dynamic variables, \code{character} vector}

\item{pairdata}{\code{data.frame} Observed transitions}

\item{statespace}{\code{data.frame} or a \code{list} of two \code{data.frame}s if the state space changes as a result of the activity}

\item{factors}{\code{character} Variables used by the activity}

\item{by}{\code{character} Variables that split the state space}

\item{prior}{function or character}
}
\value{
Transition probabilities as a \code{data.frame}
}
\description{
Estimate Transition Probabilities from Pairdata
}
\details{
Transition probabilities 'move' the forest areas allocated in the cells of state matrix
from the initial states in the beginning of a EFDM run step to the end position. This end
position will be the initial state of the next EFDM step. Length of a step (=time)
in the EFDM run is typically determined by the pairdata. It is the time difference of
tree observations. Note that the pairdata can be also constructed from single observation,
if the other (pair) observation is estimated or modelled.

Each activity needs to have a transition probability. If no pairdata is available,
transition probability matrices can be based entirely on a prior defined with
expert knowledge.

The estimation uses an iterative Bayesian algorithm that is explained in
\url{https://github.com/ec-jrc/efdm/blob/master/documents/EFDMinstructions/Seija_Mathematics_behind_EFDM.pdf}.
The transition probability estimate is the proportion of observed transitions
divided by the number of all transitions from the same starting state.
\code{prior} gives the number of prior transitions. For each \code{factor}
variable the transitions are counted in classes of all factors before the
current factor. The "most important" observations (having all classes right)
is counted \code{length(factors)} times, the second most important observations
are counted \code{length(factors)-1} times and so on.

If pairdata is NULL prior is used by itself.

Observations should have 'factor' and 'by' variables and statepairs with 0 and 1
suffixes to indicate before and after observations.

The estimation algorithm uses information across 'factor' variables, but not
across 'by' variables.

\code{prior} can either \code{character} or \code{function}.
\itemize{
\item "nochange" implies that there is one observation where state doesn't change
\item "uninformative" when no observations are given all states are as likely
\item \code{function(A, dynvar1, dynvar0)} where A is an array of zeros with
  dimnames(A) <- c(dynvar1, dynvar0). The function should fill A with the number of
  prior transitions and return it.
}

The statespace is used to fill in the transitions where there are no observations.
In special cases the statespace may change as a result of the activity.
For example changing the tree species might lead to change in the volume classification used.

This function assumes that the dynamic variables are coded as integers starting with 1.
Other variables are not restricted.
}
\examples{

# Estimation can use observed transitions with different levels of factors.
statespace <- expand.grid(a=1:2, b=1:2, vol=1:5)
pairdata <- data.frame(a=c(1,1,2,2), b=c(1,2,1,2), vol0=c(1,1,1,1), vol1=c(2,3,4,5))
state0 <- statespace
actprob <- statespace
actprob$test <- 1
state0$area <- 0
state0$area[1] <- 1

# With by=c("a", "b") there are two observations: one from prior and the other
# from the exact combination of class levels.
probs <- estimatetransprobs("vol", pairdata, statespace, by=c("a", "b"), prior="nochange")
act1 <- define_activity("test", c("vol"), probs)
runEFDM(state0, actprob, list(act1), 1)

probs <- estimatetransprobs("vol", pairdata, statespace, factors="a", by="b", prior="nochange")
act2 <- define_activity("test", c("vol"), probs)
runEFDM(state0, actprob, list(act2), 1)

probs <- estimatetransprobs("vol", pairdata, statespace, factors="b", by="a", prior="nochange")
act3 <- define_activity("test", c("vol"), probs)
runEFDM(state0, actprob, list(act3), 1)

# The order of variables in factors argument specifies the order of importance.
# Observation that differ in the first variable are counted more times.
probs <- estimatetransprobs("vol", pairdata, statespace, factors=c("a", "b"), prior="nochange")
act4 <- define_activity("test", c("vol"), probs)
runEFDM(state0, actprob, list(act4), 1)

probs <- estimatetransprobs("vol", pairdata, statespace, factors=c("b", "a"), prior="nochange")
act5 <- define_activity("test", c("vol"), probs)
runEFDM(state0, actprob, list(act5), 1)


}
