% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/drda.R
\name{drda}
\alias{drda}
\title{Fit non-linear growth curves}
\usage{
drda(
  formula,
  data,
  subset,
  weights,
  na.action,
  mean_function = "logistic4",
  lower_bound = NULL,
  upper_bound = NULL,
  start = NULL,
  max_iter = 1000
)
}
\arguments{
\item{formula}{an object of class \code{\link[stats]{formula}} (or one that
can be coerced to that class): a symbolic description of the model to be
fitted. Currently supports only formulas of the type \code{y ~ x}.}

\item{data}{an optional data frame, list or environment (or object coercible
by \code{\link[base]{as.data.frame}} to a data frame) containing the
variables in the model. If not found in \code{data}, the variables are taken
from \code{environment(formula)}, typically the environment from which \code{drda}
is called.}

\item{subset}{an optional vector specifying a subset of observations to be
used in the fitting process.}

\item{weights}{an optional vector of weights to be used in the fitting
process. If provided, weighted least squares is used with weights \code{weights}
(that is, minimizing \code{sum(weights * residuals^2)}), otherwise ordinary
least squares is used.}

\item{na.action}{a function which indicates what should happen when the data
contain \code{NA}s. The default is set by the \code{na.action} setting of
\code{\link[base]{options}}, and is \code{\link[stats]{na.fail}} if that is
unset. The 'factory-fresh' default is \code{na.omit}. Another
possible value is \code{NULL}, no action. Value \code{na.exclude} can be useful.}

\item{mean_function}{the model to be fitted. See \code{details} for available
models.}

\item{lower_bound}{numeric vector with the minimum admissible values of the
parameters. Use \code{-Inf} to specify an unbounded parameter.}

\item{upper_bound}{numeric vector with the maximum admissible values of the
parameters. Use \code{Inf} to specify an unbounded parameter.}

\item{start}{starting values for the parameters.}

\item{max_iter}{maximum number of iterations in the optimization algorithm.}
}
\value{
An object of class \code{drda} and \code{model_fit}, where \code{model} is the
chosen mean function. It is a list containing the following components:
\describe{
\item{converged}{boolean value assessing if the optimization algorithm
converged or not.}
\item{iterations}{total number of iterations performed by the
optimization algorithm}
\item{constrained}{boolean value set to \code{TRUE} if optimization was
constrained.}
\item{estimated}{boolean vector indicating which parameters were
estimated from the data.}
\item{coefficients}{maximum likelihood estimates of the model
parameters.}
\item{rss}{minimum value (found) of the residual sum of squares.}
\item{df.residuals}{residual degrees of freedom.}
\item{fitted.values}{fitted mean values.}
\item{residuals}{residuals, that is response minus fitted values.}
\item{weights}{(only for weighted fits) the specified weights.}
\item{mean_function}{model that was used for fitting.}
\item{n}{effective sample size.}
\item{sigma}{corrected maximum likelihood estimate of the standard
deviation.}
\item{loglik}{maximum value (found) of the log-likelihood function.}
\item{fisher.info}{observed Fisher information matrix evaluated at the
maximum likelihood estimator.}
\item{vcov}{approximate variance-covariance matrix of the model
parameters.}
\item{call}{the matched call.}
\item{terms}{the \code{\link[stats]{terms}} object used.}
\item{model}{the model frame used.}
\item{na.action}{(where relevant) information returned by
\code{\link[stats]{model.frame}} on the special handling of \code{NA}s.}
}
}
\description{
Use the Newton's with a trust-region method to fit non-linear growth curves
to observed data.
}
\details{
\subsection{Available models}{
\subsection{Generalized (5-parameter) logistic function}{

The 5-parameter logistic function can be selected by choosing
\code{mean_function = "logistic5"} or \code{mean_function = "l5"}. The function is
defined here as

\code{alpha + delta / (1 + nu * exp(-eta * (x - phi)))^(1 / nu)}

where \code{eta > 0} and \code{nu > 0}. When \code{delta} is positive (negative) the curve
is monotonically increasing (decreasing).

Parameter \code{alpha} is the value of the function when \code{x -> -Inf}.
Parameter \code{delta} is the (signed) height of the curve.
Parameter \code{eta} represents the steepness (growth rate) of the curve.
Parameter \code{phi} is related to the mid-value of the function.
Parameter \code{nu} affects near which asymptote maximum growth occurs.

The value of the function when \code{x -> Inf} is \code{alpha + delta}. In
dose-response studies \code{delta} can be interpreted as the maximum theoretical
achievable effect.
}

\subsection{4-parameter logistic function}{

The 4-parameter logistic function is the default model of \code{drda}. It can be
explicitly selected by choosing \code{mean_function = "logistic4"} or
\code{mean_function = "l4"}. The function is obtained by setting \code{nu = 1} in the
generalized logistic function, that is

\code{alpha + delta / (1 + exp(-eta * (x - phi)))}

where \code{eta > 0}. When \code{delta} is positive (negative) the curve is
monotonically increasing (decreasing).

Parameter \code{alpha} is the value of the function when \code{x -> -Inf}.
Parameter \code{delta} is the (signed) height of the curve.
Parameter \code{eta} represents the steepness (growth rate) of the curve.
Parameter \code{phi} represents the \code{x} value at which the curve is equal to its
mid-point, i.e. \verb{f(phi; alpha, delta, eta, phi) = alpha + delta / 2}.

The value of the function when \code{x -> Inf} is \code{alpha + delta}. In
dose-response studies \code{delta} can be interpreted as the maximum theoretical
achievable effect.
}

\subsection{2-parameter logistic function}{

The 2-parameter logistic function can be selected by choosing
\code{mean_function = "logistic2"} or \code{mean_function = "l2"}. For a monotonically
increasing curve set \code{nu = 1}, \code{alpha = 0}, and \code{delta = 1}:

\code{1 / (1 + exp(-eta * (x - phi)))}

For a monotonically decreasing curve set \code{nu = 1}, \code{alpha = 1}, and
\code{delta = -1}:

\code{1 - 1 / (1 + exp(-eta * (x - phi)))}

where \code{eta > 0}. The lower bound of the curve is zero while the upper bound
of the curve is one.

Parameter \code{eta} represents the steepness (growth rate) of the curve.
Parameter \code{phi} represents the \code{x} value at which the curve is equal to its
mid-point, i.e. \verb{f(phi; eta, phi) = 1 / 2}.
}

\subsection{Gompertz function}{

The Gompertz function is the limit for \code{nu -> 0} of the 5-parameter logistic
function. It can be selected by choosing \code{mean_function = "gompertz"} or
\code{mean_function = "gz"}. The function is defined in this package as

\code{alpha + delta * exp(-exp(-eta * (x - phi)))}

where \code{eta > 0}.

Parameter \code{alpha} is the value of the function when \code{x -> -Inf}.
Parameter \code{delta} is the (signed) height of the curve.
Parameter \code{eta} represents the steepness (growth rate) of the curve.
Parameter \code{phi} sets the displacement along the \code{x}-axis.

The value of the function when \code{x -> Inf} is \code{alpha + delta}. In
dose-response studies \code{delta} can be interpreted as the maximum theoretical
achievable effect.

The mid-point of the function, that is \code{alpha + delta / 2}, is achieved at
\code{x = phi - log(log(2)) / eta}.
}

\subsection{Generalized (5-parameter) log-logistic function}{

The 5-parameter log-logistic function is selected by setting
\code{mean_function = "loglogistic5"} or \code{mean_function = "ll5"}. The function is
defined here as

\code{alpha + delta * (x^eta / (x^eta + nu * phi^eta))^(1 / nu)}

where \code{x >= 0}, \code{eta > 0}, \code{phi > 0}, and \code{nu > 0}. When \code{delta} is
positive (negative) the curve is monotonically increasing (decreasing). The
function is defined only for positive values of the predictor variable \code{x}.

Parameter \code{alpha} is the value of the function at \code{x = 0}.
Parameter \code{delta} is the (signed) height of the curve.
Parameter \code{eta} represents the steepness (growth rate) of the curve.
Parameter \code{phi} is related to the mid-value of the function.
Parameter \code{nu} affects near which asymptote maximum growth occurs.

The value of the function when \code{x -> Inf} is \code{alpha + delta}. In
dose-response studies \code{delta} can be interpreted as the maximum theoretical
achievable effect.
}

\subsection{4-parameter log-logistic function}{

The 4-parameter log-logistic function is selected by setting
\code{mean_function = "loglogistic4"} or \code{mean_function = "ll4"}. The function is
obtained by setting \code{nu = 1} in the generalized log-logistic function, that
is

\code{alpha + delta * x^eta / (x^eta + phi^eta)}

where \code{x >= 0} and \code{eta > 0}. When \code{delta} is positive (negative) the curve
is monotonically increasing (decreasing). The function is defined only for
positive values of the predictor variable \code{x}.

Parameter \code{alpha} is the value of the function at \code{x = 0}.
Parameter \code{delta} is the (signed) height of the curve.
Parameter \code{eta} represents the steepness (growth rate) of the curve.
Parameter \code{phi} represents the \code{x} value at which the curve is equal to its
mid-point, i.e. \verb{f(phi; alpha, delta, eta, phi) = alpha + delta / 2}.

The value of the function when \code{x -> Inf} is \code{alpha + delta}. In
dose-response studies \code{delta} can be interpreted as the maximum theoretical
achievable effect.
}

\subsection{2-parameter log-logistic function}{

The 2-parameter log-logistic function is selected by setting
\code{mean_function = "loglogistic2"} or \code{mean_function = "ll2"}. For a
monotonically increasing curve set \code{nu = 1}, \code{alpha = 0}, and \code{delta = 1}:

\code{x^eta / (x^eta + phi^eta)}

For a monotonically decreasing curve set \code{nu = 1}, \code{alpha = 1}, and
\code{delta = -1}:

\code{1 - x^eta / (x^eta + phi^eta)}

where \code{x >= 0}, \code{eta > 0}, and \code{phi > 0}. The lower bound of the curve is
zero while the upper bound of the curve is one.

Parameter \code{eta} represents the steepness (growth rate) of the curve.
Parameter \code{phi} represents the \code{x} value at which the curve is equal to its
mid-point, i.e. \verb{f(phi; eta, phi) = 1 / 2}.
}

\subsection{log-Gompertz function}{

The log-Gompertz function is the limit for \code{nu -> 0} of the 5-parameter
log-logistic function. It can be selected by choosing
\code{mean_function = "loggompertz"} or \code{mean_function = "lgz"}. The function is
defined in this package as

\code{alpha + delta * exp(-(phi / x)^eta)}

where \code{x > 0}, \code{eta > 0}, and \code{phi > 0}. Note that the limit for \code{x -> 0} is
\code{alpha}. When \code{delta} is positive (negative) the curve is monotonically
increasing (decreasing). The function is defined only for positive values of
the predictor variable \code{x}.

Parameter \code{alpha} is the value of the function at \code{x = 0}.
Parameter \code{delta} is the (signed) height of the curve.
Parameter \code{eta} represents the steepness (growth rate) of the curve.
Parameter \code{phi} sets the displacement along the \code{x}-axis.

The value of the function when \code{x -> Inf} is \code{alpha + delta}. In
dose-response studies \code{delta} can be interpreted as the maximum theoretical
achievable effect.
}

\subsection{Constrained optimization}{

It is possible to search for the maximum likelihood estimates within
pre-specified interval regions.

\emph{Note}: Hypothesis testing is not available for constrained estimates
because asymptotic approximations might not be valid.
}

}
}
\examples{
# by default `drda` uses a 4-parameter logistic function for model fitting
fit_l4 <- drda(response ~ log_dose, data = voropm2)

# get a general overview of the results
summary(fit_l4)

# compare the model against a flat horizontal line and the full model
anova(fit_l4)

# 5-parameter logistic curve appears to be a better model
fit_l5 <- drda(response ~ log_dose, data = voropm2, mean_function = "l5")
plot(fit_l4, fit_l5)

# fit a 2-parameter logistic function
fit_l2 <- drda(response ~ log_dose, data = voropm2, mean_function = "l2")

# compare our models
anova(fit_l2, fit_l4)

# use log-logistic functions when utilizing doses (instead of log-doses)
# here we show the use of other arguments as well
fit_ll5 <- drda(
  response ~ dose, weights = weight, data = voropm2,
  mean_function = "loglogistic5", lower_bound = c(0.5, -1.5, 0, -Inf, 0.25),
  upper_bound = c(1.5, 0.5, 5, Inf, 3), start = c(1, -1, 3, 100, 1),
  max_iter = 10000
)

# note that the maximum likelihood estimate is outside the region of
# optimization: not only the variance-covariance matrix is now singular but
# asymptotic assumptions do not hold anymore.
}
