#' return igraph object from diffee result specified by user input
#'
#' This function can return an igraph object from diffee result for user to work with directly
#' @author Beilun Wang, Zhaoyang Wang (Author), Beilun Wang (maintainer)
#' @param x output generated from diffee function (diffee class)
#' @param type type of graph, there are four options:
#' (1) "task" (graph for each task (including shared part) specified further by subID (task number))
#' (2) "neighbour" (zoom into nodes in the graph specified further by parameter "index" (node id)
#' @param index determines which node(s) to zoom into when parameter type is "neighbour"
#' could either be an integer or vector of integers representing node ids
#' (zoom into one node or multiple nodes)
#' @return an igraph object of graph / subgraph from diffee result specified by user input
#' @details the function aims to provide users the flexibility to explore and visualize the graph own their own
#' generated from diffee
#' @examples
#' \dontrun{
#' data(exampleData)
#' result = diffee(exampleData[[1]], exampleData[[2]], 0.45)
#' graph = returngraph(result)
#' }
#' @export
#' @import igraph
#' @importFrom grDevices rainbow
returngraph<-
  function(x,
           type = "task",
           index = NULL) {
    adj = .make.adj.matrix(x$diffNet)
    diag(adj) = 0
    gadj = graph.adjacency(adj, mode = "upper", weighted = TRUE)
    if (!is.null(E(gadj)$weight)) {
      E(gadj)$color = rainbow(1)[E(gadj)$weight]
    }

    if (type == "task"){
      ### do nothing since gadj is already the graph
    }
    else if(type == "neighbour" ){
      if (!prod(index %in% (1:vcount(gadj)))) {
        stop("please specify valid index number(s)")
      }
      gadj = subgraph.edges(gadj,unlist(incident_edges(gadj, index)) , delete.vertices = FALSE)
    }
    else {
      stop("please specify a correct type")
    }
    return(gadj)
  }


#' Plot diffee result specified by user input
#'
#' This function can plot and return multiple sparse graphs distinguished by edge colors
#' from the result generated by diffee
#'
#' @author Beilun Wang, Zhaoyang Wang (Author), Beilun Wang (maintainer)
#' @param x output generated from diffee function (diffee class)
#' @param graphlabel vertex names for the graph, there are three options:
#' (1) NA (no label)
#' (2) NULL (default numeric label according to the feature order)
#' (3) a vector of labels (a vector of labels cooresponding to x)
#' deault value is NULL
#' @param type type of graph, there are four options:
#' (1) "task" (graph for each task (including shared part) specified further by subID (task number))
#' (2) "neighbour" (zoom into nodes in the graph specified further by index (node id))
#' @param index determines which node(s) to zoom into when parameter type is "neighbour"
#' could either be an integer or vector of integers representing node ids
#' (zoom into one node or multiple nodes)
#' @param graphlayout layout for the graph (two column matrix specifying x,y coordinates of each node in graph)
#' if not provided, igraph will use the default layout_nicely() function to present the graph
#' @param ... extra parameters passed to plot.igraph
#' @return a plot of graph / subgraph from diffee result specified by user input
#' @details when only the diffee is provided, the function will plot all graphs with default numeric labels.
#' User can specify multiple subID and multiple index to zoom in multiple nodes on multiple graphs.
#' Each graph will include a descriptive title.
#' @examples
#' \dontrun{
#' data(exampleData)
#' result = diffee(exampleData[[1]], exampleData[[2]], 0.45)
#' plot.diffee(result)
#' }
#' @export plot.diffee
#' @method plot diffee
#' @S3method plot diffee
#' @import igraph
#' @importFrom graphics plot
plot.diffee <-
  function(x,
           graphlabel = NULL,
           type = "task",
           index = NULL,
           graphlayout = NULL,
           ...)
  {
    .env = "environment: namespace:diffee"

    gadj = returngraph(x, type = type, index = index)

    graphlayout = .makelayout(gadj, graphlayout = graphlayout)

    title = .maketitle(type = type,
                       index = index,
                       graphlabel = graphlabel)

    plot(
      gadj,
      layout = graphlayout,
      vertex.label.font = 2,
      vertex.shape = "none",
      vertex.label.color = "gray40",
      vertex.label = graphlabel,
      vertex.label.cex = .7,
      vertex.frame.color = "white",
      vertex.size = 10 ,
      main = title
    )

  }


.make.adj.matrix <- function(theta, separate=FALSE) {
  adj = list()
  if(separate)
  {
    adj = (abs(theta) > 1e-5) * 1
  }
  if(!separate)
  {
    adj = 0*theta
    adj = adj+(abs(theta) > 1e-5) * 1
  }
  return(adj)
}

.makelayout <-
  function(x, graphlayout = NULL)
  {
    if (is.null(graphlayout)) {
      graphlayout = layout_nicely(x, dim = 2)

    }
    return(graphlayout)
  }

.maketitle <-
  function(type = "task",
           index = NULL,
           graphlabel = NULL){
    if (type == "task"){
      return ("difference graph")
    }
    if (type == "neighbour"){
      second = "on difference graph"
      if (is.null(graphlabel) || is.na(graphlabel)) {
        first = paste("Zoom in at node", paste(as.character(index), collapse = ", "))
      }

      else {
        first = paste("Zoom in at node", paste(as.character(graphlabel[index]), collapse = ", "))
      }
    }
    return (paste(first,second))

  }
